package ru.yandex.travel.orders.entities;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.money.CurrencyUnit;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.DiscriminatorColumn;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Version;

import com.google.protobuf.Message;
import lombok.AccessLevel;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.Columns;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;

import ru.yandex.travel.commons.logging.LogEntity;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.services.FiscalTitleGenerator;
import ru.yandex.travel.orders.services.payments.PaymentProfile;
import ru.yandex.travel.workflow.entities.Workflow;

@Entity
@Table(name = "invoices")
@Data
@Inheritance(strategy = InheritanceType.SINGLE_TABLE)
@DiscriminatorColumn(name = "invoice_type")
@EqualsAndHashCode(exclude = {"order"})
@ToString(exclude = {"order"})
@BatchSize(size = 100)
public abstract class Invoice implements LogEntity {
    @Id
    private UUID id;

    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    @Version
    private Integer version;

    @OneToOne
    private Workflow workflow;

    @OneToMany(mappedBy = "invoice", cascade = CascadeType.ALL)
    private List<InvoiceItem> invoiceItems;

    private String source;

    @ManyToOne
    private Order order;

    private UUID orderWorkflowId;

    // todo(tlg-13): rm this unused column, it's been moved to SimpleTrustRefund
    private boolean systemRefreshScheduled;

    private boolean backgroundJobActive;

    protected String requestedPaymentMethodId;

    @OneToOne
    private Account account;

    @OneToMany(mappedBy = "invoice", cascade = CascadeType.ALL)
    @BatchSize(size = 100)
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    private List<FiscalReceipt> fiscalReceipts;

    private String clientToken;
    private String passportId;
    private Instant expirationDate;
    private String returnPath;
    private String confirmationReturnPath;
    private String clientEmail;
    private String clientPhone;

    private String fiscalNds;
    private String fiscalTitle;

    private String purchaseToken;
    private String userIp;

    private String paymentUrl;
    private Instant nextCheckStatusAt;
    private Instant clearAt;

    private String trustPaymentId;
    private String trustStatus;
    private Instant trustPaymentTs;
    private Instant paymentStartTs;
    private Instant paymentCancelTs;

    private String authorizationErrorCode;
    private String authorizationErrorDesc;

    private Instant lastTransitionAt;

    private String userAccount;
    private String approvalCode;
    private String cardType;

    @Type(type = "custom-enum")
    private PaymentProfile paymentProfile;

    @OneToOne(fetch = FetchType.LAZY)
    private SimpleTrustRefund activeTrustRefund;

    @OneToMany(mappedBy = "invoice", fetch = FetchType.LAZY)
    private List<TrustRefund> trustRefunds;

    @ManyToOne
    private PendingInvoice pendingInvoice;

    @Type(type = "protobuf-value")
    @Columns(columns = {
            @Column(name = "payment_test_context_class_name"), @Column(name = "payment_test_context_data")
    })
    @Getter(AccessLevel.NONE)
    private Message paymentTestContext;

    private Boolean newPaymentWebForm;

    public boolean useNewCommonPaymentWebForm() {
        return Objects.requireNonNullElse(newPaymentWebForm, false);
    }

    protected static <T extends Invoice> T initInvoice(Order order, T result, List<InvoiceItemProvider> source,
                                                       AuthorizedUser owner) {
        result.setExpirationDate(order.getExpiresAt());
        result.setId(UUID.randomUUID());
        result.setClientPhone(order.getPhone());
        result.setClientEmail(order.getEmail());
        result.setPassportId(owner.getPassportId());
        result.setUserIp(order.getIp());
        result.setAccount(order.getAccount());
        result.setFiscalTitle(FiscalTitleGenerator.getFiscalTitle(order));
        order.addInvoice(result);

        List<InvoiceItem> invoiceItems = source.stream()
                .map(provider -> {
                    InvoiceItem invoiceItem = new InvoiceItem();
                    invoiceItem.setInvoice(result);
                    invoiceItem.setFiscalItemType(provider.getType());
                    invoiceItem.setFiscalItemId(provider.getFiscalItemId());
                    invoiceItem.setPriceMoney(provider.getMoneyAmount());
                    invoiceItem.setYandexPlusWithdrawMoney(provider.getYandexPlusToWithdraw());
                    invoiceItem.setOriginalPrice(provider.getMoneyAmount());
                    invoiceItem.setClearedSum(BigDecimal.ZERO);
                    invoiceItem.setRefundedSum(BigDecimal.ZERO);
                    invoiceItem.setFiscalTitle(provider.getTitle());
                    invoiceItem.setFiscalNds(provider.getVatType());
                    invoiceItem.setFiscalInn(provider.getInn());
                    return invoiceItem;
                }).collect(Collectors.toList());
        result.setInvoiceItems(invoiceItems);

        return result;
    }

    public List<FiscalReceipt> getFiscalReceipts() {
        return fiscalReceipts == null ? Collections.emptyList() : Collections.unmodifiableList(fiscalReceipts);
    }

    public List<InvoiceItem> getInvoiceItems() {
        return invoiceItems == null ? Collections.emptyList() : Collections.unmodifiableList(invoiceItems);
    }

    public BigDecimal calculateCurrentAmount() {
        return getInvoiceItems().stream().map(InvoiceItem::getPrice).reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    public MoneyMarkup calculateCurrentAmountMarkup() {
        return getInvoiceItems().stream()
                .map(InvoiceItem::getPriceMarkup)
                .reduce(MoneyMarkup.zero(getCurrency()), MoneyMarkup::add);
    }

    public void addInvoiceItem(InvoiceItem item) {
        if (this.invoiceItems == null) {
            this.invoiceItems = new ArrayList<>();
        }
        item.setInvoice(this);
        this.invoiceItems.add(item);
    }

    public void initAcquireFiscalReceipt() {
        boolean exists = getFiscalReceipts().stream().anyMatch(i -> i.getReceiptType() == FiscalReceiptType.ACQUIRE);
        if (!exists) {
            addFiscalReceipt(FiscalReceipt.createReceipt(FiscalReceiptType.ACQUIRE, purchaseToken, paymentProfile,
                    null));
        }
    }

    public void initClearFiscalReceipt(UUID orderRefundId) {
        boolean exists = getFiscalReceipts().stream().anyMatch(i -> i.getReceiptType() == FiscalReceiptType.CLEAR);
        if (!exists) {
            addFiscalReceipt(FiscalReceipt.createReceipt(FiscalReceiptType.CLEAR, purchaseToken, paymentProfile,
                    orderRefundId));
        }
    }

    public void addRefundFiscalReceipt(String refundId, UUID orderRefundId) {
        addFiscalReceipt(FiscalReceipt.createReceipt(FiscalReceiptType.REFUND, refundId, paymentProfile,
                orderRefundId));
    }

    private void addFiscalReceipt(FiscalReceipt fiscalReceipt) {
        if (this.fiscalReceipts == null) {
            this.fiscalReceipts = new ArrayList<>();
        }
        fiscalReceipt.setInvoice(this);
        this.fiscalReceipts.add(fiscalReceipt);
    }

    public void rescheduleNextCheckStatusAt(Duration duration) {
        this.nextCheckStatusAt = Instant.now().plus(duration);
    }

    public abstract Enum<?> getInvoiceState();

    public abstract EInvoiceType getPublicType();

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }

    public Message getEffectivePaymentTestContext() {
        if (paymentTestContext != null) {
            return paymentTestContext;
        } else if (getOrder() != null) {
            return getOrder().getPaymentTestContext();
        } else {
            return null;
        }
    }

    public CurrencyUnit getCurrency() {
        // a few nasty hacks for acceptance tests that don't attach any order to the invoice and for legacy unit tests
        if (account != null && account.getCurrency() != null) {
            return account.getCurrency();
        }
        if (order != null && order.getCurrency() != null) {
            return order.getCurrency();
        }
        throw new IllegalStateException("Can't determine currency of the invoice: id " + id);
    }
}
