package ru.yandex.travel.orders.entities;

import java.math.BigDecimal;
import java.time.LocalDateTime;

import javax.money.CurrencyUnit;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.SequenceGenerator;
import javax.persistence.Table;
import javax.persistence.Version;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.ToString;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;
import org.javamoney.moneta.Money;

@Entity
@Table(name = "invoice_items")
@Data
@NoArgsConstructor
@AllArgsConstructor
@Builder
@EqualsAndHashCode(exclude = {"invoice"})
@ToString(exclude = {"invoice"})
public class InvoiceItem {
    @Id
    @GeneratedValue(
            strategy = GenerationType.SEQUENCE,
            generator = "invoice_items_id_seq"
    )
    @SequenceGenerator(
            name = "invoice_items_id_seq",
            sequenceName = "invoice_items_id_seq",
            allocationSize = 1
    )
    private Long id;

    @Type(type = "custom-enum")
    private FiscalItemType fiscalItemType;

    @ManyToOne
    private Invoice invoice;

    @CreationTimestamp
    private LocalDateTime createdAt;

    @UpdateTimestamp
    private LocalDateTime updatedAt;

    @Version
    private Integer version;

    private Long fiscalItemId;

    private String trustOrderId;

    // total price, includes yandex account money if any
    private BigDecimal price;

    private BigDecimal originalPrice;

    private BigDecimal clearedSum;
    private BigDecimal refundedSum;

    @Type(type = "custom-enum")
    private VatType fiscalNds;

    private String fiscalTitle;

    private String fiscalInn;

    /**
     * Amount of yandex plus points being paid by the user for the order.
     */
    private BigDecimal yandexPlusWithdraw;

    public CurrencyUnit getCurrency() {
        return getInvoice().getCurrency();
    }

    public Money getPriceMoney() {
        return Money.of(price, getCurrency());
    }

    public void setPriceMoney(Money amount) {
        price = checkCurrencyAndGetValue(amount);
    }

    public void setOriginalPrice(Money amount) {
        originalPrice = checkCurrencyAndGetValue(amount);
    }

    public void setYandexPlusWithdrawMoney(Money amount) {
        yandexPlusWithdraw = checkCurrencyAndGetValue(amount);
    }

    private BigDecimal checkCurrencyAndGetValue(Money amount) {
        Preconditions.checkArgument(amount.getCurrency().equals(getCurrency()),
                "Received sum in %s while order currency is %s",
                amount.getCurrency(), getCurrency());
        return amount.getNumberStripped();
    }

    public BigDecimal getYandexPlusWithdraw() {
        return yandexPlusWithdraw != null ? yandexPlusWithdraw : BigDecimal.ZERO;
    }

    public MoneyMarkup getPriceMarkup() {
        Money yandexPlus = Money.of(getYandexPlusWithdraw(), getCurrency());
        return MoneyMarkup.builder()
                .card(getPriceMoney().subtract(yandexPlus))
                .yandexAccount(yandexPlus)
                .build();
    }
}
