package ru.yandex.travel.orders.entities;

import java.util.function.BiFunction;
import java.util.function.Function;

import javax.money.CurrencyUnit;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.base.Preconditions;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.NonNull;
import lombok.Value;
import lombok.extern.jackson.Jacksonized;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.lang.MoneyUtils;

/**
 * Split between real money and yandex plus points withdrawn. The original price is the sum.
 */
@Value
@AllArgsConstructor(access = AccessLevel.PRIVATE)
@Builder(toBuilder = true)
@Jacksonized
public class MoneyMarkup {
    @NonNull
    private final Money card;
    @NonNull
    private final Money yandexAccount;

    @JsonIgnore
    public Money getTotal() {
        return card.add(yandexAccount);
    }

    @JsonIgnore
    public boolean isCardOnly() {
        return yandexAccount.isZero();
    }

    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    public boolean hasNegativeValues() {
        return card.isNegative() || yandexAccount.isNegative();
    }

    public void ensureNoNegativeValues() {
        Preconditions.checkState(!hasNegativeValues(), "Unexpected negative values in money markup: %s", this);
    }

    public static MoneyMarkup zero(CurrencyUnit currency) {
        return MoneyMarkup.builder()
                .card(Money.zero(currency))
                .yandexAccount(Money.zero(currency))
                .build();
    }

    public static MoneyMarkup cardOnly(Money card) {
        return MoneyMarkup.builder()
                .card(card)
                .yandexAccount(Money.zero(card.getCurrency()))
                .build();
    }

    public MoneyMarkup addSafe(MoneyMarkup another) {
        return another != null ? add(another) : this;
    }

    public MoneyMarkup add(MoneyMarkup another) {
        return MoneyMarkup.builder()
                .card(card.add(another.card))
                .yandexAccount(yandexAccount.add(another.yandexAccount))
                .build();
    }

    public MoneyMarkup subtract(MoneyMarkup another) {
        return MoneyMarkup.builder()
                .card(card.subtract(another.card))
                .yandexAccount(yandexAccount.subtract(another.yandexAccount))
                .build();
    }

    public MoneyMarkup merge(MoneyMarkup another, BiFunction<Money, Money, Money> mergeFunction) {
        return MoneyMarkup.builder()
                .card(mergeFunction.apply(card, another.card))
                .yandexAccount(mergeFunction.apply(yandexAccount, another.yandexAccount))
                .build();
    }

    public MoneyMarkup map(Function<Money, Money> mapFunction) {
        return MoneyMarkup.builder()
                .card(mapFunction.apply(card))
                .yandexAccount(mapFunction.apply(yandexAccount))
                .build();
    }

    public void ensureValuesScale() {
        MoneyUtils.ensureDecimalScale(card);
        MoneyUtils.ensureIntegerScale(yandexAccount);
    }
}
