package ru.yandex.travel.orders.entities;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.money.CurrencyUnit;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.DiscriminatorColumn;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.NamedAttributeNode;
import javax.persistence.NamedEntityGraph;
import javax.persistence.NamedSubgraph;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.OrderColumn;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import javax.persistence.Table;
import javax.persistence.Version;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NonNull;
import lombok.Setter;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.Columns;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;
import org.hibernate.annotations.Where;
import org.hibernate.envers.NotAudited;
import org.javamoney.moneta.Money;

import ru.yandex.bolts.collection.Option;
import ru.yandex.travel.commons.experiments.KVExperiment;
import ru.yandex.travel.commons.experiments.OrderExperiments;
import ru.yandex.travel.commons.logging.LogEntity;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EPromoCodeApplicationResultType;
import ru.yandex.travel.orders.entities.context.OrderStateContext;
import ru.yandex.travel.orders.entities.promo.FiscalItemDiscount;
import ru.yandex.travel.orders.entities.promo.OrderGeneratedPromoCodes;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivation;
import ru.yandex.travel.orders.entities.promo.PromoCodeApplication;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.entities.Workflow;

@Entity
@Table(name = "orders")
@Getter
@Setter
@EqualsAndHashCode(of = "id")
@Inheritance(strategy = InheritanceType.SINGLE_TABLE)
@DiscriminatorColumn(name = "order_type")
@NamedEntityGraph(
        name = "orders-with-all-details",
        attributeNodes = {
                @NamedAttributeNode(value = "orderItems", subgraph = "items-subgraph"),
                @NamedAttributeNode(value = "invoices", subgraph = "invoice-subgraph"),
                @NamedAttributeNode("workflow"),
                @NamedAttributeNode("account")
        },
        subgraphs = {
                @NamedSubgraph(
                        name = "items-subgraph",
                        attributeNodes = {
                                @NamedAttributeNode("fiscalItems"),
                                @NamedAttributeNode("workflow")
                        }),
                @NamedSubgraph(
                        name = "invoice-subgraph",
                        attributeNodes = {
                                @NamedAttributeNode("workflow")
                        }),
        }
)
@NamedEntityGraph(
        name = "order-for-cpa",
        attributeNodes = {
                @NamedAttributeNode(value = "currentInvoice")
        }
)
public abstract class Order implements LogEntity {
    public static final String DEDUPLICATION_KEY_CONSTRAINT_NAME = "orders_deduplication_key_unique";
//    public static final String WORKFLOW_ENTITY_TYPE_NAME = "order";

    @Id
    private UUID id;

    private UUID deduplicationKey;

    private String prettyId;

    @OneToMany(mappedBy = "order", cascade = CascadeType.ALL)
    @OrderColumn(name = "item_position")
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    @BatchSize(size = 50)
    @Where(clause = "archived = false")
    private List<OrderItem> orderItems;

    @OneToMany(mappedBy = "order", cascade = CascadeType.ALL)
    @OrderColumn(name = "item_position")
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    @BatchSize(size = 50)
    @Where(clause = "archived = true")
    private List<OrderItem> archivedOrderItems;

    @CreationTimestamp
    private Instant createdAt;
    @UpdateTimestamp
    private Instant updatedAt;

    @Type(type = "proto-currency-unit")
    private ProtoCurrencyUnit currency;

    @Type(type = "jsonb-object")
    private FxRate fxRate;

    @Version
    private Integer version;

    @OneToOne
    private Account account;

    @OneToOne
    private Workflow workflow;

    private Instant expiresAt;

    private String ip;
    private String email;
    private String phone;
    private String label;
    private Boolean allowsSubscription;
    private Integer geoId;

    @OneToOne
    private Invoice currentInvoice;

    @OneToMany(mappedBy = "order")
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    @BatchSize(size = 100)
    private Set<Invoice> invoices;

    @OneToMany(mappedBy = "order", cascade = CascadeType.ALL)
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    @BatchSize(size = 100)
    private Set<PendingInvoice> pendingInvoices;

    private String documentUrl;
    private String businessTripDocUrl;

    private boolean userActionScheduled;

    private LocalDateTime servicedAt;

    private Instant lastTransitionAt;

    @Type(type = "proto-enum")
    private EDisplayOrderType displayType;

    @OneToMany(mappedBy = "order", cascade = CascadeType.ALL)
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    @BatchSize(size = 100)
    private List<PromoCodeApplication> promoCodeApplications;

    @OneToMany(mappedBy = "order", cascade = CascadeType.ALL)
    @OrderColumn(name = "item_position")
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    @BatchSize(size = 100)
    private List<MoneyRefund> moneyRefunds;

    @OneToMany(mappedBy = "order", cascade = CascadeType.ALL)
    @NotAudited
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    @BatchSize(size = 100)
    private List<OrderRefund> orderRefunds;

    // NASTY HACK TO AVOID EAGER LOADING
    @OneToMany(fetch = FetchType.LAZY, cascade = CascadeType.ALL, orphanRemoval = true)
    @JoinColumn(name = "order_id")
    @Getter(AccessLevel.NONE)
    @BatchSize(size = 100)
    private List<OrderLabelParams> orderLabelParams;

    @Getter(AccessLevel.NONE)
    private Boolean mockPayment;

    @Type(type = "protobuf-value")
    @Columns(columns = {
            @Column(name = "payment_test_context_class_name"), @Column(name = "payment_test_context_data")
    })
    private Message paymentTestContext;

    //Boolean as we want to be able to handle null cases
    private Boolean removed;

    @Type(type = "jsonb-object")
    private OrderStateContext stateContext;

    @OneToOne(cascade = CascadeType.ALL)
    private PaymentSchedule paymentSchedule;

    @Getter(value = AccessLevel.NONE)
    private Boolean useDeferredPayment;

    private Boolean eligibleForDeferredPayment;

    private UUID activePaymentWorkflowId;
    private String removeReason;

    private boolean trainRebookingEnabled;

    @Type(type = "jsonb-object")
    @Getter(value = AccessLevel.NONE)
    private OrderExperiments experiments;

    /**
        Здесь хранятся все UAAS эксперименты, не только те, которые нужны непосредственно Оркестратору. Используется
        для прозрачного пробрасывания экспериментов в promo service
     */
    @Type(type = "jsonb-object")
    @Getter(value = AccessLevel.NONE)
    private List<KVExperiment> kVExperiments;

    @OneToOne(cascade = CascadeType.ALL)
    private OrderGeneratedPromoCodes generatedPromoCodes;

    @PrePersist
    @PreUpdate
    public void preInsertAndUpdate() {
        if (removed == null) {
            removed = false;
        }
    }

    public boolean nullSafeRemoved() {
        if (removed == null) {
            return false;
        }
        return removed;
    }

    public abstract Enum<?> getEntityState();

    public abstract EOrderType getPublicType();

    public boolean canCalculateTotalCost() {
        return getOrderItems().stream().allMatch(OrderItem::canCalculateTotalCost);
    }

    public Money calculateInitialPaymentAmount() {
        if (getPaymentSchedule() != null) {
            return getPaymentSchedule().getInitialPendingInvoice().getTotalAmount();
        } else if (isFullyPostPaid()) {
            return Money.zero(currency);
        } else {
            return calculateTotalCost();
        }
    }

    public Money calculateTotalCost() {
        return getOrderItems().stream().map(OrderItem::calculateTotalCostWithPreliminaryFallback).reduce(Money::add).get();
    }

    public Money calculateOriginalTotalCost() {
        return getOrderItems().stream().map(OrderItem::calculateOriginalCostWithPreliminaryFallback).reduce(Money::add).get();
    }

    public Money calculatePaidAmount() {
        return getPayments().stream().map(Payment::getPaidAmount).reduce(Money::add).orElse(Money.zero(getCurrency()));
    }

    public Money calculateStrikeThroughDiscountAmount() {
        return getOrderItems().stream()
                .flatMap(oi -> oi.getFiscalItems().stream().flatMap(fi -> fi.getFiscalItemDiscounts().stream().filter(fid -> fid.getPromoCodeApplication() == null)))
                .map(FiscalItemDiscount::getDiscount)
                .reduce(Money::add)
                .orElseGet(() -> Money.zero(ProtoCurrencyUnit.RUB));
    }

    public Money calculatePromoDiscountAmount() {
        return getPromoCodeApplications().stream()
                .filter(pa -> pa.getApplicationResultType() == EPromoCodeApplicationResultType.ART_SUCCESS)
                .map(PromoCodeApplication::getDiscount).reduce(Money::add)
                .orElseGet(() -> Money.zero(ProtoCurrencyUnit.RUB)); //TODO (mbobrov): remove hardcoded RUB
    }

    public Money calculateDiscountAmount() {
        return calculatePromoDiscountAmount().add(calculateStrikeThroughDiscountAmount());
    }

    public Money calculateDiscountRemainder() {
        return getPromoCodeApplications().stream()
                .filter(pa -> pa.getApplicationResultType() == EPromoCodeApplicationResultType.ART_SUCCESS)
                .map(PromoCodeApplication::getDiscountRemainder).reduce(Money::add)
                .orElseGet(() -> Money.zero(ProtoCurrencyUnit.RUB));
    }


    /**
     * Falls back to item payloads for non reserved orders
     *
     * @return calculated amount for cpa platform
     */
    public Money calculateTotalCostForCpa() {
        return getOrderItems().stream().map(OrderItem::calculateTotalCostWithPreliminaryFallback).reduce(Money::add).get();
    }

    public List<OrderItem> getOrderItems() {
        // the services should be properly ordered as the outer code relies on this ordering
        return orderItems == null ? Collections.emptyList() : orderItems.stream()
                .sorted(Comparator.comparing(OrderItem::getItemNumber))
                .collect(Collectors.toUnmodifiableList());
    }

    public List<OrderItem> getArchivedOrderItems() {
        return archivedOrderItems == null ? Collections.emptyList() : Collections.unmodifiableList(archivedOrderItems);
    }

    public List<PromoCodeApplication> getPromoCodeApplications() {
        return promoCodeApplications == null ? Collections.emptyList() :
                Collections.unmodifiableList(promoCodeApplications);
    }

    public List<OrderRefund> getOrderRefunds() {
        return orderRefunds == null ? Collections.emptyList() : orderRefunds;
    }

    public OrderLabelParams getOrderLabelParams() {
        if (orderLabelParams != null && orderLabelParams.size() > 0) {
            return orderLabelParams.get(0);
        } else {
            return null;
        }
    }

    public void addOrderItem(OrderItem item) {
        if (orderItems == null) {
            orderItems = new ArrayList<>();
        }
        if (archivedOrderItems == null) {
            archivedOrderItems = new ArrayList<>();
        }
        item.setOrder(this);
        if (this.getWorkflow() != null) {
            item.setOrderWorkflowId(this.getWorkflow().getId());
        }
        if (servicedAt == null || servicedAt.isAfter(item.getServicedAt())) {
            servicedAt = item.getServicedAt();
        }
        if (item.isArchived()) {
            archivedOrderItems.add(item);
        } else {
            orderItems.add(item);
        }
    }

    public void addOrderRefund(OrderRefund orderRefund) {
        if (orderRefunds == null) {
            orderRefunds = new ArrayList<>();
        }
        orderRefund.setOrder(this);
        orderRefunds.add(orderRefund);
    }

    public void addRequestedPromoCodeActivation(PromoCodeActivation promoCodeActivation) {
        var promoCodeApplication = new PromoCodeApplication();
        promoCodeApplication.setId(UUID.randomUUID());
        promoCodeApplication.setOrder(this);
        promoCodeApplication.setApplicationResultType(EPromoCodeApplicationResultType.ART_UNKNOWN);
        promoCodeApplication.setPromoCodeActivation(promoCodeActivation);
        if (promoCodeApplications == null) {
            promoCodeApplications = new ArrayList<>();
        }
        promoCodeApplications.add(promoCodeApplication);
    }

    public Option<OrderItem> findOrderItemById(UUID serviceId) {
        for (OrderItem orderItem : getOrderItems()) {
            if (orderItem.getId().equals(serviceId)) {
                return Option.of(orderItem);
            }
        }
        return Option.empty();
    }

    public boolean allItemsAreIn(@NonNull Set<Enum> states) {
        for (OrderItem orderItem : getOrderItems()) {
            if (!states.contains(orderItem.getItemState())) {
                return false;
            }
        }
        return true;
    }

    public boolean atLeastOneItemIsIn(@NonNull Set<Enum> state) {
        for (OrderItem orderItem : getOrderItems()) {
            if (state.contains(orderItem.getItemState())) {
                return true;
            }
        }
        return false;
    }

//    public void transitionWorkflowStateTo(OrderWorkflowState newState) {
//        OrderWorkflowStateTransition newTransition = new OrderWorkflowStateTransition();
//        newTransition.setFromState(orderWorkflowState);
//        newTransition.setToState(newState);
//        newTransition.setOrder(this);
//        if (orderWorkflowStateTransitions == null) {
//            orderWorkflowStateTransitions = new ArrayList<>();
//        }
//        orderWorkflowStateTransitions.add(newTransition);
//        orderWorkflowState = newState;
//    }

    public BigDecimal getExchangeRateFor(CurrencyUnit currency) {
        ProtoCurrencyUnit protoCurrencyUnit = ProtoCurrencyUnit.fromCurrencyCode(currency.getCurrencyCode());
        if (this.getCurrency() == null || this.getFxRate() == null) {
            throw new RuntimeException("Exchange rates not set");
        }
        if (this.getCurrency().equals(ProtoCurrencyUnit.fromProtoCurrencyUnit(protoCurrencyUnit.getProtoCurrency()))) {
            return BigDecimal.valueOf(1L, 0);
        } else {
            BigDecimal exchangeRate = this.getFxRate().get(protoCurrencyUnit.getProtoCurrency());
            if (exchangeRate == null) {
                throw new RuntimeException(String.format("Exchange rate for %s not set", currency.toString()));
            }
            return exchangeRate;
        }
    }

    public void refreshExpiresAt() {
        Optional<Instant> newExpiresAt = orderItems.stream()
                .filter((oi) -> oi.getExpiresAt() != null)
                .map(OrderItem::getExpiresAt)
                .min(Comparator.naturalOrder());

        newExpiresAt.ifPresent(localDateTime -> expiresAt = localDateTime);
    }

    public void addInvoice(Invoice invoice) {
        if (invoices == null) {
            invoices = new HashSet<>();
        }
        invoices.add(invoice);
        invoice.setOrder(this);
        if (this.getWorkflow() != null) {
            invoice.setOrderWorkflowId(this.getWorkflow().getId());
        }
        // every newly added invoice will become active
        currentInvoice = invoice;
    }

    public void addPendingInvoice(PendingInvoice pendingInvoice) {
        if (pendingInvoices == null) {
            pendingInvoices = new HashSet<>();
        }
        pendingInvoices.add(pendingInvoice);
        pendingInvoice.setOrder(this);
    }

    public List<Invoice> getInvoices() {
        return invoices == null ? Collections.emptyList() : List.copyOf(invoices);
    }

    public void setInvoices(List<Invoice> invoices) {
        this.invoices = new HashSet<>(invoices);
    }

    public List<Payment> getPayments() {
        return Stream.concat(
                        Stream.ofNullable(paymentSchedule),
                        Objects.requireNonNullElse(pendingInvoices, Collections.<PendingInvoice>emptySet())
                                .stream()
                                .sorted(Comparator.comparing(PendingInvoice::getCreatedAt)))
                .collect(Collectors.toList());
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }

    public List<MoneyRefund> getMoneyRefunds() {
        return moneyRefunds == null ? Collections.emptyList() : Collections.unmodifiableList(moneyRefunds);
    }

    public void addMoneyRefund(MoneyRefund moneyRefund) {
        Preconditions.checkArgument(moneyRefund.getOrder() == null, "MoneyRefund should be detached");
        if (moneyRefunds == null) {
            moneyRefunds = new ArrayList<>();
        }
        moneyRefunds.add(moneyRefund);
        moneyRefund.setOrder(this);
    }

    public MoneyRefund getPendingMoneyRefund() {
        if (moneyRefunds == null) {
            return null;
        }
        for (MoneyRefund refund : moneyRefunds) {
            if (refund.getState() == MoneyRefundState.PENDING) {
                return refund;
            }
        }
        return null;
    }

    public MoneyRefund getInProgressMoneyRefund() {
        if (moneyRefunds == null) {
            return null;
        }
        for (MoneyRefund refund : moneyRefunds) {
            if (refund.getState() == MoneyRefundState.IN_PROGRESS) {
                return refund;
            }
        }
        return null;
    }

    public boolean isBroken() {
        return this.getWorkflow() == null || this.getWorkflow().getState() == EWorkflowState.WS_CRASHED;
    }

    public boolean isMockPayment() {
        return mockPayment != null && mockPayment;
    }

    public boolean getUseDeferredPayment() {
        return Objects.requireNonNullElse(useDeferredPayment, false);
    }

    public boolean isFullyPostPaid() {
        return orderItems.stream().allMatch(OrderItem::isPostPaid);
    }

    public boolean isPostPayEligible() {
        return orderItems.stream().allMatch(OrderItem::isPostPayEligible);
    }

    public OrderExperiments getExperiments() {
        if (experiments == null) {
            return new OrderExperiments();
        }
        return experiments;
    }

    public List<KVExperiment> getKVExperiments() {
        if (kVExperiments == null) {
            return List.of();
        }
        return kVExperiments;
    }

    public void toggleUserActionScheduled(boolean newState) {
        Preconditions.checkState(userActionScheduled != newState,
                "The userActionScheduled flag is in an unexpected '%s' state", userActionScheduled);
        userActionScheduled = newState;
    }

    public EPaymentState getPaymentState() {
        if (getPaymentSchedule() == null) {
            Set<EPaymentState> states = getPayments().stream()
                    .map(Payment::getState)
                    .collect(Collectors.toSet());
            if (states.size() == 1) {
                return states.iterator().next();
            } else if (states.contains(EPaymentState.PS_FULLY_PAID) &&
                    !states.contains(EPaymentState.PS_CANCELLED)) {
                return EPaymentState.PS_PARTIALLY_PAID;
            } else {
                // ATM not quite clear how to handle complex cases.
                return EPaymentState.PS_UNKNOWN;
            }
        } else {
            return getPaymentSchedule().getState();
        }
    }

    public String getPrettyId() {
        return prettyId;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }
}
