package ru.yandex.travel.orders.entities;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

import javax.annotation.Nullable;
import javax.money.CurrencyUnit;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.DiscriminatorColumn;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.OrderColumn;
import javax.persistence.Table;
import javax.persistence.Version;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import lombok.AccessLevel;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.Columns;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.GenericGenerator;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.logging.LogEntity;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.partners.BillingPartnerAgreement;
import ru.yandex.travel.workflow.entities.Workflow;

@Entity
@Table(name = "order_items")
@Data
@EqualsAndHashCode(exclude = {"order"})
@ToString(exclude = {"order"})
@Inheritance(strategy = InheritanceType.SINGLE_TABLE)
@DiscriminatorColumn(name = "item_type")
public abstract class OrderItem implements LogEntity {
    @Id
    @GeneratedValue(
            generator = "uuid"
    )
    @GenericGenerator(name = "uuid", strategy = "org.hibernate.id.UUIDGenerator")
    private UUID id;

    private Integer itemNumber;

    @Column(name = "item_type", updatable = false, insertable = false)
    private String type;

    @Nullable
    private String providerId;

    @ManyToOne
    private Order order;

    private UUID orderWorkflowId;

    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    private Instant expiresAt;

    private boolean isExpired;

    /**
     * The time when a partner approved that the order has been created.
     * <p>
     * Typically, a few seconds after the 1st payment has been committed.
     */
    private Instant confirmedAt;
    private Instant refundedAt;

    @OneToMany(mappedBy = "orderItem", cascade = CascadeType.ALL)
    @OrderColumn(name = "item_position")
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    @BatchSize(size = 100)
    private List<FiscalItem> fiscalItems;

    @Version
    private Integer version;

    @OneToOne
    private Workflow workflow;

    private Instant lastTransitionAt;

    @Type(type = "protobuf-value")
    @Columns(columns = {
            @Column(name = "test_context_class_name"), @Column(name = "test_context_data")
    })
    private Message testContext;

    private boolean archived;

    public Money totalCostAfterReservation() {
        Preconditions.checkArgument(!getFiscalItems().isEmpty(),
                "Cannot calculate total cost: no fiscal items for this order item; itemType=%s, id=%s",
                getClass().getSimpleName(), getId());
        return getFiscalItems().stream()
                .map(FiscalItem::getMoneyAmount)
                .reduce(Money::add).get();
    }

    /**
     * We need to recalculate discount by promo code, therefore we need to get the correct price without promocode
     * but with discount by special promo campaigns.
     */
    public Money totalCostAfterReservationExceptPromo() {
        Preconditions.checkArgument(!getFiscalItems().isEmpty(),
                "Cannot calculate total cost: no fiscal items for this order item; itemType=%s, id=%s",
                getClass().getSimpleName(), getId());
        return getFiscalItems().stream()
                .map(FiscalItem::getMoneyAmountExceptPromo)
                .reduce(Money::add).get();
    }

    public Money originalCostAfterReservation() {
        Preconditions.checkArgument(!getFiscalItems().isEmpty(),
                "Cannot calculate total cost: no fiscal items for this order item; itemType=%s, id=%s",
                getClass().getSimpleName(), getId());
        return getFiscalItems().stream()
                .map(FiscalItem::getOriginalCost)
                .reduce(Money::add).get();
    }

    public Money calculateTotalCostWithPreliminaryFallback() {
        if (getFiscalItems().isEmpty()) {
            return preliminaryTotalCost();
        } else {
            return totalCostAfterReservation();
        }
    }

    public Money calculateOriginalCostWithPreliminaryFallback() {
        if (getFiscalItems().isEmpty()) {
            return preliminaryTotalCost();
        } else {
            return originalCostAfterReservation();
        }
    }

    public boolean canCalculateTotalCost() {
        return !getFiscalItems().isEmpty();
    }

    // Total cost usually estimated before the booking, we can get if from payload
    public abstract Money preliminaryTotalCost();

    public List<FiscalItem> getFiscalItems() {
        return fiscalItems == null ? Collections.emptyList() : Collections.unmodifiableList(fiscalItems);
    }

    public void addFiscalItem(FiscalItem fiscalItem) {
        if (fiscalItems == null) {
            fiscalItems = new ArrayList<>();
        }
        Preconditions.checkState(fiscalItem.getOrderItem() == null, "Fiscal item must be unbound");
        fiscalItem.setOrderItem(this);
        fiscalItems.add(fiscalItem);
    }

    public void removeFiscalItem(FiscalItem fiscalItem) {
        Preconditions.checkState(fiscalItems != null, "Fiscal items should be initialized");
        Preconditions.checkArgument(fiscalItem.getOrderItem().equals(this), "Fiscal item belongs to a wrong order item");
        fiscalItem.setOrderItem(null);
        fiscalItems.remove(fiscalItem);
    }

    public Money getTotalDiscount() {
        CurrencyUnit currency = preliminaryTotalCost().getCurrency();
        return getFiscalItems().stream()
                .map(FiscalItem::getDiscountAmount)
                .reduce(Money::add).orElse(Money.of(0, currency));
    }

    public MoneyMarkup getTotalMoneyMarkup() {
        CurrencyUnit currency = preliminaryTotalCost().getCurrency();
        return getFiscalItems().stream()
                .map(FiscalItem::getMoneyAmountMarkup)
                .reduce(MoneyMarkup::add).orElse(MoneyMarkup.zero(currency));
    }

    public abstract Enum<?> getItemState();

    public abstract EServiceType getPublicType();

    public abstract LocalDateTime getServicedAt();

    public Object getPayload() {
        return null;
    }

    public abstract BillingPartnerAgreement getBillingPartnerAgreement();

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }

    public int getItemNumber() {
        return itemNumber == null ? 0 : itemNumber;
    }

    public boolean isRebookingAllowed() {
        return false;
    }

    public Order getOrder() {
        return order;
    }

    public boolean isPostPaid() {
        return false;
    }

    public boolean isPostPayEligible() {
        return false;
    }
}
