package ru.yandex.travel.orders.entities;

import java.time.Instant;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.OrderColumn;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.proto.EPaymentType;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.workflow.entities.Workflow;

@Entity
@Table(name = "payment_schedules")
@Getter
@Setter
@AllArgsConstructor
@NoArgsConstructor
@Builder(toBuilder = true)
@BatchSize(size = 100)
public class PaymentSchedule implements Payment {
    @Id
    private UUID id;

    @OneToOne(mappedBy = "paymentSchedule")
    private Order order;

    @Type(type = "proto-enum")
    private EPaymentState state;

    @OneToOne(cascade = CascadeType.ALL)
    @BatchSize(size = 100)
    private PendingInvoice initialPendingInvoice;

    @OneToMany(mappedBy = "schedule", cascade = CascadeType.ALL)
    @OrderColumn(name = "item_position")
    @BatchSize(size = 100)
    private List<PaymentScheduleItem> items;

    private Boolean allowPaidItemsWhenCancelling;

    private boolean expired;

    @OneToOne
    private Workflow workflow;

    private Instant lastTransitionAt;

    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    @Version
    private Integer version;

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.PAYMENT_SCHEDULE.getDiscriminatorValue();
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }

    @Override
    public UUID getOwnerWorkflowId() {
        return order.getWorkflow().getId();
    }

    @Override
    public Money getTotalAmount() {
        return getInitialPendingInvoice().getTotalAmount();
    }

    @Override
    public MoneyMarkup getTotalAmountMarkup() {
        return getInitialPendingInvoice().getTotalAmountMarkup();
    }

    @Override
    public Money getPaidAmount() {
        return getAllInvoices().stream().map(PendingInvoice::getPaidAmount).reduce(Money::add).orElse(Money.zero(getOrder().getCurrency()));
    }

    public List<PendingInvoice> getAllInvoices() {
        return Stream.concat(Stream.of(initialPendingInvoice),
                items.stream().map(PaymentScheduleItem::getPendingInvoice))
                .collect(Collectors.toList());
    }

    @Override
    public Set<UUID> getLogEntityIds() {
        return Stream.concat(
                Stream.of(this.id),
                getAllInvoices().stream().map(PendingInvoice::getId))
                .collect(Collectors.toSet());
    }

    @Override
    public Invoice getLastAttempt() {
        switch (state) {
            case PS_DRAFT:
            case PS_INVOICE_PENDING:
            case PS_PAYMENT_IN_PROGRESS:
                return initialPendingInvoice.getLastAttempt();
            case PS_PARTIALLY_PAID:
            case PS_FULLY_PAID:
                return items.stream()
                        .map(i -> i.getPendingInvoice().getLastAttempt())
                        .filter(Objects::nonNull)
                        .reduce(null, (p, n) -> n); // get last in stream
            default:
                return getAllInvoices().stream()
                        .map(PendingInvoice::getLastAttempt)
                        .filter(Objects::nonNull)
                        .reduce(null, (r, n) -> n);
        }
    }

    @Override
    public Invoice getLastPaidAttempt() {
        switch (state) {
            case PS_PARTIALLY_PAID:
            case PS_PAYMENT_IN_PROGRESS:
                return initialPendingInvoice.getLastPaidAttempt();
            case PS_FULLY_PAID:
                return  items.stream()
                    .map(i -> i.getPendingInvoice().getLastPaidAttempt())
                    .filter(Objects::nonNull)
                    .reduce(null, (p, n) -> n); // get last i
            default:
                return null;
        }
    }

    public boolean getAllowPaidItemsWhenCancelling() {
        if (allowPaidItemsWhenCancelling == null) {
            return false;
        }
        return allowPaidItemsWhenCancelling;
    }

    @Override
    public List<Payment> getNextPayments() {
        if (items == null) {
            return Collections.emptyList();
        } else {
            return items.stream()
                    .filter(i -> i.getPendingInvoice().getState() != EPaymentState.PS_CANCELLED)
                    .map(paymentScheduleItem ->
                            paymentScheduleItem.getPendingInvoice().toBuilder()
                                    .expiresAt(paymentScheduleItem.getPaymentEndsAt())
                                    .build())
                    .collect(Collectors.toList());

        }
    }

    @Override
    public List<FiscalReceipt> getReceipts() {
        return getAllInvoices().stream()
                .map(PendingInvoice::getLastAttempt)
                .filter(Objects::nonNull)
                .flatMap(i -> i.getFiscalReceipts().stream())
                .collect(Collectors.toList());
    }

    @Override
    public Instant getPaymentEndsAt() {
        return initialPendingInvoice.getPaymentEndsAt();
    }

    @Override
    public Instant getClosedAt() {
        switch (state) {
            case PS_FULLY_PAID:
                return items.stream()
                        .filter(i -> i.getPendingInvoice().getState() == EPaymentState.PS_FULLY_PAID && i.getPendingInvoice().getClosedAt() != null)
                        .map(i -> i.getPendingInvoice().getClosedAt())
                        .max(Comparator.comparing(Function.identity()))
                        .orElse(null);
            case PS_CANCELLED:
                return items.stream()
                        .filter(i -> i.getPendingInvoice().getState() == EPaymentState.PS_CANCELLED && i.getPendingInvoice().getClosedAt() != null)
                        .map(i -> i.getPendingInvoice().getClosedAt())
                        .max(Comparator.comparing(Function.identity()))
                        .orElse(null);
            default:
                return null;
        }
    }

    @Override
    public EPaymentType getPaymentType() {
        return EPaymentType.PT_PAYMENT_SCHEDULE;
    }
}
