package ru.yandex.travel.orders.entities;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.OrderColumn;
import javax.persistence.Table;
import javax.persistence.Version;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import org.apache.commons.collections.CollectionUtils;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.proto.EPaymentType;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.workflow.entities.Workflow;

@Entity
@Table(name = "pending_invoices")
@Getter
@Setter
@AllArgsConstructor
@NoArgsConstructor
@Builder(toBuilder = true)
@BatchSize(size = 100)
public class PendingInvoice implements Payment {
    @Id
    private UUID id;

    private String title;

    @ManyToOne
    private Order order;

    @ManyToOne(cascade = CascadeType.ALL)
    private PaymentSchedule paymentSchedule;

    @OneToMany(mappedBy = "pendingInvoice", cascade = CascadeType.ALL)
    @OrderColumn(name = "payment_attempt_number")
    @BatchSize(size = 100)
    private List<Invoice> paymentAttempts;

    @OneToMany(mappedBy = "invoice", cascade = CascadeType.ALL)
    @OrderColumn(name = "item_position")
    @BatchSize(size = 100)
    private List<PendingInvoiceItem> pendingInvoiceItems;

    @Type(type = "proto-enum")
    private EPaymentState state;

    @OneToOne
    private Workflow workflow;

    private Instant lastTransitionAt;

    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    private Instant closedAt;

    private Instant expiresAt;

    @Version
    private Integer version;

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.PENDING_INVOICE.getDiscriminatorValue();
    }

    @Override
    public Money getTotalAmount() {
        return pendingInvoiceItems.stream().map(PendingInvoiceItem::getPrice).reduce(Money::add).orElse(Money.zero(getOwnerOrder().getCurrency()));
    }

    @Override
    public MoneyMarkup getTotalAmountMarkup() {
        return pendingInvoiceItems.stream()
                .map(PendingInvoiceItem::getMoneyAmountMarkup)
                .reduce(MoneyMarkup::add)
                .orElse(MoneyMarkup.zero(getOwnerOrder().getCurrency()));
    }

    @Override
    public Money getPaidAmount() {
        if (state == EPaymentState.PS_FULLY_PAID) {
            return getTotalAmount();
        } else {
            return Money.zero(getOwnerOrder().getCurrency());
        }
    }

    @Override
    public Set<UUID> getLogEntityIds() {
        return Set.of(id);
    }

    @Override
    public Invoice getLastAttempt() {
        if (paymentAttempts == null || paymentAttempts.size() == 0) {
            return null;
        } else {
            return paymentAttempts.get(paymentAttempts.size() - 1);
        }
    }

    @Override
    public Invoice getLastPaidAttempt() {
        if (paymentAttempts == null || state != EPaymentState.PS_FULLY_PAID) {
            return null;
        } else {
            return getLastAttempt();
        }
    }

    @Override
    public List<Payment> getNextPayments() {
        return Collections.emptyList();
    }

    public Order getOwnerOrder() {
        if (order != null) {
            return order;
        } else {
            return paymentSchedule.getOrder();
        }
    }

    public void addFullItemForFiscalItem(FiscalItem fiscalItem) {
        addItemForFiscalItem(fiscalItem, fiscalItem.getMoneyAmount(), fiscalItem.getYandexPlusToWithdraw());
    }

    public void addItemForFiscalItem(FiscalItem fiscalItem, Money amount, Money yandexPlusToWithdraw) {
        if (pendingInvoiceItems == null) {
            pendingInvoiceItems = new ArrayList<>();
        }
        pendingInvoiceItems.add(PendingInvoiceItem.builder()
                .fiscalItem(fiscalItem)
                .price(amount)
                .yandexPlusToWithdraw(yandexPlusToWithdraw)
                .invoice(this)
                .build());
    }

    public void addAttempt(Invoice attempt) {
        Preconditions.checkState(attempt.getPendingInvoice() == null, "Attempt belongs to another invoice");
        attempt.setPendingInvoice(this);
        if (paymentAttempts == null) {
            paymentAttempts = new ArrayList<>();
        }
        paymentAttempts.add(attempt);
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }

    public List<Invoice> getPaymentAttempts() {
        if (paymentAttempts == null) {
            return Collections.emptyList();
        } else {
            return Collections.unmodifiableList(paymentAttempts);
        }
    }

    @Override
    public Instant getPaymentEndsAt() {
        return expiresAt;
    }

    @Override
    public EPaymentType getPaymentType() {
        return EPaymentType.PT_PENDING_INVOICE;
    }

    public boolean isClosed() {
        return state == EPaymentState.PS_FULLY_PAID || state == EPaymentState.PS_CANCELLED;
    }

    /**
     * @return true if there's a single 0 pending invoice item. Means that a schedule with first 0 payment
     * has been used.
     */
    public boolean isZeroPayment() {
        return CollectionUtils.size(getPendingInvoiceItems()) == 1 &&
                getPendingInvoiceItems().get(0).getPrice() != null &&
                getPendingInvoiceItems().get(0).getPrice().isZero();
    }

    @Override
    public UUID getOwnerWorkflowId() {
        if (order != null) {
            return order.getWorkflow().getId();
        }
        if (paymentSchedule != null) {
            return paymentSchedule.getWorkflow().getId();
        }
        throw new IllegalStateException("Owner not set");
    }

    @Override
    public List<FiscalReceipt> getReceipts() {
        var last = getLastAttempt();
        if (last == null) {
            return Collections.emptyList();
        } else {
            return last.getFiscalReceipts();
        }
    }
}
