package ru.yandex.travel.orders.entities;

import java.time.LocalDateTime;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.SequenceGenerator;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import org.hibernate.annotations.Columns;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;
import org.javamoney.moneta.Money;

@Entity
@Table(name = "pending_invoice_items")
@Getter
@Setter
@AllArgsConstructor
@NoArgsConstructor
@Builder(toBuilder = true)
public class PendingInvoiceItem implements InvoiceItemProvider {
    @Id
    @GeneratedValue(
            strategy = GenerationType.SEQUENCE,
            generator = "pending_invoice_items_id_seq"
    )
    @SequenceGenerator(
            name = "pending_invoice_items_id_seq",
            sequenceName = "pending_invoice_items_id_seq",
            allocationSize = 1
    )
    private Long id;

    @ManyToOne
    private PendingInvoice invoice;

    @ManyToOne
    private FiscalItem fiscalItem;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "price_amount"), @Column(name = "price_currency")
    })
    private Money price;

    // The value is also included in the total 'price' field (a part of it)
    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "yandex_plus_withdraw_amount"), @Column(name = "yandex_plus_withdraw_currency")
    })
    private Money yandexPlusToWithdraw;

    @CreationTimestamp
    private LocalDateTime createdAt;

    @UpdateTimestamp
    private LocalDateTime updatedAt;

    @Version
    private Integer version;

    @Override
    public Long getFiscalItemId() {
        return fiscalItem.getId();
    }

    @Override
    public Money getMoneyAmount() {
        return price;
    }

    public MoneyMarkup getMoneyAmountMarkup() {
        return MoneyMarkup.builder()
                .card(price.subtract(getYandexPlusToWithdraw()))
                .yandexAccount(getYandexPlusToWithdraw())
                .build();
    }

    @Override
    public Money getYandexPlusToWithdraw() {
        return yandexPlusToWithdraw != null ? yandexPlusToWithdraw : Money.zero(price.getCurrency());
    }

    @Override
    public FiscalItemType getType() {
        return fiscalItem.getType();
    }

    @Override
    public String getTitle() {
        return fiscalItem.getTitle();
    }

    @Override
    public VatType getVatType() {
        return fiscalItem.getVatType();
    }

    @Override
    public String getInn() {
        return fiscalItem.getInn();
    }
}
