package ru.yandex.travel.orders.entities;

import java.time.Instant;
import java.util.List;
import java.util.UUID;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.Data;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;

import ru.yandex.travel.orders.workflow.orderitem.train.insurancerefund.proto.ETrainInsuranceRefundState;
import ru.yandex.travel.train.model.refund.InsuranceItemInfo;
import ru.yandex.travel.train.model.refund.InsuranceRefund;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

@Entity
@Table(name = "train_insurance_refunds")
@Data
public class TrainInsuranceRefund implements WorkflowEntity<ETrainInsuranceRefundState> {
    @Id
    private UUID id;

    @OneToOne
    private Workflow workflow;

    @Type(type = "proto-enum")
    private ETrainInsuranceRefundState state;

    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    private Instant lastTransitionAt;

    private Instant nextCheckAt;

    private boolean backgroundJobActive;

    @ManyToOne
    private OrderRefund orderRefund;

    @ManyToOne
    private OrderItem orderItem;

    @Type(type = "jsonb-object")
    private InsuranceRefund payload;

    @Version
    private Integer version;

    public static TrainInsuranceRefund createInsuranceAutoRefund(TrainOrderItem orderItem,
                                                                 List<InsuranceItemInfo> itemsToRefund) {
        return createInsuranceRefund(orderItem, itemsToRefund, null);
    }

    public static TrainInsuranceRefund createInsuranceRefund(TrainOrderItem orderItem,
                                                             List<InsuranceItemInfo> itemsToRefund,
                                                             OrderRefund orderRefund) {
        var refund = new TrainInsuranceRefund();
        refund.setId(UUID.randomUUID());
        refund.setState(ETrainInsuranceRefundState.RS_NEW);
        refund.setOrderItem(orderItem);
        refund.setOrderRefund(orderRefund);
        var payload = new InsuranceRefund();
        refund.setPayload(payload);
        payload.setPartnerOrderId(orderItem.getPayload().getPartnerOrderId());
        payload.setCheckRefundCounter(0);
        payload.setItems(itemsToRefund);
        return refund;
    }

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.TRAIN_INSURANCE_REFUND.getDiscriminatorValue();
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }
}
