package ru.yandex.travel.orders.entities;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.OneToMany;

import com.google.common.annotations.VisibleForTesting;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.hibernate.annotations.Type;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.partners.TrainBillingPartnerAgreement;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

@DiscriminatorValue(WellKnownOrderItemDiscriminator.ORDER_ITEM_TRAIN)
@Getter
@Setter
@Entity
public class TrainOrderItem extends OrderItem implements WorkflowEntity<EOrderItemState> {
    @Type(type = "proto-enum")
    private EOrderItemState state;

    @Type(type = "jsonb-object")
    @Column(name = "payload")
    private TrainReservation reservation;

    private Instant nextCheckConfirmedAt;

    private boolean backgroundJobActive;

    @OneToMany(mappedBy = "orderItem", cascade = CascadeType.ALL)
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    private List<TrainTicketRefund> trainTicketRefunds;

    @Type(type = "jsonb-object")
    @Column(name = "agreement")
    private TrainBillingPartnerAgreement billingPartnerAgreement;

    @Override
    public TrainReservation getPayload() {
        return reservation;
    }

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.TRAIN_ORDER_ITEM.getDiscriminatorValue();
    }

    @Override
    public EServiceType getPublicType() {
        return EServiceType.PT_TRAIN;
    }

    @Override
    public LocalDateTime getServicedAt() {
        if (reservation.getReservationRequestData() == null || reservation.getReservationRequestData().getDepartureTime() == null) {
            return null;
        }
        return reservation.getReservationRequestData().getDepartureTime().atOffset(ZoneOffset.UTC).toLocalDateTime();
    }

    @Override
    public EOrderItemState getItemState() {
        return state;
    }

    public void setReservation(TrainReservation reserv) {
        this.reservation = reserv;
    }

    public List<TrainTicketRefund> getTrainTicketRefunds() {
        return trainTicketRefunds == null ? Collections.emptyList() : Collections.unmodifiableList(trainTicketRefunds);
    }

    /**
     * @apiNote This is only for testing. Do not use in production code
     */
    @VisibleForTesting
    void setTrainTicketRefunds(List<TrainTicketRefund> trainTicketRefunds) {
        this.trainTicketRefunds = new ArrayList<>();
        this.trainTicketRefunds.addAll(trainTicketRefunds);
    }

    @Override
    public Money totalCostAfterReservation() {
        Money result = Money.of(BigDecimal.ZERO, ProtoCurrencyUnit.RUB);
        for (var passenger : this.reservation.getPassengers()) {
            var ticket = passenger.getTicket();
            if (ticket != null) {
                result = result.add(ticket.calculateTotalCost());
            }
            if (passenger.getInsurance() != null &&
                    this.reservation.getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
                result = result.add(passenger.getInsurance().getAmount());
            }
        }
        return result;
    }

    @Override
    public Money preliminaryTotalCost() {
        // TODO (mbobrov): we can't estimate total cost for the moment
        return Money.zero(ProtoCurrencyUnit.RUB);
    }

    public boolean isRebookingAllowed() {
        return isExpired();
    }
}
