package ru.yandex.travel.orders.entities;

import java.time.Instant;
import java.util.List;
import java.util.UUID;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.Data;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;

import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.train.model.refund.TicketRefund;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

@Entity
@Table(name = "train_ticket_refunds")
@Data
public class TrainTicketRefund implements WorkflowEntity<ETrainTicketRefundState> {
    @Id
    private UUID id;

    @OneToOne
    private Workflow workflow;

    @Type(type = "proto-enum")
    private ETrainTicketRefundState state;

    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    private Instant lastTransitionAt;

    private Instant nextCheckAt;

    private boolean backgroundJobActive;

    @ManyToOne
    private TrainOrderItem orderItem;

    private UUID orderItemWorkflowId;

    @Type(type = "jsonb-object")
    private TicketRefund payload;

    @ManyToOne
    private OrderRefund orderRefund;

    @Version
    private Integer version;

    public static TrainTicketRefund createRefund(TrainOrderItem orderItem,
                                                 List<PassengerRefundInfo> itemsToRefund,
                                                 OrderRefund orderRefund) {
        var refund = new TrainTicketRefund();
        refund.setId(UUID.randomUUID());
        refund.setState(ETrainTicketRefundState.RS_NEW);
        refund.setOrderItem(orderItem);
        refund.setOrderItemWorkflowId(orderItem.getWorkflow().getId());
        refund.setOrderRefund(orderRefund);
        var payload = new TicketRefund();
        refund.setPayload(payload);
        payload.setCheckDocumentNumber(orderItem.getPayload().getPassengers().get(0).getDocumentNumber());
        payload.setPartnerOrderId(orderItem.getPayload().getPartnerOrderId());
        // TODO(ganintsev): deprecated setBuyOperationId for backward compatibility with old orders
        payload.setBuyOperationId(orderItem.getPayload().getPartnerBuyOperationId());
        payload.setCheckRefundCounter(0);
        payload.setItems(itemsToRefund);
        return refund;
    }

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.TRAIN_TICKET_REFUND.getDiscriminatorValue();
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }
}
