package ru.yandex.travel.orders.entities;

import java.time.LocalDateTime;

import javax.persistence.Column;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;

import lombok.Getter;
import lombok.Setter;
import org.hibernate.annotations.Type;

import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.partners.BillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreementProvider;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.workflow.entities.WorkflowEntity;


@Entity
@DiscriminatorValue(WellKnownOrderItemDiscriminator.ORDER_ITEM_TRAVELLINE)
@Getter
@Setter
public class TravellineOrderItem extends HotelOrderItem implements WorkflowEntity<ETravellineItemState>,
        DirectHotelBillingPartnerAgreementProvider {
    @Type(type = "proto-enum")
    private ETravellineItemState state;

    @Type(type = "jsonb-object")
    @Column(name = "payload")
    private TravellineHotelItinerary itinerary;

    @Type(type = "jsonb-object")
    @Column(name = "agreement")
    private DirectHotelBillingPartnerAgreement agreement;

    @Override
    public Object getPayload() {
        return itinerary;
    }

    @Override
    public ETravellineItemState getItemState() {
        return state;
    }

    @Override
    public EServiceType getPublicType() {
        return EServiceType.PT_TRAVELLINE_HOTEL;
    }

    @Override
    public LocalDateTime getServicedAt() {
        return itinerary.getOrderDetails().getCheckoutDate().atStartOfDay();
    }

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.TRAVELLINE_ORDER_ITEM.getDiscriminatorValue();
    }

    @Override
    public ETravellineItemState getState() {
        return state;
    }

    @Override
    public void setState(ETravellineItemState state) {
        this.state = state;
    }

    @Override
    public HotelOrderItemState getPublicState() {
        //noinspection DuplicatedCode
        switch (getState()) {
            case IS_RESERVED:
                return HotelOrderItemState.RESERVED;
            case IS_NEW:
                return HotelOrderItemState.NEW;
            case IS_CONFIRMED:
                return HotelOrderItemState.CONFIRMED;
            case IS_CANCELLED:
                return HotelOrderItemState.CANCELLED;
            case IS_REFUNDED:
                return HotelOrderItemState.REFUNDED;
            default:
                return HotelOrderItemState.OTHER;
        }
    }

    @Override
    public TravellineHotelItinerary getHotelItinerary() {
        return itinerary;
    }

    @Override
    public EPartnerId getPartnerId() {
        return EPartnerId.PI_TRAVELLINE;
    }

    @Override
    public BillingPartnerAgreement getBillingPartnerAgreement() {
        return agreement;
    }

}
