package ru.yandex.travel.orders.entities;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;

import com.google.common.base.Strings;
import lombok.Getter;
import lombok.Setter;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.Type;

import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.payments.InvoicePaymentFlags;
import ru.yandex.travel.orders.services.payments.PaymentProfile;
import ru.yandex.travel.orders.services.payments.model.TrustTerminalForPartner;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

/**
 * Attempt to pay for the order in Trust.
 */
@Entity
@DiscriminatorValue(WellKnownInvoiceDiscriminator.INVOICE_TRUST)
@Getter
@Setter
@BatchSize(size = 100)
public class TrustInvoice extends Invoice implements WorkflowEntity<ETrustInvoiceState> {
    public final static Set<ETrustInvoiceState> UNPAID_INVOICE_STATES = Set.of(
            ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED,
            ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
            ETrustInvoiceState.IS_NEW);
    public final static Set<Enum<?>> PENDING_INVOICE_STATES = Set.of(
            ETrustInvoiceState.IS_NEW, ETrustInvoiceState.IS_WAIT_FOR_PAYMENT);
    private final static String DEFAULT_PAYMENT_METHOD_ID = "trust_web_page";
    private String rrn;

    private Instant clearingScheduledAt;

    private Boolean requireRubSettlement;

    private Boolean forceThreeDs;

    private Boolean useMirPromo;

    private Boolean processThroughYt;

    @Enumerated(EnumType.STRING)
    private TrustTerminalForPartner terminalForPartner;

    private String actualPaymentMethodId;

    private Instant trustClearedAt;

    private Instant trustRealClearedAt;

    @Type(type = "proto-enum")
    private ETrustInvoiceState state = ETrustInvoiceState.IS_NEW;

    // used by tests only so far
    public static TrustInvoice createEmptyInvoice() {
        TrustInvoice invoice = new TrustInvoice();
        invoice.setId(UUID.randomUUID());
        return invoice;
    }

    // used by tests only so far
    public static TrustInvoice createInvoice(Order order, AuthorizedUser owner,
                                             InvoicePaymentFlags paymentFlags) {
        return createInvoice(order, owner,
                order.getOrderItems().stream().flatMap(oi -> oi.getFiscalItems().stream()).collect(Collectors.toList()),
                paymentFlags);
    }

    public static TrustInvoice createInvoice(Order order, AuthorizedUser owner, List<InvoiceItemProvider> source,
                                             InvoicePaymentFlags paymentFlags) {
        TrustInvoice trustInvoice = new TrustInvoice();
        if (OrderCompatibilityUtils.isHotelOrder(order)) {
            EServiceType hotelServiceType = OrderCompatibilityUtils.getOnlyOrderItem(order).getPublicType();
            switch (hotelServiceType) {
                case PT_TRAVELLINE_HOTEL:
                case PT_DOLPHIN_HOTEL:
                case PT_BNOVO_HOTEL:
                case PT_BRONEVIK_HOTEL:
                    trustInvoice.setPaymentProfile(PaymentProfile.HOTEL_RUB_SETTLEMENT);
                    trustInvoice.setRequireRubSettlement(true);
                    break;
                case PT_EXPEDIA_HOTEL:
                    trustInvoice.setPaymentProfile(PaymentProfile.HOTEL);
                    break;
                default:
                    throw new RuntimeException(
                            String.format("Don't know which profile to use with the order containing service %s",
                                    hotelServiceType));
            }
        } else if (OrderCompatibilityUtils.isTrainOrder(order)) {
            trustInvoice.setPaymentProfile(PaymentProfile.TRAIN);
            trustInvoice.setProcessThroughYt(paymentFlags.isProcessThroughYt());
        } else if (OrderCompatibilityUtils.isAeroexpressOrder(order)) {
            trustInvoice.setPaymentProfile(PaymentProfile.AEROEXPRESS);
        } else if (OrderCompatibilityUtils.isSuburbanOrder(order)) {
            trustInvoice.setPaymentProfile(PaymentProfile.SUBURBAN);
            trustInvoice.setTerminalForPartner(paymentFlags.getTerminalForPartner());
        } else if (OrderCompatibilityUtils.isBusOrder(order)) {
            trustInvoice.setPaymentProfile(PaymentProfile.BUS);
        } else {
            throw new RuntimeException(
                    String.format("Don't know which profile to use with the order of type %s", order.getClass())
            );
        }
        trustInvoice.setForceThreeDs(paymentFlags.isForce3ds());
        trustInvoice.setNewPaymentWebForm(paymentFlags.isNewCommonPaymentWebForm());
        trustInvoice.useMirPromo = paymentFlags.isUseMirPromo();
        return initInvoice(order, trustInvoice, source, owner);
    }

    public String getEntityType() {
        return WellKnownWorkflowEntityType.TRUST_INVOICE.getDiscriminatorValue();
    }

    @Override
    public EInvoiceType getPublicType() {
        return EInvoiceType.IT_TRUST;
    }

    @Override
    public ETrustInvoiceState getInvoiceState() {
        return state;
    }

    public String getRequestedPaymentMethodId() {
        if (Strings.isNullOrEmpty(requestedPaymentMethodId)) {
            return DEFAULT_PAYMENT_METHOD_ID;
        }
        return requestedPaymentMethodId;
    }
}
