package ru.yandex.travel.orders.entities;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.UUID;

import javax.persistence.CascadeType;
import javax.persistence.DiscriminatorColumn;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Version;

import com.google.common.base.Preconditions;
import lombok.AccessLevel;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.GenericGenerator;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.logging.LogEntity;
import ru.yandex.travel.workflow.entities.Workflow;

import static java.math.BigDecimal.ZERO;
import static ru.yandex.travel.commons.lang.ComparatorUtils.isEqualByCompareTo;

@Entity
@Table(name = "trust_refunds")
@Data
@Inheritance(strategy = InheritanceType.SINGLE_TABLE)
@DiscriminatorColumn(name = "type")
@EqualsAndHashCode(exclude = {"invoice"})
@ToString(exclude = {"invoice"})
public abstract class TrustRefund implements LogEntity {
    @Id
    @GeneratedValue(
            generator = "uuid"
    )
    @GenericGenerator(name = "uuid", strategy = "org.hibernate.id.UUIDGenerator")
    private UUID id;

    @ManyToOne
    private Invoice invoice;
    @OneToOne
    private Workflow workflow;

    private String trustRefundId;
    private Instant trustConfirmTs;

    private String description;


    @OneToMany(mappedBy = "trustRefund", cascade = CascadeType.ALL)
    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    protected List<TrustRefundItem> refundItems;

    @CreationTimestamp
    private Instant createdAt;

    private Instant lastTransitionAt;
    private UUID orderRefundId;

    @Version
    private Integer version;

    public abstract Enum<?> getItemState();

    public void addRefundItem(String trustOrderId, BigDecimal originalAmount, BigDecimal targetAmount,
                              MoneyMarkup originalMarkup, MoneyMarkup targetMarkup) {
        if (targetMarkup == null) {
            // it's okay not to have a markup for a full refund;
            // eventually only markup values should be passed here
            Preconditions.checkArgument(originalMarkup.isCardOnly() || isEqualByCompareTo(targetAmount, ZERO),
                    "Non full item refund with plus points has to specify a money markup; " +
                            "trust order id %s, current %s", trustOrderId, originalMarkup);
            targetMarkup = MoneyMarkup.cardOnly(Money.of(targetAmount, originalMarkup.getTotal().getCurrency()));
        }
        Preconditions.checkArgument(isEqualByCompareTo(originalAmount, originalMarkup.getTotal().getNumberStripped()),
                "Original amount doesn't match its markup: total %s, markup %s",
                originalAmount, originalMarkup.getTotal().getNumberStripped());
        Preconditions.checkArgument(isEqualByCompareTo(targetAmount, targetMarkup.getTotal().getNumberStripped()),
                "Target amount doesn't match its markup: total %s, markup %s",
                targetAmount, targetMarkup.getTotal().getNumberStripped());
        TrustRefundItem trustRefundItem = new TrustRefundItem();
        trustRefundItem.setTrustOrderId(trustOrderId);
        trustRefundItem.setOriginalAmount(originalAmount);
        trustRefundItem.setTargetAmount(targetAmount);
        trustRefundItem.setOriginalYandexAccount(originalMarkup.getYandexAccount().getNumberStripped());
        trustRefundItem.setTargetYandexAccount(targetMarkup.getYandexAccount().getNumberStripped());
        trustRefundItem.setTrustRefund(this);
        if (refundItems == null) {
            refundItems = new ArrayList<>();
        }
        refundItems.add(trustRefundItem);
    }

    public BigDecimal getTotalRefundSum() {
        return refundItems.stream().map(TrustRefundItem::getRefundDelta).reduce(BigDecimal::add).orElseThrow();
    }

    public Money getTotalRefundMoney() {
        String currency = invoice.getOrder().getCurrency().getCurrencyCode();
        return Money.of(getTotalRefundSum(), currency);
    }

    public List<TrustRefundItem> getRefundItems() {
        if (refundItems == null) {
            return Collections.emptyList();
        } else {
            return Collections.unmodifiableList(refundItems);
        }
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }
}
