package ru.yandex.travel.orders.entities;

import java.math.BigDecimal;
import java.util.Objects;

import javax.money.CurrencyUnit;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.SequenceGenerator;
import javax.persistence.Table;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;
import org.javamoney.moneta.Money;

@Entity
@Table(name = "trust_refund_items")
@Data
@EqualsAndHashCode(exclude = {"trustRefund"})
@ToString(exclude = {"trustRefund"})
public class TrustRefundItem {
    @Id
    @GeneratedValue(
            strategy = GenerationType.SEQUENCE,
            generator = "trust_refund_items_id_seq"
    )
    @SequenceGenerator(
            name = "trust_refund_items_id_seq",
            sequenceName = "trust_refund_items_id_seq",
            allocationSize = 1
    )
    private Long id;

    @ManyToOne
    private TrustRefund trustRefund;

    private String trustOrderId;

    /**
     * originalAmount and targetAmount are total money values.
     * They include all specific money types such as plus points or any other.
     */
    private BigDecimal originalAmount;
    private BigDecimal targetAmount;

    /**
     * originalYandexAccount and targetYandexAccount are represent Yandex Plus points
     * to be returned by this refund operation. The values are include in originalAmount and targetAmount.
     */
    private BigDecimal originalYandexAccount;
    private BigDecimal targetYandexAccount;

    public BigDecimal getRefundDelta() {
        return originalAmount.subtract(targetAmount);
    }

    public MoneyMarkup getOriginalMarkup() {
        return getMarkup(originalAmount, originalYandexAccount);
    }

    public MoneyMarkup getTargetMarkup() {
        return getMarkup(targetAmount, targetYandexAccount);
    }

    private MoneyMarkup getMarkup(BigDecimal total, BigDecimal yandexAccount) {
        CurrencyUnit currency = trustRefund.getInvoice().getCurrency();
        yandexAccount = Objects.requireNonNullElse(yandexAccount, BigDecimal.ZERO);
        return MoneyMarkup.builder()
                .card(Money.of(total.subtract(yandexAccount), currency))
                .yandexAccount(Money.of(yandexAccount, currency))
                .build();
    }

    public MoneyMarkup getRefundDeltaMarkup() {
        return getOriginalMarkup().subtract(getTargetMarkup());
    }
}
