package ru.yandex.travel.orders.entities;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.Table;
import javax.persistence.Version;
import javax.validation.constraints.NotNull;

import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;

/**
 * Entity used for first order promo code restriction
 */
@Entity
@Table(name = "user_order_counters")
@Getter
@Setter
@Slf4j
public class UserOrderCounter {
    @Id
    private Long passportId;

    private Integer confirmedHotelOrders;
    private Integer confirmedAviaOrders;
    private Integer confirmedTrainOrders;
    private Integer confirmedBusOrders;
    private Integer confirmedSuburbanOrders;

    @Version
    private Long version;

    public static UserOrderCounter initForUser(Long passportId) {
        UserOrderCounter result = new UserOrderCounter();
        result.setPassportId(passportId);
        result.setConfirmedHotelOrders(0);
        result.setConfirmedAviaOrders(0);
        result.setConfirmedTrainOrders(0);
        result.setConfirmedBusOrders(0);
        result.setConfirmedSuburbanOrders(0);
        return result;
    }

    public void orderConfirmed(@NotNull EDisplayOrderType type) {
        switch (type) {
            case DT_HOTEL:
                confirmedHotelOrders++;
                break;
            case DT_AVIA:
                confirmedAviaOrders++;
                break;
            case DT_TRAIN:
                confirmedTrainOrders++;
                break;
            case DT_BUS:
                confirmedBusOrders++;
                break;
            case DT_SUBURBAN:
                confirmedSuburbanOrders++;
                break;
            default:
                throw new UnsupportedOperationException("unsupported type: " + type);
        }
    }

    public void orderRefunded(@NotNull EDisplayOrderType type) {
        int value;
        switch (type) {
            case DT_HOTEL:
                value = --confirmedHotelOrders;
                break;
            case DT_AVIA:
                value = --confirmedAviaOrders;
                break;
            case DT_TRAIN:
                value = --confirmedTrainOrders;
                break;
            case DT_BUS:
                value = --confirmedBusOrders;
                break;
            case DT_SUBURBAN:
                value = --confirmedSuburbanOrders;
                break;
            default:
                throw new UnsupportedOperationException("unsupported type: " + type);
        }
        if (value < 0) {
            log.warn("Order counter for {} type went below 0: {}. For user {}", type.name(), value, passportId);
        }
    }

    public Integer getConfirmedOrders(EDisplayOrderType type) {
        switch (type) {
            case DT_HOTEL:
                return confirmedHotelOrders;
            case DT_AVIA:
                return confirmedAviaOrders;
            case DT_TRAIN:
                return confirmedTrainOrders;
            case DT_BUS:
                return confirmedBusOrders;
            case DT_SUBURBAN:
                return confirmedSuburbanOrders;
            default:
                throw new UnsupportedOperationException("unsupported type: " + type);
        }
    }
}
