package ru.yandex.travel.orders.entities;

import java.time.Instant;
import java.util.UUID;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.Getter;
import lombok.Setter;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;

import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherState;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherType;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

/**
Сущность для асинхронной генерации файлов и их хранения.
Изначально генерировались только ваучеры, поэтому такое название, однако сейчас используется для разных типов.
 */

@Entity
@Table(name = "vouchers")
@Getter
@Setter
public class Voucher implements WorkflowEntity<EVoucherState> {
    @Id
    private UUID id;

    @OneToOne
    private Workflow workflow;

    private Instant lastTransitionAt;

    @Type(type = "proto-enum")
    private EVoucherState state;
    @Type(type = "proto-enum")
    private EVoucherType voucherType;
    @Type(type = "jsonb-object")
    private VoucherPayload payload;
    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    private UUID orderId;
    private UUID orderWorkflowId;
    private String orderPrettyId;

    @Version
    private Integer version;
    private String voucherName;
    private String voucherUrl;
    private Instant nextCheckCreatedAt;

    private boolean crashWorkflowOnError = true;

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.VOUCHER.getDiscriminatorValue();
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }

    public EVoucherType getVoucherType() {
        if (voucherType == null) {
            return EVoucherType.VT_HOTELS;
        }
        return voucherType;
    }

    public VoucherPayload getPayload() {
        if (payload == null) {
            payload = new VoucherPayload();
        }
        return payload;
    }

    private static Voucher createVoucherBase(Order order) {
        Voucher voucher = new Voucher();
        voucher.setState(EVoucherState.VS_NEW);
        voucher.setId(UUID.randomUUID());
        voucher.setOrderId(order.getId());
        voucher.setOrderPrettyId(order.getPrettyId());
        voucher.setOrderWorkflowId(order.getWorkflow().getId());
        voucher.setPayload(new VoucherPayload());
        return voucher;
    }

    public static Voucher createForOrder(Order order) {
        Voucher voucher = createVoucherBase(order);
        if (OrderCompatibilityUtils.isHotelOrder(order)) {
            voucher.setVoucherType(EVoucherType.VT_HOTELS);
            String fileName = "document_" + order.getPrettyId() + ".pdf";
            voucher.setVoucherName(fileName);
        } else if (OrderCompatibilityUtils.isBusOrder(order)) {
            voucher.setVoucherType(EVoucherType.VT_BUSES_TICKET);
            String fileName = "buses/ticket_" + order.getPrettyId() + ".pdf";
            voucher.setVoucherName(fileName);
        } else {
            throw new RuntimeException("Can't create voucher for order");
        }
        return voucher;
    }

    public static Voucher createDocument(EVoucherType documentType, String fileName, Order order) {
        Voucher voucher = createVoucherBase(order);
        voucher.setVoucherType(documentType);
        voucher.setVoucherName(fileName);
        return voucher;
    }
}
