package ru.yandex.travel.orders.entities;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.util.UUID;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.services.payments.PaymentProfile;
import ru.yandex.travel.orders.workflow.plus.proto.EYandexPlusTopupState;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

@Entity
@Table(name = "yandex_plus_topups")
@Getter
@Setter
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class YandexPlusTopup implements WorkflowEntity<EYandexPlusTopupState> {
    @Id
    private UUID id;

    @CreationTimestamp
    private Instant createdAt;
    @UpdateTimestamp
    private Instant updatedAt;
    @Version
    private Integer version;
    private Instant lastTransitionAt;

    @Type(type = "proto-enum")
    @Builder.Default
    private EYandexPlusTopupState state = EYandexPlusTopupState.PS_NEW;
    @OneToOne
    private Workflow workflow;
    private Instant nextCheckStatusAt;
    private boolean backgroundJobActive;

    private String externalOrderId;
    private BigDecimal totalAmountForPayload;
    private BigDecimal commissionAmountForPayload;
    private BigDecimal vatCommissionAmountForPayload;
    private boolean skipFinancialEvents;

    @ManyToOne
    private OrderItem orderItem;
    private Integer amount;
    @Type(type = "proto-currency-unit")
    private ProtoCurrencyUnit currency;
    private String passportId;

    @Type(type = "custom-enum")
    private PaymentProfile paymentProfile;
    private String userIp;

    private String trustPaymentId;
    private String purchaseToken;
    private Instant authorizedAt;
    private String authorizationErrorCode;
    private String authorizationErrorDesc;

    public String getEntityType() {
        return WellKnownWorkflowEntityType.PLUS_POINTS_TOPUP.getDiscriminatorValue();
    }

    public void rescheduleNextCheckStatusAt(Duration duration) {
        this.nextCheckStatusAt = Instant.now().plus(duration);
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }

    public ProtoCurrencyUnit getCurrency() {
        return currency != null ? currency : orderItem.getOrder().getCurrency();
    }

    public Money getAmountMoney() {
        return Money.of(amount, getCurrency());
    }

    public boolean isSuccessfullyCompleted() {
        if (skipFinancialEvents) {
            return state == EYandexPlusTopupState.PS_CLEARED;
        } else {
            return state == EYandexPlusTopupState.PS_FINANCIAL_EVENT_SENT;
        }
    }
}
