package ru.yandex.travel.orders.entities.context;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import lombok.Data;

import ru.yandex.travel.orders.commons.proto.ECancellationReason;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;

@Data
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonInclude(JsonInclude.Include.NON_NULL)
public class OrderStateContext {
    public static final List<EOrderItemState> COMPLETED_STATES = List.of(
            EOrderItemState.IS_CONFIRMED,
            EOrderItemState.IS_CANCELLED
    );

    private boolean moneyAcquired;
    private Boolean pendingCancellation;
    private boolean checkedOut;
    private boolean promoCodesApplied;
    private List<OrderItemContextState> orderItems;
    private Instant refundsUpdatedAt;
    private Instant trainRegistrationChangedAt;
    private Instant documentsUpdatedAt;
    private ECancellationReason cancellationReason;

    // train-specific states
    private SimpleMultiItemBatch updateTicketsTasks;
    private SimpleMultiItemBatch trainRegistrationChangeTasks;

    public void init(EDisplayOrderType displayType) {
        orderItems = new ArrayList<>();
        //noinspection SwitchStatementWithTooFewBranches
        switch (displayType) {
            case DT_TRAIN:
                updateTicketsTasks = new SimpleMultiItemBatch("updateTickets");
                trainRegistrationChangeTasks = new SimpleMultiItemBatch("trainRegistrationChange");
                break;
        }
    }
    private Boolean trainInsuranceAdded;
    private Boolean trainInsuranceAutoReturnPending;

    @JsonIgnore
    public boolean isTrainInsuranceAutoReturnPending() {
        return trainInsuranceAutoReturnPending != null && trainInsuranceAutoReturnPending;
    }

    public void serviceAdded(OrderItem service) {
        orderItems.add(new OrderItemContextState(service.getId(), service.getWorkflow().getId(),
                (EOrderItemState) service.getItemState()));
    }

    public void serviceReserved(UUID serviceId) {
        OrderItemContextState item = getItem(serviceId);
        item.setState(EOrderItemState.IS_RESERVED);
    }

    public void serviceCancelled(UUID serviceId) {
        OrderItemContextState item = getItem(serviceId);
        item.setState(EOrderItemState.IS_CANCELLED);
    }

    public void serviceConfirmed(UUID serviceId) {
        OrderItemContextState item = getItem(serviceId);
        item.setState(EOrderItemState.IS_CONFIRMED);
    }

    public OrderItemContextState getItem(UUID serviceId) {
        return orderItems.stream().filter(x -> x.getId().equals(serviceId)).findFirst()
                .orElseThrow(() -> new RuntimeException(String.format("Can not find item state by id %s", serviceId)));
    }

    @JsonIgnore
    public boolean allItemsReserved() {
        if (orderItems == null || orderItems.size() == 0) {
            return false;
        }
        return orderItems.stream().allMatch(x -> x.getState() == EOrderItemState.IS_RESERVED);
    }

    @JsonIgnore
    public boolean allItemsCancelled() {
        if (orderItems == null || orderItems.size() == 0) {
            return false;
        }
        return orderItems.stream().allMatch(x -> x.getState() == EOrderItemState.IS_CANCELLED);
    }

    @JsonIgnore
    public boolean allItemsConfirmed() {
        if (orderItems == null || orderItems.size() == 0) {
            return false;
        }
        return orderItems.stream().allMatch(x -> x.getState() == EOrderItemState.IS_CONFIRMED);
    }

    @JsonIgnore
    public boolean allItemsCompleted() {
        if (orderItems == null || orderItems.size() == 0) {
            return false;
        }
        return orderItems.stream().allMatch(x -> COMPLETED_STATES.contains(x.getState()));
    }

    @JsonIgnore
    public boolean anyServiceRefunding() {
        if (orderItems == null || orderItems.size() == 0) {
            return false;
        }
        return orderItems.stream().anyMatch(x -> x.getState() == EOrderItemState.IS_REFUNDING);
    }

    @JsonIgnore
    public boolean allItemsRefunded() {
        if (orderItems == null || orderItems.size() == 0) {
            return false;
        }
        return orderItems.stream().allMatch(x -> x.getState() == EOrderItemState.IS_REFUNDED);
    }
}
