package ru.yandex.travel.orders.entities.context;

import java.util.Map;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

public abstract class ProcessingBatch<TASK_ID, STATE> {
    @JsonProperty
    protected abstract String getName();

    @JsonProperty
    protected abstract Map<TASK_ID, STATE> getStates();

    @JsonIgnore
    public boolean isEmpty() {
        return getStates().isEmpty();
    }

    public void ensureEmpty() {
        Preconditions.checkArgument(isEmpty(), "Batch %s is not empty", getName());
    }

    public void addTask(TASK_ID taskId, STATE state) {
        Preconditions.checkArgument(!getStates().containsKey(taskId),
                "Batch %s already contains task %s", getName(), taskId);
        getStates().put(taskId, state);
    }

    public void changeTaskState(TASK_ID taskId, STATE oldState, STATE newState) {
        STATE current = getStates().get(taskId);
        Preconditions.checkState(getStates().containsKey(taskId) && Objects.equals(current, oldState),
                "Batch %s: task %s transition %s -> %s fails because of current state %s",
                getName(), taskId, oldState, newState, current);
        getStates().put(taskId, newState);
    }

    public boolean allTasksInState(STATE state) {
        return getStates().values().stream().allMatch(current -> Objects.equals(current, state));
    }

    public void clear() {
        Preconditions.checkArgument(!isEmpty(), "Batch %s is already empty", getName());
        getStates().clear();
    }
}
