package ru.yandex.travel.orders.entities.finances;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.persistence.Version;

import lombok.AccessLevel;
import lombok.Data;
import lombok.Getter;
import lombok.NoArgsConstructor;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;

@Entity
@Table(name = "bank_order_payments")
@Data
@NoArgsConstructor
public class BankOrderPayment {

    @Id
    private String paymentBatchId;

    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    @Version
    private Integer version;

    private Instant nextCheckAt;

    private Integer checkAttempt;

    @Type(type = "custom-enum")
    private BankOrderPaymentDetailsStatus status;

    /**
     * Расшифровка платежного поручения
     */
    @OneToMany(
            mappedBy = "bankOrderPayment",
            fetch = FetchType.LAZY,
            cascade = CascadeType.ALL
    )
    @BatchSize(size = 100)
    @Getter(AccessLevel.NONE)
    private List<BankOrderDetail> details;

    /**
     * Платежные поручения, связанные с paymentBatchId
     */
    @OneToMany(
            mappedBy = "bankOrderPayment",
            fetch = FetchType.LAZY,
            cascade = CascadeType.ALL
    )
    @BatchSize(size = 10)
    private List<BankOrder> orders;

    public static BankOrderPayment fromId(String paymentBatchId) {
        final BankOrderPayment result = new BankOrderPayment();
        result.setPaymentBatchId(paymentBatchId);
        return result;
    }

    public List<BankOrderDetail> getDetails() {
        if (details == null) {
            return Collections.emptyList();
        } else {
            return Collections.unmodifiableList(details);
        }
    }

    public int safeGetCheckAttempt() {
        if (checkAttempt == null) {
            checkAttempt = 0;
        }
        return checkAttempt;
    }

    public void addBankOrderDetail(BankOrderDetail bankOrderDetail) {
        if (details == null) {
            details = new ArrayList<>();
        }
        if (details.stream().map(BankOrderDetail::getYtId).anyMatch(ytId -> bankOrderDetail.getYtId().equals(ytId))) {
            //skip
            return;
        }
        bankOrderDetail.setBankOrderPayment(this);
        details.add(bankOrderDetail);
    }

    public void rescheduleNextCheckAtAndRegisterAttempt(Instant moment) {
        if (checkAttempt == null) {
            checkAttempt = 0;
        }
        checkAttempt = checkAttempt + 1;
        this.nextCheckAt = moment;
    }

    /**
     * Совпадает ли сумма в разбивке с суммой п/п. API биллинга только eventually consistent.
     * <p>
     * Статус FETCHED должен означать то же самое.
     */
    @Transient
    public boolean isConsistent() {
        BigDecimal sumInDetails = Objects.requireNonNullElse(getDetails(), Collections.<BankOrderDetail>emptyList())
                .stream()
                .map(bankOrderDetail -> {
                    if (bankOrderDetail.getTransactionType() == BillingTransactionType.REFUND) {
                        return bankOrderDetail.getSum().negate();
                    } else {
                        return bankOrderDetail.getSum();
                    }
                })
                .filter(Objects::nonNull)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal sumInOrders = Objects.requireNonNullElse(getOrders(), Collections.<BankOrder>emptyList())
                .stream()
                .map(BankOrder::getSum)
                .filter(Objects::nonNull)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        return sumInDetails.compareTo(sumInOrders) == 0;
    }
}
