package ru.yandex.travel.orders.entities.finances;

import java.time.Instant;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.SequenceGenerator;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.hibernate.annotations.Columns;
import org.hibernate.annotations.Type;
import org.javamoney.moneta.Money;

/**
 * Transaction's representation that gets created from {@link FinancialEvent} and gets exported to the Billing system.
 * <p>
 * Lifecycle:
 * <ol>
 *     <li>
 *         Gets created from {@link FinancialEvent}
 *     (see {@link ru.yandex.travel.orders.services.finances.billing.BillingTransactionGenerator BillingTransactionGenerator})
 *     </li>
 *     <li>When {@link #payoutAt payout time} comes, if not {@link #paused}, {@link #ytId} gets set</li>
 *     <li>
 *         Then {@link ru.yandex.travel.orders.services.finances.billing.BillingTransactionYtExporter BillingTransactionYtExporter}
 *         exports a bulk of transactions to YT and sets {@link #exportedToYt} to <code>true</code>
 *     </li>
 *     <li>
 *         When {@link #accountingActAt accounting time comes}
 *         {@link ru.yandex.travel.orders.services.finances.billing.BillingTransactionActCommitter BillingTransactionActCommitter}
 *         updates accountingAct in Billing and sets {@link #actCommitted} to true
 *     </li>
 * </ol>
 */
@Entity
@Table(name = "billing_transactions")
@Data
@Builder(toBuilder = true)
@NoArgsConstructor
@AllArgsConstructor(access = AccessLevel.PRIVATE)
public class BillingTransaction {
    @Id
    @GeneratedValue(
            strategy = GenerationType.SEQUENCE,
            generator = "billing_transactions_id_seq"
    )
    @SequenceGenerator(
            name = "billing_transactions_id_seq",
            sequenceName = "billing_transactions_id_seq",
            allocationSize = 1
    )
    private Long id;

    private BillingTransactionKind kind = BillingTransactionKind.PAYMENT;

    /**
     * @implNote gets set in {@link ru.yandex.travel.orders.services.finances.billing.BillingTransactionYtIdGenerator}
     */
    private Long ytId;

    private Long serviceId;

    @ManyToOne
    private FinancialEvent sourceFinancialEvent;

    @ManyToOne
    private BillingTransaction originalTransaction;

    @Type(type = "custom-enum")
    private BillingTransactionType transactionType;

    @Type(type = "custom-enum")
    private BillingTransactionPaymentType paymentType;

    @Type(type = "custom-enum")
    private BillingTransactionPaymentSystemType paymentSystemType;

    private Long partnerId;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "value_amount"), @Column(name = "value_currency")
    })
    private Money value;

    private String trustPaymentId;

    private Long clientId;

    private String serviceOrderId;

    private Instant createdAt;

    private Instant payoutAt;

    private Instant accountingActAt;

    @Version
    private Integer version;

    private boolean exportedToYt;

    private Instant exportedToYtAt;

    private boolean actCommitted;

    private Instant actCommittedAt;

    /**
     * Can pause process of assigning yt_id.
     */
    private boolean paused;

    public String getDescription() {
        return String.format("Billing transaction %s [serviceId=%s, type=%s, payment=%s, paysys=%s, " +
                        "partnerId=%s, value=%s, ytId=%s, payoutAt=%s, actAt=%s]",
                id != null ? id : "<new>", serviceId, transactionType, paymentType, paymentSystemType,
                partnerId, value, ytId, payoutAt, accountingActAt);
    }
}
