package ru.yandex.travel.orders.entities.finances;

import java.time.Instant;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.SequenceGenerator;
import javax.persistence.Table;
import javax.persistence.Version;

import com.google.common.base.Preconditions;
import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.ToString;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.Columns;
import org.hibernate.annotations.Type;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;

import static java.util.Arrays.stream;

/**
 * Financial event like payment/refund.
 * <p>
 * Holds info about all currencies/fee/promo/plus money. Gets created on payment/refund.
 * Depending on the type, either *_refund* fields get filled or their counterparts without "refund".
 * Doesn't necessary represent the real money. If payment is created for 1 ruble, fields
 * will be containing the whole sum.
 *
 * @see BillingTransaction
 * @see ru.yandex.travel.orders.services.finances.providers.FinancialDataProvider FinancialDataProvider
 */
@Entity
@Table(name = "financial_events")
@Data
@NoArgsConstructor
@AllArgsConstructor(access = AccessLevel.PRIVATE) // for builder
@Builder(toBuilder = true)
@EqualsAndHashCode(exclude = {"order", "orderItem"})
@ToString(exclude = {"order", "orderItem"})
@BatchSize(size = 100)
public class FinancialEvent {
    @Id
    @GeneratedValue(
            strategy = GenerationType.SEQUENCE,
            generator = "financial_events_id_seq"
    )
    @SequenceGenerator(
            name = "financial_events_id_seq",
            sequenceName = "financial_events_id_seq",
            allocationSize = 1
    )
    private Long id;

    @ManyToOne
    private Order order;

    private String orderPrettyId;

    @ManyToOne
    private OrderItem orderItem;

    private String trustPaymentId;

    @ManyToOne
    private FinancialEvent originalEvent;

    private Long billingClientId;

    @Nullable
    private Long billingContractId;

    @Type(type = "custom-enum")
    private FinancialEventType type;

    @Type(type = "custom-enum")
    private FinancialEventPaymentScheme paymentScheme;

    /**
     * De facto is the time of object creation. Not used anywhere else ATM.
     */
    private Instant accrualAt;

    /**
     * The date when the BillingTransactions created from this FinancialEvent will be exported to Billing.
     */
    private Instant payoutAt;

    private Instant accountingActAt;

    /**
     * Money paid to the partner.
     * <p>
     * If promocode was used, money from {@link #promoCodePartnerAmount} also go to the partner.
     */
    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "partner_amount"), @Column(name = "partner_currency")
    })
    private Money partnerAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "partner_refund_amount"), @Column(name = "partner_refund_currency")
    })
    private Money partnerRefundAmount;

    /**
     * our money.
     * <p>
     * See {@link #promoCodePartnerAmount} for a case when promo code is used.
     */
    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "fee_amount"), @Column(name = "fee_currency")
    })
    private Money feeAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "fee_refund_amount"), @Column(name = "fee_refund_currency")
    })
    private Money feeRefundAmount;

    /**
     * Money that we pay to the partner if the user used a promocode.
     * <p>
     * Creates a separate {@link BillingTransaction}. The source of money is Yandex's promo budget.
     *
     * @see ru.yandex.travel.orders.services.finances.providers.FullMoneySplitCalculator
     */
    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "promo_code_partner_amount"), @Column(name = "promo_code_partner_currency")
    })
    private Money promoCodePartnerAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "promo_code_fee_amount"), @Column(name = "promo_code_fee_currency")
    })
    private Money promoCodeFeeAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "promo_code_partner_refund_amount"), @Column(name = "promo_code_partner_refund_currency")
    })
    private Money promoCodePartnerRefundAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "promo_code_fee_refund_amount"), @Column(name = "promo_code_fee_refund_currency")
    })
    private Money promoCodeFeeRefundAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "partner_fee_amount"), @Column(name = "partner_fee_currency")
    })
    private Money partnerFeeAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "partner_fee_refund_amount"), @Column(name = "partner_fee_refund_currency")
    })
    private Money partnerFeeRefundAmount;

    /**
     * Money paid from plus points to the partner.
     *
     * @see #plusFeeAmount
     */
    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "plus_partner_amount"), @Column(name = "plus_partner_currency")
    })
    private Money plusPartnerAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "plus_partner_refund_amount"), @Column(name = "plus_partner_refund_currency")
    })
    private Money plusPartnerRefundAmount;

    /**
     * Money we "pay to ourselves" from plus points.
     *
     * @see #plusPartnerAmount
     */
    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "plus_fee_amount"), @Column(name = "plus_fee_currency")
    })
    private Money plusFeeAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "plus_fee_refund_amount"), @Column(name = "plus_fee_refund_currency")
    })
    private Money plusFeeRefundAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "plus_topup_amount"), @Column(name = "plus_topup_currency")
    })
    private Money plusTopupAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "plus_topup_refund_amount"), @Column(name = "plus_topup_refund_currency")
    })
    private Money plusTopupRefundAmount;

    /**
     * Money that user is paying to the partner in case of post pay.
     * We need it to calculate other stuff
     */
    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "post_pay_user_amount"), @Column(name = "post_pay_user_currency")
    })
    private Money postPayUserAmount;

    /**
     * Money that partner pays to us in case of post pay
     */
    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "post_pay_partner_payback"), @Column(name = "post_pay_partner_currency")
    })
    private Money postPayPartnerPayback;

    /**
     * Money that we "refund" to user (no actual money returned) in case of post pay
     */
    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "post_pay_user_refund_amount"), @Column(name = "post_pay_user_refund_currency")
    })
    private Money postPayUserRefund;

    /**
     * Money that we "refund" (actually we cancel the previous charge) to partner in case of post pay
     */
    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "post_pay_partner_refund_amount"), @Column(name = "post_pay_partner_refund_currency")
    })
    private Money postPayPartnerRefund;

    @Version
    private Integer version;

    private boolean processed;

    private String comment;

    private FinancialEvent init(OrderItem orderItem, FinancialEventType eventType, FinancialEventPaymentScheme paymentScheme, Instant accrualAt) {
        Preconditions.checkNotNull(paymentScheme, "paymentScheme can't be null");
        this.order = orderItem.getOrder();
        this.orderPrettyId = orderItem.getOrder().getPrettyId();
        this.orderItem = orderItem;
        this.type = eventType;
        this.paymentScheme = paymentScheme;
        this.accrualAt = accrualAt;
        return this;
    }

    private FinancialEvent initWithoutOrder(String orderPrettyId, FinancialEventType eventType, FinancialEventPaymentScheme paymentScheme, Instant accrualAt) {
        Preconditions.checkNotNull(paymentScheme, "paymentScheme can't be null");
        this.orderPrettyId = orderPrettyId;
        this.type = eventType;
        this.paymentScheme = paymentScheme;
        this.accrualAt = accrualAt;
        return this;
    }

    public static FinancialEvent create(OrderItem orderItem, FinancialEventType eventType, FinancialEventPaymentScheme paymentScheme, Instant accrualAt) {
        return new FinancialEvent().init(orderItem, eventType, paymentScheme, accrualAt);
    }

    public static FinancialEvent createWithoutOrder(String orderPrettyId, FinancialEventType eventType, FinancialEventPaymentScheme paymentScheme, Instant accrualAt) {
        return new FinancialEvent().initWithoutOrder(orderPrettyId, eventType, paymentScheme, accrualAt);
    }

    public boolean isPayment() {
        return type == FinancialEventType.PAYMENT ||
                type == FinancialEventType.MANUAL_PAYMENT ||
                type == FinancialEventType.INSURANCE_PAYMENT ||
                type == FinancialEventType.YANDEX_ACCOUNT_TOPUP_PAYMENT;
    }

    public boolean isRefund() {
        return type == FinancialEventType.REFUND ||
                type == FinancialEventType.MANUAL_REFUND ||
                type == FinancialEventType.INSURANCE_REFUND ||
                type == FinancialEventType.YANDEX_ACCOUNT_TOPUP_REFUND;
    }

    public boolean isYandexAccountTopup() {
        return type == FinancialEventType.YANDEX_ACCOUNT_TOPUP_PAYMENT
                || type == FinancialEventType.YANDEX_ACCOUNT_TOPUP_REFUND;
    }

    private Money[] getAllMoneyFields() {
        return new Money[]{
                partnerAmount,
                feeAmount,
                partnerRefundAmount,
                feeRefundAmount,
                promoCodePartnerAmount,
                promoCodeFeeAmount,
                promoCodePartnerRefundAmount,
                promoCodeFeeRefundAmount,
                partnerFeeAmount,
                partnerFeeRefundAmount,
                // plus fields
                plusPartnerAmount,
                plusFeeAmount,
                plusFeeRefundAmount,
                plusPartnerRefundAmount,
                plusTopupAmount,
                plusTopupRefundAmount,
                // post pay fields
                postPayUserAmount,
                postPayPartnerPayback,
                postPayUserRefund,
                postPayPartnerRefund
        };
    }

    public Money getTotalAmount() {
        return stream(getAllMoneyFields())
                .filter(Objects::nonNull)
                .reduce(Money::add)
                .orElse(null);
    }

    public String getDescription() {
        return String.format("Financial event %s [type=%s, value=%s, payoutAt=%s, actAt=%s]",
                id, type, getTotalAmount(), payoutAt, accountingActAt);
    }

    public void ensureNoMoneyValues() {
        if (!hasNoMoneyValues()) {
            throw new IllegalStateException("The event is expected to have no money values set: " + this);
        }
    }

    private boolean hasNoMoneyValues() {
        return stream(getAllMoneyFields())
                .noneMatch(Objects::nonNull);
    }

    public void ensureNoNegativeValues() {
        Preconditions.checkState(!hasNegativeValues(), "Negative values aren't expected; %s", this);
    }

    private boolean hasNegativeValues() {
        return stream(getAllMoneyFields())
                .anyMatch(v -> v != null && v.isNegative());
    }

    public void setOriginalEvent(FinancialEvent originalEvent) {
         /*
         when scheduled payment is used, there can be a situation when refund operation has payout date
         before the original transaction.
         The mechanism is the following: the original event has `payoutAt = lastPayment + 3 days`
         the refund event has `payoutAt = max(now, orderItem.confirmedAt)`
         */
        if (originalEvent != null && originalEvent.getPayoutAt() != null && this.getPayoutAt() != null &&
                originalEvent.getPayoutAt().isAfter(this.getPayoutAt())) {
            this.setPayoutAt(originalEvent.getPayoutAt());
        }
        this.originalEvent = originalEvent;
    }
}
