package ru.yandex.travel.orders.entities.mock.suburban;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.orders.commons.proto.TSuburbanTestContext;
import ru.yandex.travel.orders.commons.proto.TSuburbanTestContextHandlerError;
import ru.yandex.travel.orders.repository.mock.MockSuburbanOrderItemRepository;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientIOException;
import ru.yandex.travel.train.partners.im.ImClientParseException;
import ru.yandex.travel.train.partners.im.model.AutoReturnRequest;
import ru.yandex.travel.train.partners.im.model.AutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.ElectronicRegistrationRequest;
import ru.yandex.travel.train.partners.im.model.ElectronicRegistrationResponse;
import ru.yandex.travel.train.partners.im.model.OrderReservationBlankRequest;
import ru.yandex.travel.train.partners.im.model.OrderReservationTicketBarcodeRequest;
import ru.yandex.travel.train.partners.im.model.OrderReservationTicketBarcodeResponse;
import ru.yandex.travel.train.partners.im.model.ReservationConfirmResponse;
import ru.yandex.travel.train.partners.im.model.ReservationCreateRequest;
import ru.yandex.travel.train.partners.im.model.ReservationCreateResponse;
import ru.yandex.travel.train.partners.im.model.ReturnAmountRequest;
import ru.yandex.travel.train.partners.im.model.ReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.UpdateBlanksResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutRequest;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingRequest;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceReturnRequest;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceReturnResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemBlank;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;
import ru.yandex.travel.train.partners.im.model.orderlist.OrderListRequest;
import ru.yandex.travel.train.partners.im.model.orderlist.OrderListResponse;


@Slf4j
@RequiredArgsConstructor
public class MockImSuburbanClient implements ImClient {
    private final MockSuburbanOrderItem mockItem;
    private final SuburbanReservation suburbanReservation;
    private final TSuburbanTestContext testContext;
    private final TransactionTemplate transactionTemplate;
    private final MockSuburbanOrderItemRepository mockSuburbanOrderItemRepository;

    private final Duration BOOK_VALID_TIME = Duration.ofMinutes(30);
    private final Integer TICKET_NUMBER = 4242;
    private final String TICKET_BODY = "MDAwODg5NDc1OTQ0ODA1NjExMjI2NTE5ODEwMzAzMDc5NjMzICAgICBzQnQ2WWl3MDZzQUNSZ1J5WXJwLzdWRVg4Wkx1RnpCbGpuWE00clp4VGRjcDZ6UlFHbnhBVWpHVVRuS2RiMm8r";

    @Override
    public ReservationCreateResponse reservationCreate(ReservationCreateRequest request, Object data) {
        MockSuburbanOrderItem.ImOrderState state = getOrCreateImOrderState();

        possibleErrorScenario(testContext.getBookHandler(), state.getCreateHandler());
        if(state.getOrderId() != null) {
            throw new ImClientParseException("Order is already created for this test context. " +
                    "Something is wrong - Order/V1/Reservation/Create shouldn't be called twice.");
        }

        state.setOrderId(1234);
        state.setTravelDate(suburbanReservation.getImReservation().getDate());
        state.setFromExpressId(suburbanReservation.getImReservation().getStationFromExpressId());
        state.setToExpressId(suburbanReservation.getImReservation().getStationToExpressId());

        Duration validFor;
        if(testContext.getValidForSeconds() != -1) {
            validFor = Duration.ofSeconds(testContext.getValidForSeconds());
        } else {
            validFor = BOOK_VALID_TIME;
        }
        state.setValidDate(Instant.now().plus(validFor));

        if(testContext.getActualPrice() != -1) {
            state.setPrice(BigDecimal.valueOf(testContext.getActualPrice()));
        } else {
            state.setPrice(suburbanReservation.getPrice().getNumberStripped());
        }

        saveState();

        var response = new ReservationCreateResponse();
        response.setOrderId(state.getOrderId());
        response.setConfirmTill(instantToMoscow(state.getValidDate()));
        response.setAmount(state.getPrice());
        return response;
    }

    @Override
    public ReservationConfirmResponse reservationConfirm(int orderId) {
        MockSuburbanOrderItem.ImOrderState state = getOrCreateImOrderState();
        checkOrderId(state, orderId);
        possibleErrorScenario(testContext.getConfirmHandler(), state.getConfirmHandler());
        return new ReservationConfirmResponse();
    }

    @Override
    public CompletableFuture<OrderInfoResponse> orderInfoAsync(int orderId, Duration timeout) {
        MockSuburbanOrderItem.ImOrderState state = getOrCreateImOrderState();
        checkOrderId(state, orderId);
        possibleErrorScenario(testContext.getOrderInfoHandler(), state.getOrderInfoHandler());

        state.setTicketNumber(
                String.valueOf(testContext.getTicketNumber() == -1 ? TICKET_NUMBER : testContext.getTicketNumber()));
        state.setBlankOrderItemId(42420);
        state.setBlankId(424200);
        saveState();

        var blank = new OrderItemBlank();
        blank.setBlankNumber(String.valueOf(testContext.getTicketNumber()));
        blank.setOrderItemBlankId(state.getBlankId());

        var imOrderItem = new OrderItemResponse();
        List<OrderItemBlank> blanks = new ArrayList<>();

        imOrderItem.setOrderItemId(state.getBlankOrderItemId());
        imOrderItem.setOrderItemBlanks(blanks);
        blanks.add(blank);

        var orderItems = new ArrayList<OrderItemResponse>();
        orderItems.add(imOrderItem);

        var response = new OrderInfoResponse();
        response.setOrderItems(orderItems);
        return CompletableFuture.completedFuture(response);
    }

    @Override
    public CompletableFuture<OrderReservationTicketBarcodeResponse> orderReservationTicketBarcodeAsync(OrderReservationTicketBarcodeRequest request, Duration timeout) {
        MockSuburbanOrderItem.ImOrderState state = getOrCreateImOrderState();
        possibleErrorScenario(testContext.getTicketBarcodeHandler(), state.getTicketBarcodeHandler());

        if (!request.getOrderItemId().equals(state.getBlankOrderItemId())) {
            throw new ImClientParseException(String.format("Unknown orderItemId %s (this mock knows %s order item id)",
                    request.getOrderItemId(), state.getBlankOrderItemId()));
        }
        if (!request.getOrderItemBlankId().equals(state.getBlankId())) {
            throw new ImClientParseException(String.format("Unknown blankId %s (this mock knows %s blank id)",
                    request.getOrderItemBlankId(), state.getBlankId()));
        }

        state.setTicketBody(testContext.getTicketBody().equals("") ? TICKET_BODY : testContext.getTicketBody());
        saveState();

        var response = new OrderReservationTicketBarcodeResponse();
        response.setTicketBarcodeText(state.getTicketBody());
        return CompletableFuture.completedFuture(response);
    }

    @Override
    public CompletableFuture<byte[]> orderReservationBlankAsync(OrderReservationBlankRequest request,
                                                                Duration timeout) {
        MockSuburbanOrderItem.ImOrderState state = getOrCreateImOrderState();
        checkOrderId(state, request.getOrderId());
        possibleErrorScenario(testContext.getBlankPdfHandler(), state.getBlankHandler());

        byte[] response = String.format("Test pdf content. Ticket: %s; Date: %s, Station from: %s, Station to: %s",
            state.getTicketNumber(),
            state.getTravelDate(),
            state.getFromExpressId(),
            state.getToExpressId()
        ).getBytes();
        return CompletableFuture.completedFuture(response);
    }

    @Override
    public AutoReturnResponse autoReturn(AutoReturnRequest request) {
        throw new UnsupportedOperationException("autoReturn not implemented");
    }

    @Override
    public ElectronicRegistrationResponse changeElectronicRegistration(ElectronicRegistrationRequest request) {
        throw new UnsupportedOperationException("changeElectronicRegistration not implemented");
    }

    @Override
    public ReturnAmountResponse getReturnAmount(ReturnAmountRequest request) {
        throw new UnsupportedOperationException("insurancePricing not implemented");
    }

    @Override
    public InsuranceCheckoutResponse insuranceCheckout(InsuranceCheckoutRequest request) {
        throw new UnsupportedOperationException("insuranceCheckout not implemented");
    }

    @Override
    public InsurancePricingResponse insurancePricing(InsurancePricingRequest request) {
        throw new UnsupportedOperationException("insurancePricing not implemented");
    }

    @Override
    public InsuranceReturnResponse insuranceReturn(InsuranceReturnRequest request) {
        throw new UnsupportedOperationException("insuranceReturn not implemented");
    }

    @Override
    public OrderListResponse orderList(OrderListRequest request) {
        throw new UnsupportedOperationException("orderList not implemented");
    }

    @Override
    public void reservationCancel(int orderId) {
        throw new UnsupportedOperationException("reservationCancel not implemented");
    }

    @Override
    public UpdateBlanksResponse updateBlanks(int orderItemId, Duration timeout) {
        throw new UnsupportedOperationException("updateBlanks not implemented");
    }

    private void possibleErrorScenario(TSuburbanTestContextHandlerError handlerScenario,
                                       MockSuburbanOrderItem.HandlerState handlerState) {
        int errorsCount = handlerState.getErrorsCount();
        if(errorsCount < handlerScenario.getErrorsCount()) {
            handlerState.setErrorsCount(errorsCount + 1);
            String msg = String.format("Test context IM exception %s", handlerState.errorsCount);
            saveState();
            switch (handlerScenario.getErrorType()) {
                case ET_REQUEST_ERROR:
                case ET_PARSE_RESPONSE_ERROR:
                case ET_UNKNOWN:
                    throw new ImClientParseException(msg);
                case ET_RETRYABLE:
                    throw new ImClientIOException(msg);
            }
        }
    }

    private void checkOrderId(MockSuburbanOrderItem.ImOrderState imOrder, Integer orderId) {
        if (!imOrder.orderId.equals(orderId)) {
            throw new ImClientParseException(String.format("Unknown orderId %s (this mock knows %s order id)",
                    orderId, imOrder.orderId));
        }
    }

    private MockSuburbanOrderItem.ImOrderState getOrCreateImOrderState() {
        if (mockItem.getImOrder() == null) {
            var state = new MockSuburbanOrderItem.ImOrderState();
            mockItem.setImOrder(state);
        }
        saveState();
        return mockItem.getImOrder();
    }

    private void saveState() {
        withTx(null, (unused) -> {
            mockSuburbanOrderItemRepository.save(mockItem);
            return null;
        });
    }

    private <ReqT, RspT> RspT withTx(ReqT request, Function<ReqT, RspT> handler) {
        return transactionTemplate.execute((ignored) -> handler.apply(request));
    }

    private static LocalDateTime instantToMoscow(Instant instant) {
        return instant != null ? LocalDateTime.ofInstant(instant, ZoneId.of("Europe/Moscow")) : null;
    }
}
