package ru.yandex.travel.orders.entities.mock.suburban;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.function.Function;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.orders.commons.proto.TSuburbanTestContext;
import ru.yandex.travel.orders.commons.proto.TSuburbanTestContextHandlerError;
import ru.yandex.travel.orders.repository.mock.MockSuburbanOrderItemRepository;
import ru.yandex.travel.suburban.exceptions.SuburbanRetryableException;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.suburban.partners.movista.MovistaClient;
import ru.yandex.travel.suburban.partners.movista.MovistaErrorCode;
import ru.yandex.travel.suburban.partners.movista.MovistaModel;
import ru.yandex.travel.suburban.partners.movista.MovistaOrderStatus;
import ru.yandex.travel.suburban.partners.movista.exceptions.MovistaParseResponseException;
import ru.yandex.travel.suburban.partners.movista.exceptions.MovistaRequestException;
import ru.yandex.travel.suburban.partners.movista.exceptions.MovistaUnknownException;


@Slf4j
@RequiredArgsConstructor
public class MockMovistaClient implements MovistaClient {
    private final MockSuburbanOrderItem mockItem;
    private final SuburbanReservation suburbanReservation;
    private final TSuburbanTestContext testContext;
    private final TransactionTemplate transactionTemplate;
    private final MockSuburbanOrderItemRepository mockSuburbanOrderItemRepository;

    private final Duration BOOK_VALID_TIME = Duration.ofMinutes(30);
    private final Integer TICKET_NUMBER = 4242;
    private final String TICKET_BODY = "FkYADAAAAAAAAwAAdFWKYCNSEgC5BQDASOG937+CZ5WAvpwn+PsJWkY3h7OqHpWs4BaEz7ETppYdOlgsBHud/ZIkr9g2YfsrnAXy+2YZQJIY23VNmOduTw==";

    @Override
    public MovistaModel.OrderResponse book(MovistaModel.BookRequest request) {
        MockSuburbanOrderItem.MovistaOrderState state = getOrCreateMovistaOrderState();

        possibleErrorScenario(testContext.getBookHandler(), state.getBookHandler());
        if(state.getOrderId() != null) {
            throw new MovistaUnknownException("Order is already created for this test context. " +
                    "Something is wrong - book shouldn't be called twice.");
        }

        state.setOrderId(1234);
        state.setStatus(MovistaOrderStatus.CREATED);
        state.setTravelDate(suburbanReservation.getMovistaReservation().getDate());
        state.setFromExpressId(suburbanReservation.getMovistaReservation().getStationFromExpressId());
        state.setToExpressId(suburbanReservation.getMovistaReservation().getStationToExpressId());
        state.setFareId(suburbanReservation.getMovistaReservation().getFareId());

        Duration validFor;
        if(testContext.getValidForSeconds() != -1) {
            validFor = Duration.ofSeconds(testContext.getValidForSeconds());
        } else {
            validFor = BOOK_VALID_TIME;
        }
        state.setValidDate(Instant.now().plus(validFor));

        if(testContext.getActualPrice() != -1) {
            state.setPrice(BigDecimal.valueOf(testContext.getActualPrice()));
        } else {
            state.setPrice(suburbanReservation.getPrice().getNumberStripped());
        }

        saveState();
        return orderToResponse(state);
    }

    @Override
    public MovistaModel.OrderResponse confirm(MovistaModel.ConfirmRequest request) {
        MockSuburbanOrderItem.MovistaOrderState state = getOrCreateMovistaOrderState();
        checkOrderId(state, request.getOrderId());

        possibleErrorScenario(testContext.getConfirmHandler(), state.getConfirmHandler());
        checkOrderBookTimeValid(state);

        state.setStatus(MovistaOrderStatus.CONFIRMED);
        state.setTicketNumber(testContext.getTicketNumber() == -1 ? TICKET_NUMBER : testContext.getTicketNumber());
        state.setTicketBody(testContext.getTicketBody().equals("") ? TICKET_BODY : testContext.getTicketBody());
        state.setConfirmDate(Instant.now());
        saveState();
        return orderToResponse(state);
    }

    @Override
    public MovistaModel.OrderResponse orderInfo(MovistaModel.OrderInfoRequest request) {
        MockSuburbanOrderItem.MovistaOrderState state = getOrCreateMovistaOrderState();
        checkOrderId(state, request.getOrderId());
        possibleErrorScenario(testContext.getOrderInfoHandler(), state.getConfirmHandler());
        return orderToResponse(mockItem.getMovistaOrder());
    }

    @Override
    public byte[] getBlankPdf(MovistaModel.BlankPdfRequest request) {
        MockSuburbanOrderItem.MovistaOrderState state = getOrCreateMovistaOrderState();
        checkOrderId(state, request.getOrderId());

        possibleErrorScenario(testContext.getBlankPdfHandler(), state.getBlankPdfHandler());

        if (!state.status.equals(MovistaOrderStatus.CONFIRMED)) {
            // Экспериментально проверено, что Мовиста отдает 503, если билет не confirmed
            throw new SuburbanRetryableException(String.format("Order %s is not confirmed: status is %s",
                    state.getOrderId(), state.getStatus()));
        }
        return String.format("Test pdf content. Ticket: %s; Date: %s, Station from: %s, Station to: %s",
                state.getTicketNumber(),
                state.getTravelDate(),
                state.getFromExpressId(),
                state.getToExpressId()
        ).getBytes();
    }

    private void possibleErrorScenario(TSuburbanTestContextHandlerError handlerScenario,
                                       MockSuburbanOrderItem.HandlerState handlerState) {
        int errorsCount = handlerState.getErrorsCount();
        if(errorsCount < handlerScenario.getErrorsCount()) {
            handlerState.setErrorsCount(errorsCount + 1);
            String msg = String.format("Test context movista exception %s", handlerState.errorsCount);
            saveState();
            switch (handlerScenario.getErrorType()) {
                case ET_REQUEST_ERROR:
                    throw new MovistaRequestException(MovistaErrorCode.FARE_TSOPPD_NOT_FOUND, msg);
                case ET_PARSE_RESPONSE_ERROR:
                    throw new MovistaParseResponseException(msg);
                case ET_RETRYABLE:
                    throw new SuburbanRetryableException(msg);
                case ET_UNKNOWN:
                    throw new MovistaUnknownException(msg);
            }
        }
    }

    private void checkOrderId(MockSuburbanOrderItem.MovistaOrderState movistaOrder, Integer orderId) {
        if (!movistaOrder.orderId.equals(orderId)) {
            throw new MovistaUnknownException(String.format("Unknown orderId %s (this mock knows %s order id)",
                    orderId, movistaOrder.orderId));
        }
    }

    private void checkOrderBookTimeValid(MockSuburbanOrderItem.MovistaOrderState movistaOrder) {
        if(movistaOrder.validDate.isBefore(Instant.now())) {
            movistaOrder.setStatus(MovistaOrderStatus.CANCELLED);
            saveState();
            throw new MovistaRequestException(
                    MovistaErrorCode.ORDER_STATUS_INVALID,
                    String.format("Order %s is expired at %s",
                            movistaOrder.orderId, instantToMoscow(movistaOrder.validDate)));
        }
    }

    private MovistaModel.OrderResponse orderToResponse(MockSuburbanOrderItem.MovistaOrderState state) {
        return getDefaultOrderState().toBuilder()
                .orderId(state.orderId)
                .status(state.status)
                .travelDate(state.travelDate)
                .validDate(instantToMoscow(state.validDate))
                .fareId(state.fareId)
                .price(state.price)
                .fromExpressId(state.fromExpressId)
                .toExpressId(state.toExpressId)

                // available after confirm
                .ticketNumber(state.ticketNumber)
                .ticketBody(state.ticketBody)
                .confirmDate(instantToMoscow(state.confirmDate))
                .build();
    }

    private static MovistaModel.OrderResponse getDefaultOrderState() {
        return MovistaModel.OrderResponse.builder()
                .fromStationName("stationNameFrom4242")
                .toStationName("stationNameTo4343")
                .type(1)
                .farePlan("Пассажирский")
                .ticketType("Разовый полный")
                .instruction("MID2Tutorial")
                .provider("АО «ЦЕНТРАЛЬНАЯ ППК»")
                .inn("7705705370")
                .phone("+71111111111").build();
    }

    private MockSuburbanOrderItem.MovistaOrderState getOrCreateMovistaOrderState() {
        if (mockItem.getMovistaOrder() == null) {
            var state = new MockSuburbanOrderItem.MovistaOrderState();
            mockItem.setMovistaOrder(state);
        }
        saveState();
        return mockItem.getMovistaOrder();
    }

    private void saveState() {
        withTx(null, (unused) -> {
            mockSuburbanOrderItemRepository.save(mockItem);
            return null;
        });
    }

    private <ReqT, RspT> RspT withTx(ReqT request, Function<ReqT, RspT> handler) {
        return transactionTemplate.execute((ignored) -> handler.apply(request));
    }

    private static LocalDateTime instantToMoscow(Instant instant) {
        return instant != null ? LocalDateTime.ofInstant(instant, ZoneId.of("Europe/Moscow")) : null;
    }
}
