package ru.yandex.travel.orders.entities.notifications;

import java.time.Instant;
import java.util.ArrayList;
import java.util.UUID;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Version;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.ToString;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;

import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.entities.notifications.suburban.SuburbanCouponAttachmentProviderData;
import ru.yandex.travel.orders.workflow.notification.proto.EAttachmentState;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

@Entity
@Table(name = "attachments")
@Data
@EqualsAndHashCode(exclude = {"notification"})
@ToString(exclude = {"notification"})
public class Attachment implements WorkflowEntity<EAttachmentState> {
    @Id
    private UUID id;

    @OneToOne
    private Workflow workflow;

    @ManyToOne
    private Notification notification;

    private UUID notificationWorkflowId;

    @Type(type = "proto-enum")
    private EAttachmentState state;

    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    private Instant lastTransitionAt;

    private String filename;

    private String mimeType;

    @Column(name = "data_s3_id")
    private String dataS3Id;

    private boolean required;

    @Type(type = "custom-enum")
    private AttachmentProviderType provider;

    @Type(type = "jsonb-object")
    private BaseAttachmentProviderData providerData;

    @Version
    private Integer version;

    private static Attachment createAttachment(Notification notification, String filename, String mimeType,
                                               boolean required, AttachmentProviderType provider,
                                               BaseAttachmentProviderData providerData) {
        var attachment = new Attachment();
        attachment.setId(UUID.randomUUID());
        attachment.setState(EAttachmentState.AS_NEW);
        attachment.setNotification(notification);
        attachment.setNotificationWorkflowId(notification.getWorkflow().getId());
        attachment.setFilename(filename);
        attachment.setMimeType(mimeType);
        attachment.setRequired(required);
        attachment.setProvider(provider);
        attachment.setProviderData(providerData);

        if (notification.getAttachments() == null) {
            notification.setAttachments(new ArrayList<>());
        }
        notification.getAttachments().add(attachment);
        return attachment;
    }

    public static Attachment createImAttachment(Notification notification, String filename, String mimeType,
                                                boolean required, ImAttachmentProviderData providerData) {
        return createAttachment(notification, filename, mimeType, required, AttachmentProviderType.IM, providerData);
    }

    public static Attachment createUrlAttachment(Notification notification, String filename, String mimeType,
                                                 boolean required, String url) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(url), "Attachment url can't be empty");
        UrlAttachmentProviderData providerData = new UrlAttachmentProviderData();
        providerData.setUrl(url);
        return createAttachment(notification, filename, mimeType, required, AttachmentProviderType.URL, providerData);
    }

    public static Attachment createFiscalReceiptAttachment(Notification notification, String filename, String mimeType,
                                                           boolean required, long fiscalReceiptId) {
        FiscalReceiptAttachmentProviderData providerData = new FiscalReceiptAttachmentProviderData();
        providerData.setFiscalReceiptId(fiscalReceiptId);
        return createAttachment(notification, filename, mimeType, required, AttachmentProviderType.FISCAL_RECEIPT,
                providerData);
    }

    public static Attachment createReadyAttachment(Notification notification, String filename, String mimeType,
                                                   AttachmentProviderType provider) {
        Attachment result = createAttachment(notification, filename, mimeType, true, provider, null);
        result.setState(EAttachmentState.AS_FETCHED);
        return result;
    }

    public static Attachment createSuburbanCouponAttachmentData(Notification notification, String filename,
                                                                String mimeType,
                                                                boolean required,
                                                                SuburbanCouponAttachmentProviderData providerData) {
        return createAttachment(notification, filename, mimeType, required, AttachmentProviderType.SUBURBAN_COUPON, providerData);
    }

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.ATTACHMENT.getDiscriminatorValue();
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }
}
