package ru.yandex.travel.orders.entities.notifications;

import java.time.Instant;
import java.util.List;
import java.util.UUID;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;

import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.workflow.notification.proto.ENotificationState;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

@Entity
@Table(name = "notifications")
@Data
@NoArgsConstructor
@AllArgsConstructor
@Builder
public class Notification implements WorkflowEntity<ENotificationState> {
    @Id
    private UUID id;

    @OneToOne
    private Workflow workflow;

    private Instant lastTransitionAt;

    @Type(type = "proto-enum")
    private ENotificationState state;
    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    private UUID orderId;
    private UUID orderWorkflowId;
    private String orderPrettyId;

    @OneToMany(mappedBy = "notification", cascade = CascadeType.ALL)
    @BatchSize(size = 10)
    private List<Attachment> attachments;

    @Type(type = "custom-enum")
    private NotificationChannelType channel;

    @Type(type = "jsonb-object")
    private BaseChannelInfo channelInfo;

    @Deprecated
    private boolean sendCompleteMessage;
    private boolean expired;

    private Instant preparingAttachmentsTill;

    @Version
    private Integer version;

    private static Notification createNotification(Order order, NotificationChannelType channel,
                                                   BaseChannelInfo channelInfo) {
        var notification = new Notification();
        notification.setState(ENotificationState.NS_NEW);
        notification.setId(UUID.randomUUID());
        if (order != null) {
            notification.setOrderId(order.getId());
            notification.setOrderWorkflowId(order.getWorkflow().getId());
            notification.setOrderPrettyId(order.getPrettyId());
        }
        notification.setChannel(channel);
        notification.setChannelInfo(channelInfo);
        return notification;
    }

    public static Notification createEmailNotification(EmailChannelInfo channelInfo) {
        return createNotification(null, NotificationChannelType.EMAIL, channelInfo);
    }

    public static Notification createEmailNotification(Order order, EmailChannelInfo channelInfo) {
        return createNotification(order, NotificationChannelType.EMAIL, channelInfo);
    }

    public static Notification createTemplatedEmailNotification(Order order, TemplatedEmailChannelInfo channelInfo) {
        return createNotification(order, NotificationChannelType.TEMPLATED_EMAIL, channelInfo);
    }

    public static Notification createSmsNotification(Order order, SmsChannelInfo channelInfo) {
        return createNotification(order, NotificationChannelType.SMS, channelInfo);
    }

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.NOTIFICATION.getDiscriminatorValue();
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }
}
