package ru.yandex.travel.orders.entities.partners;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.Data;

import ru.yandex.travel.hotels.administrator.export.proto.HotelAgreement;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.entities.VatType;

@Data
public class DirectHotelBillingPartnerAgreement implements BillingPartnerAgreement {

    private Long id;

    private String hotelId;

    private String inn;

    private VatType vatType;

    private Instant agreementStartDate;

    private Long financialClientId;

    private Long financialContractId;

    private BigDecimal orderConfirmedRate;

    private BigDecimal orderRefundedRate;

    private Instant agreementEndDate;

    private Integer priority;

    public static DirectHotelBillingPartnerAgreement fromProto(HotelAgreement hotelAgreement) {
        DirectHotelBillingPartnerAgreement agreement = new DirectHotelBillingPartnerAgreement();
        agreement.setId(hotelAgreement.getId());
        agreement.setHotelId(hotelAgreement.getHotelId());
        agreement.setInn(hotelAgreement.getInn());
        agreement.setVatType(VatType.fromEVat(hotelAgreement.getVatType()));
        agreement.setFinancialClientId(hotelAgreement.getFinancialClientId());
        agreement.setFinancialContractId(hotelAgreement.getFinancialContractId());
        agreement.setAgreementStartDate(Instant.ofEpochMilli(hotelAgreement.getAgreementStartDate()));
        if (hotelAgreement.getAgreementEndDate() > 0) { //instead of null there will be 0 here
            agreement.setAgreementEndDate(Instant.ofEpochMilli(hotelAgreement.getAgreementEndDate()));
        }
        agreement.setOrderConfirmedRate(new BigDecimal(hotelAgreement.getOrderConfirmedRate()));
        agreement.setOrderRefundedRate(new BigDecimal(hotelAgreement.getOrderRefundedRate()));
        if (hotelAgreement.hasPriority()) {
            agreement.setPriority(hotelAgreement.getPriority());
        }
        return agreement;
    }

    public HotelAgreement toProto(EPartnerId partnerId) {
        var agreement = HotelAgreement.newBuilder();
        agreement.setId(this.getId());
        agreement.setHotelId(this.getHotelId());
        agreement.setInn(this.getInn());
        Optional.ofNullable(this.getVatType()).map(VatType::getProtoValue).ifPresent(agreement::setVatType);
        agreement.setFinancialClientId(this.getFinancialClientId());
        agreement.setFinancialContractId(this.getFinancialContractId());
        agreement.setAgreementStartDate(this.getAgreementStartDate().toEpochMilli());
        if (this.getAgreementEndDate() != null) {
            agreement.setAgreementEndDate(this.getAgreementEndDate().toEpochMilli());
        }
        if (this.getOrderConfirmedRate() != null) {
            agreement.setOrderConfirmedRate(this.getOrderConfirmedRate().toString());
        }
        if (this.getOrderRefundedRate() != null) {
            agreement.setOrderRefundedRate(this.getOrderRefundedRate().toString());
        }
        if (this.getPriority() != null) {
            agreement.setPriority(this.getPriority());
        }
        agreement.setPartnerId(partnerId);
        return agreement.build();
    }

    @JsonIgnore
    @Override
    public Long getBillingClientId() {
        return financialClientId;
    }
}
