package ru.yandex.travel.orders.entities.promo;

import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.persistence.Transient;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import lombok.Data;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;

@Data
@JsonIgnoreProperties(ignoreUnknown = true)
public class DiscountApplicationConfig {
    private Money minTotalCost;
    private UserTypeRestriction userTypeRestriction;
    private boolean addsUpWithOtherActions;
    /**
     * @deprecated use {@link #firstUsageLimits}. The getters/setters remain solely for existing
     * objects in DB and existing external API.
     */
    @Deprecated
    private Integer maxConfirmedHotelOrders;
    /**
     * When values are set to 1 - limits the promo action applicability to the 1st order only for a user.
     * <p>
     * This is the only anticipated usage so far, if values >1 are expected, some API methods should be revised.
     */
    private Map<EDisplayOrderType, Integer> firstUsageLimits;

    private List<HotelRestriction> hotelRestrictions;

    /**
     * For promos with percent nominal type - the maximum promo code discount.
     */
    private Money maxNominalDiscount;

    public Integer getMaxConfirmedHotelOrders() {
        if (firstUsageLimits != null) {
            return firstUsageLimits.getOrDefault(EDisplayOrderType.DT_HOTEL, maxConfirmedHotelOrders);
        }
        return maxConfirmedHotelOrders;
    }

    public void setMaxConfirmedHotelOrders(Integer maxConfirmedHotelOrders) {
        if (firstUsageLimits == null) {
            firstUsageLimits = new EnumMap<>(EDisplayOrderType.class);
        }
        firstUsageLimits.put(EDisplayOrderType.DT_HOTEL, maxConfirmedHotelOrders);
    }

    public boolean limitedForFirstOrderOnlyFor(EDisplayOrderType type) {
        if (firstUsageLimits == null) {
            return false;
        }
        return Objects.equals(1, firstUsageLimits.get(type));
    }

    /**
     * As it's the only anticipated usage so far, Set is more convenient to work with and share than a Map.
     */
    @Transient
    @JsonIgnore
    public Set<EDisplayOrderType> getFirstOrderOnlyFor() {
        if (firstUsageLimits == null) {
            return null;
        }
        return firstUsageLimits.entrySet()
                .stream()
                .filter(entry -> Objects.equals(1, entry.getValue()))
                .map(Map.Entry::getKey)
                .collect(Collectors.toSet());
    }

    public void setFirstOrderOnlyFor(Set<EDisplayOrderType> type) {
        if (type == null) {
            firstUsageLimits = null;
        } else {
            firstUsageLimits = type.stream()
                    .collect(Collectors.toMap(key -> key, key -> 1));
        }
    }

    @JsonIgnore
    public boolean hasHotelRestrictions() {
        return hotelRestrictions != null && !hotelRestrictions.isEmpty();
    }
}
