package ru.yandex.travel.orders.entities.promo;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.Objects;
import java.util.UUID;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.AccessLevel;
import lombok.Data;
import lombok.Getter;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;

import ru.yandex.travel.orders.commons.proto.EPromoCodeNominalType;

@Entity
@Table(name = "promo_codes")
@Data
@BatchSize(size = 100)
public class PromoCode {

    @Id
    private UUID id;

    @ManyToOne
    private PromoAction promoAction;

    @Column(unique = true)
    private String code;

    private BigDecimal nominal;

    @Type(type = "proto-enum")
    private EPromoCodeNominalType nominalType;

    @CreationTimestamp
    private Instant createdAt;

    @Version
    private Long version;

    /**
     * Means activations PER USER. If set to 1, multiple users can use it but only once. Gets set on promocode creation.
     */
    @Getter(AccessLevel.NONE)
    private Integer allowedUsageCount;

    /**
     * Works with {@link #allowedActivationsTotal}. Doesn't affect {@link #allowedUsageCount}.
     */
    @Getter(AccessLevel.NONE)
    @Type(type = "custom-enum")
    private PromoCodeActivationsStrategy activationsStrategy;

    /**
     * Number of users that can use the promo code in total. If set to 1, only one user can use it
     * for {@link #allowedUsageCount} times.
     * <p><pre>
     * In the following scenario:
     * allowedUsageCount = 3
     * activationsStrategy = LIMITED_ACTIVATIONS
     * allowedActivationsTotal = 4
     * 4 users can use it 3 times each. 12 usages in total max.
     * </pre>
     */
    @Getter(AccessLevel.NONE)
    private Integer allowedActivationsTotal;

    /**
     * Number of unique users who have used the code.
     * Corresponds to number of linked {@link PromoCodeActivation} entities.
     * <br>
     * Should be compared with {@link #allowedActivationsTotal}.
     */
    private Integer allowedActivationsCount;

    private Instant validFrom;
    private Instant validTill;

    @Type(type = "custom-enum")
    private PromoCodeBehaviourOverride behaviourOverride;

    @ManyToOne
    @JoinColumn(name = "order_generated_id")
    private OrderGeneratedPromoCodes orderGenerated;

    private boolean blacklisted;

    public Integer getAllowedUsageCount() {
        return Objects.requireNonNullElse(allowedUsageCount, 1);
    }

    public PromoCodeActivationsStrategy getActivationsStrategy() {
        return Objects.requireNonNullElse(this.activationsStrategy, PromoCodeActivationsStrategy.UNLIMITED_ACTIVATIONS);
    }

    public void setAllowedActivationsTotal(Integer allowedActivationsTotal) {
        if (this.allowedActivationsTotal == null &&
                getActivationsStrategy() == PromoCodeActivationsStrategy.LIMITED_ACTIVATIONS) {
            this.allowedActivationsTotal = allowedActivationsTotal;
        } else {
            throw new UnsupportedOperationException("Trying to set readOnly field allowedActivationsTotal");
        }
    }

    public Integer getAllowedActivationsTotal() {
        return Objects.requireNonNullElse(this.allowedActivationsTotal, Integer.MAX_VALUE);
    }

    public Integer getAllowedActivationsCount() {
        return Objects.requireNonNullElse(this.allowedActivationsCount, 0);
    }

    public void incrementAllowedActivationsCount() {
        if (allowedActivationsCount == null) {
            allowedActivationsCount = 1;
        } else {
            allowedActivationsCount = allowedActivationsCount + 1;
        }
    }

    public void decrementAllowedActivationsCount() {
        if (allowedActivationsCount == null) {
            allowedActivationsCount = 0;
        } else {
            allowedActivationsCount = allowedActivationsCount - 1;
        }
    }
}
