package ru.yandex.travel.orders.entities.promo;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.AccessLevel;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.Columns;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.commons.proto.EPromoCodeApplicationResultType;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.Order;

/**
 * Связывает заказ и {@link PromoCodeActivation}. Т.е. именно эта сущность отражает конкретное применение промокода.
 *
 * @see PromoCodeActivation
 */
@Entity
@Table(name = "promo_code_applications")
@Data
@EqualsAndHashCode(exclude = {"fiscalItemDiscounts"})
@ToString(exclude = {"fiscalItemDiscounts"})
public class PromoCodeApplication {

    @Id
    private UUID id;

    @ManyToOne
    private PromoCodeActivation promoCodeActivation;

    @ManyToOne
    private Order order;

    private Instant appliedAt;

    @CreationTimestamp
    private Instant createdAt;

    @Version
    private Long version;

    @Getter(AccessLevel.NONE)
    @Setter(AccessLevel.NONE)
    @OneToMany(mappedBy = "promoCodeApplication", cascade = CascadeType.ALL)
    @BatchSize(size = 100)
    private List<FiscalItemDiscount> fiscalItemDiscounts;

    @Type(type = "proto-enum")
    private EPromoCodeApplicationResultType applicationResultType;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "discount_amount"), @Column(name = "discount_currency")
    })
    private Money discount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "discount_remainder_amount"), @Column(name = "discount_remainder_currency")
    })
    private Money discountRemainder;

    public List<FiscalItemDiscount> getFiscalItemDiscounts() {
        return fiscalItemDiscounts != null ?
                Collections.unmodifiableList(fiscalItemDiscounts) :
                Collections.emptyList();
    }

    public void setDiscountIfNull(Map<FiscalItem, Money> discountMap) {
        Money totalAmount = discountMap.values().stream().reduce(Money.zero(order.getCurrency()), Money::add);
        if (discount != null) {
            if (!discount.isEqualTo(totalAmount)) {
                throw new IllegalArgumentException("mismatched values for discount amount! " +
                        discount.getNumber() + "!=" + totalAmount.getNumber());
            }
            // discount is already set, skipping
            return;
        }
        fiscalItemDiscounts = new ArrayList<>();
        for (var entry : discountMap.entrySet()) {
            FiscalItemDiscount itemDiscount = new FiscalItemDiscount();
            itemDiscount.setPromoCodeApplication(this);
            itemDiscount.setFiscalItem(entry.getKey());
            itemDiscount.setDiscount(entry.getValue());
            fiscalItemDiscounts.add(itemDiscount);
        }


        discount = totalAmount;
        discountRemainder = totalAmount;
    }

    public boolean hasBeenActivated() {
        return applicationResultType == EPromoCodeApplicationResultType.ART_SUCCESS &&
                appliedAt != null;
    }
}
