package ru.yandex.travel.orders.grpc;

import java.util.UUID;
import java.util.function.BiFunction;

import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.proto.AuthorizationInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TAuthorizeForOrderReq;
import ru.yandex.travel.orders.proto.TAuthorizeForOrderRsp;
import ru.yandex.travel.orders.services.AuthorizationService;

@GrpcService(authenticateUser = true, authenticateService = true)
@Slf4j
@RequiredArgsConstructor
public class AuthorizationGrpcService extends AuthorizationInterfaceV1Grpc.AuthorizationInterfaceV1ImplBase {
    private final TransactionTemplate transactionTemplate;
    private final AuthorizationService authorizationService;

    @Override
    public void checkAuthorization(TAuthorizeForOrderReq request, StreamObserver<TAuthorizeForOrderRsp> observer) {
        synchronouslyWithTx(request, observer, (orderId, uc) -> {
                    log.info("Checking access to order '{}' for user '{}'", orderId, uc);
                    boolean res = checkAuthImpl(uc, orderId);
                    return TAuthorizeForOrderRsp.newBuilder()
                            .setAuthorized(res).build();
                }
        );
    }

    private boolean checkAuthImpl(UserCredentials credentials, UUID orderId) {
        return authorizationService.checkAuthorizationFor(
                orderId,
                credentials.getSessionKey(),
                credentials.getPassportId());
    }

    @Override
    public void authorize(TAuthorizeForOrderReq request, StreamObserver<TAuthorizeForOrderRsp> observer) {
        synchronouslyWithTx(request, observer, (orderId, credentials) -> {
                    if (checkAuthImpl(credentials, orderId)) {
                        log.info("Access to order '{}' for user '{}' was already granted", orderId, credentials);
                    } else {
                        log.info("Granting access to order '{}' for user '{}'", orderId, credentials);
                        authorizationService.authorizeForOrder(orderId, credentials,
                                AuthorizedUser.OrderUserRole.VIEWER);
                    }
                    return TAuthorizeForOrderRsp.newBuilder().setAuthorized(true).build();
                }
        );
    }

    @SuppressWarnings("Duplicates")
    private UserCredentials getCredentials() {
        UserCredentials credentials = UserCredentials.get();
        if (credentials == null) {
            throw Error.with(EErrorCode.EC_PERMISSION_DENIED, "credentials are not enabled").toEx();
        }
        return credentials;
    }

    private <RspT> void synchronouslyWithTx(TAuthorizeForOrderReq request, StreamObserver<RspT> observer,
                                            BiFunction<UUID, UserCredentials, RspT> handler) {
        ServerUtils.synchronously(log, request, observer,
                rq -> {
                    UserCredentials credentials = getCredentials();
                    UUID orderId = ProtoChecks.checkStringIsUuid("order_id", request.getOrderId());
                    try (var ignored = NestedMdc.forEntity(orderId, null)) {
                        return transactionTemplate.execute(notUsed -> handler.apply(orderId, credentials));
                    }
                },
                ex -> GrpcExceptionHelper.mapStatusException(log, request, ex)
        );
    }
}
