package ru.yandex.travel.orders.grpc;

import java.time.Duration;
import java.time.LocalDate;
import java.time.format.DateTimeParseException;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.protobuf.Timestamp;
import io.grpc.stub.StreamObserver;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotCarrier;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotSegment;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotSegmentNode;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotVariant;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TDate;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.basic_info.v1.AviaFlightTitle;
import ru.yandex.travel.orders.basic_info.v1.AviaOrderItem;
import ru.yandex.travel.orders.basic_info.v1.AviaOriginDestination;
import ru.yandex.travel.orders.basic_info.v1.AviaSegment;
import ru.yandex.travel.orders.basic_info.v1.BasicOrderInfo;
import ru.yandex.travel.orders.basic_info.v1.BasicOrderInfoAPIGrpc;
import ru.yandex.travel.orders.basic_info.v1.BusOrderItem;
import ru.yandex.travel.orders.basic_info.v1.GeoRegion;
import ru.yandex.travel.orders.basic_info.v1.GetBasicOrderRequest;
import ru.yandex.travel.orders.basic_info.v1.GetBasicOrderResponse;
import ru.yandex.travel.orders.basic_info.v1.HotelOrderItem;
import ru.yandex.travel.orders.basic_info.v1.TrainOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.services.EDisplayOrderStateMapper;
import ru.yandex.travel.orders.services.OrderInfoMapper;
import ru.yandex.travel.train.model.TrainReservation;

import static com.google.common.base.Strings.nullToEmpty;

@GrpcService(authenticateService = true)
@Slf4j
@AllArgsConstructor
public class BasicOrderInfoService extends BasicOrderInfoAPIGrpc.BasicOrderInfoAPIImplBase {
    private final AuthorizationService authService;
    private final OrderRepository orderRepository;
    private final TransactionTemplate transactionTemplate;
    private final OrderInfoMapper orderInfoMapper;

    @Override
    public void getOrder(GetBasicOrderRequest request, StreamObserver<GetBasicOrderResponse> responseObserver) {
        synchronouslyWithTx(request, responseObserver, this::getOrder);
    }

    private <ReqT, RspT> void synchronouslyWithTx(ReqT request, StreamObserver<RspT> observer,
                                                  Function<ReqT, RspT> handler) {
        ServerUtils.synchronously(log, request, observer,
                rq -> transactionTemplate.execute((ignored) -> handler.apply(rq)),
                ex -> GrpcExceptionHelper.mapStatusException(log, request, ex)
        );
    }

    private GetBasicOrderResponse getOrder(GetBasicOrderRequest req) {
        Optional<Order> optOrder = Optional.empty();

        switch (req.getOneOfOrderIdsCase()) {
            case ORDER_ID:
                UUID orderId = ProtoChecks.checkStringIsUuid("Order id", req.getOrderId());
                optOrder = orderRepository.findById(orderId);
                break;
            case PRETTY_ID:
                optOrder = orderRepository.findOrderByPrettyId(req.getPrettyId());
                break;
            case ONEOFORDERIDS_NOT_SET:
                Error.with(EErrorCode.EC_INVALID_ARGUMENT, "One of order Ids must be set").andThrow();
        }
        if (optOrder.isEmpty()) {
            throw Error.with(EErrorCode.EC_NOT_FOUND,
                    "Order with IDs (" + req.getOrderId() + ", " + req.getPrettyId() + ") not found").toEx();
        }
        GetBasicOrderResponse.Builder responseBuilder = GetBasicOrderResponse.newBuilder();
        Order order = optOrder.get();

        BasicOrderInfo.Builder orderInfo = BasicOrderInfo.newBuilder()
                .setId(ProtoUtils.toStringOrEmpty(order.getId()))
                .setOwner(orderInfoMapper.buildUserInfo(order, authService.getOrderOwner(order.getId())))
                .setType(order.getDisplayType())
                .setState(EDisplayOrderStateMapper.fromOrder(order));


        addOrderItems(order, orderInfo);

        responseBuilder.setOrderInfo(orderInfo);

        return responseBuilder.build();
    }

    private void addOrderItems(Order order, BasicOrderInfo.Builder builder) {
        for (OrderItem orderItem : order.getOrderItems()) {
            if (orderItem instanceof ru.yandex.travel.orders.entities.AeroflotOrderItem) {
                builder.addAviaOrderItems(mapAviaOrderItem((ru.yandex.travel.orders.entities.AeroflotOrderItem) orderItem));
            } else if (orderItem instanceof ru.yandex.travel.orders.entities.BusOrderItem) {
                builder.addBusOrderItems(mapBusOrderItem((ru.yandex.travel.orders.entities.BusOrderItem) orderItem));
            } else if (orderItem instanceof ru.yandex.travel.orders.entities.HotelOrderItem) {
                builder.addHotelOrderItems(mapHotelOrderItem((ru.yandex.travel.orders.entities.HotelOrderItem) orderItem));
            } else if (orderItem instanceof ru.yandex.travel.orders.entities.TrainOrderItem) {
                builder.addTrainOrderItems(mapTrainOrderItem((ru.yandex.travel.orders.entities.TrainOrderItem) orderItem));
            } else {
                log.warn("Unknown order item type {}", orderItem.getClass().toString());
            }
        }
    }


    private AviaOrderItem mapAviaOrderItem(ru.yandex.travel.orders.entities.AeroflotOrderItem orderItem) {
        AeroflotVariant variant = orderItem.getPayload().getVariant();
        return AviaOrderItem.newBuilder()
                .addAllOriginDestinations(
                        mapOriginDestinationsForVariant(variant)
                )
                .build();

    }

    private List<AviaOriginDestination> mapOriginDestinationsForVariant(AeroflotVariant variant) {
        return variant.getOriginDestinations().stream().map(
                od -> AviaOriginDestination.newBuilder()
                        .setDepartureStation(nullToEmpty(od.getDepartureCode()))
                        .setArrivalStation(nullToEmpty(od.getArrivalCode()))
                        .addAllSegments(mapAviaSegments(variant.getOriginDestinationSegments(od.getId())))
                        .build()
        ).collect(Collectors.toList());
    }

    private List<AviaSegment> mapAviaSegments(List<AeroflotSegment> segments) {
        return segments.stream().map(
                segment -> {
                    AviaSegment.Builder proto = AviaSegment.newBuilder()
                            .setMarketingTitle(mapAviaCarrier(segment.getMarketingCarrier()))
                            .setOperatingTitle(mapAviaCarrier(segment.getOperatingCarrier()))
                            .setFlightDurationMinutes((int) mapAviaSegmentFlightDuration(segment.getFlightDuration()));
                    fillAviaSegmentNode(segment.getDeparture(),
                            proto::setDepartureStation,
                            proto::setDepartureDate,
                            proto::setDepartureDatetime);
                    fillAviaSegmentNode(segment.getArrival(),
                            proto::setArrivalStation, proto::setArrivalDate,
                            proto::setArrivalDatetime);
                    return proto.build();
                }
        ).collect(Collectors.toList());
    }

    private long mapAviaSegmentFlightDuration(String s_duration) {
        if (s_duration == null) {
            return 0;
        }
        try {
            return Duration.parse(s_duration).toMinutes();
        } catch (DateTimeParseException e) {
            log.warn("Unexpected duration format: {}", s_duration);
        }
        return 0;
    }

    private void fillAviaSegmentNode(
            AeroflotSegmentNode node,
            Consumer<String> airportCode,
            Consumer<TDate> localDate,
            Consumer<Timestamp> localDateTime) {
        String s_date = node.getDate();
        airportCode.accept(nullToEmpty(node.getAirportCode()));
        localDate.accept(ProtoUtils.toTDate(LocalDate.parse(s_date)));
        localDateTime.accept(ProtoUtils.fromLocalDateTime(node.getDateTime()));
    }


    private AviaFlightTitle mapAviaCarrier(AeroflotCarrier carrier) {
        return AviaFlightTitle.newBuilder()
                .setAirlineId(nullToEmpty(carrier.getAirlineId()))
                .setFlightNumber(nullToEmpty(carrier.getFlightNumber()))
                .build();
    }

    private BusOrderItem mapBusOrderItem(ru.yandex.travel.orders.entities.BusOrderItem orderItem) {
        BusReservation payload = orderItem.getPayload();
        return BusOrderItem.newBuilder()
                .setRideId(nullToEmpty(payload.getRide().getRideId()))
                .setDepartureStation(nullToEmpty(payload.getRide().getPointFrom().getPointKey()))
                .setArrivalStation(nullToEmpty(payload.getRide().getPointTo().getPointKey()))
                .setDepartureDate(ProtoUtils.toTDate(payload.getRide().getLocalDepartureTime().toLocalDate()))
                .build();
    }


    private TrainOrderItem mapTrainOrderItem(ru.yandex.travel.orders.entities.TrainOrderItem orderItem) {
        TrainReservation payload = orderItem.getPayload();
        return TrainOrderItem.newBuilder()
                .setTrainNumber(nullToEmpty(payload.getTrainNumber()))
                .setCarNumber(nullToEmpty(payload.getCarNumber()))
                .setDepartureStation(nullToEmpty(payload.getStationFromCode()))
                .setArrivalStation(nullToEmpty(payload.getStationToCode()))
                .setDepartureTime(ProtoUtils.fromInstantSafe(payload.getDepartureTime()))
                .setArrivalTime(ProtoUtils.fromInstantSafe(payload.getArrivalTime()))
                .setCorrelationId(nullToEmpty(payload.getCorrelationId()))
                .build();
    }


    private HotelOrderItem mapHotelOrderItem(ru.yandex.travel.orders.entities.HotelOrderItem orderItem) {
        return HotelOrderItem.newBuilder()
                .setHotelName(nullToEmpty(orderItem.getHotelItinerary().getOrderDetails().getHotelName()))
                .setHotelAddress(nullToEmpty(orderItem.getHotelItinerary().getOrderDetails().getAddressByYandex()))
                .setHotelPhone(nullToEmpty(orderItem.getHotelItinerary().getOrderDetails().getHotelPhone()))
                .setCheckInDate(ProtoUtils.toTDate(orderItem.getHotelItinerary().getOrderDetails().getCheckinDate()))
                .setCheckOutDate(ProtoUtils.toTDate(orderItem.getHotelItinerary().getOrderDetails().getCheckoutDate()))
                .addAllGeoRegions(mapGeoRegions(orderItem.getHotelItinerary().getOrderDetails().getHotelGeoRegions()))
                .build();
    }

    private List<GeoRegion> mapGeoRegions(List<ru.yandex.travel.hotels.common.orders.GeoRegion> regions) {
        if (regions == null) {
            return List.of();
        }
        return regions.stream()
                .map(region -> GeoRegion.newBuilder()
                        .setGeoId(region.getGeoId())
                        .setType(region.getType())
                        .build())
                .collect(Collectors.toList());
    }

}
