package ru.yandex.travel.orders.grpc;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.data.domain.PageRequest;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.configurations.jdbc.TxScopeType;
import ru.yandex.travel.orders.cpa.CpaOrderSnapshotsServiceGrpc;
import ru.yandex.travel.orders.cpa.ECpaOrderStatus;
import ru.yandex.travel.orders.cpa.TBoyOrderSnapshot;
import ru.yandex.travel.orders.cpa.TBoyOrdersReq;
import ru.yandex.travel.orders.cpa.TBoyOrdersRsp;
import ru.yandex.travel.orders.cpa.TListSnapshotsReqV2;
import ru.yandex.travel.orders.cpa.TListSnapshotsRspV2;
import ru.yandex.travel.orders.cpa.TOrderSnapshot;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.grpc.helpers.TxCallWrapper;
import ru.yandex.travel.orders.infrastructure.CallDescriptor;
import ru.yandex.travel.orders.repository.cpa.CpaOrderRepository;
import ru.yandex.travel.orders.services.cpa.CpaSnapshotProvider;

import static java.util.stream.Collectors.toList;

@GrpcService(trace = false)
@Slf4j
@RequiredArgsConstructor
@EnableConfigurationProperties(CpaProperties.class)
public class CpaOrderSnapshotService extends CpaOrderSnapshotsServiceGrpc.CpaOrderSnapshotsServiceImplBase {

    private final List<CpaSnapshotProvider> services;
    private final TxCallWrapper txCallWrapper;
    private final CpaOrderRepository cpaOrderRepository;

    @Override
    public void getSnapshotsV2(TListSnapshotsReqV2 request, StreamObserver<TListSnapshotsRspV2> responseObserver) {
        txCallWrapper.synchronouslyWithTx(
                CallDescriptor.readOnly(request), responseObserver, log, this::innerGetSnapshotsV2,
                TxScopeType.READ_ONLY
        );
    }

    @Override
    public void getBoyOrders(TBoyOrdersReq request, StreamObserver<TBoyOrdersRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(
                CallDescriptor.readOnly(request), responseObserver, log, this::innerGetBoyOrders,
                TxScopeType.READ_ONLY
        );
    }

    private TBoyOrdersRsp innerGetBoyOrders(TBoyOrdersReq request) {
        Error.checkArgument(request.hasUpdatedAtFrom(), "UpdatedFrom should be NOT NULL");
        List<TBoyOrderSnapshot> snapshots;
        snapshots = loadGenericOrderSnapshotsWithLimit(request);
        boolean hasMore = snapshots.size() > request.getMaxSnapshots();
        if (request.getMaxSnapshots() > 0 && snapshots.size() > request.getMaxSnapshots()) {
            snapshots = snapshots.subList(0, request.getMaxSnapshots());
        }
        return TBoyOrdersRsp.newBuilder()
                .addAllOrderSnapshots(snapshots)
                .setHasMore(hasMore)
                .build();
    }

    private TListSnapshotsRspV2 innerGetSnapshotsV2(TListSnapshotsReqV2 request) {
        Error.checkArgument(request.hasUpdatedAtFrom(), "UpdatedFrom should be NOT NULL");
        List<TOrderSnapshot> snapshots = services.stream()
                .filter(service -> service.supports(request))
                .findAny()
                .orElseThrow(() -> Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Unsupported order type")
                        .withAttribute("value", request.getOrderType())
                        .withAttribute("serviceType", request.getServiceType())
                        .toEx())
                .getSnapshotsV2(request);

        boolean hasMore = snapshots.size() > request.getMaxSnapshots();
        if (request.getMaxSnapshots() > 0 && snapshots.size() > request.getMaxSnapshots()) {
            snapshots = snapshots.subList(0, request.getMaxSnapshots());
        }
        return TListSnapshotsRspV2.newBuilder()
                .addAllOrderSnapshots(snapshots)
                .setHasMore(hasMore)
                .build();
    }

    private List<TBoyOrderSnapshot> loadGenericOrderSnapshotsWithLimit(TBoyOrdersReq request) {
        List<Order> orders;

        Instant updatedAtFrom = ProtoUtils.toInstant(request.getUpdatedAtFrom());
        Instant updatedAtTo = getUpdatedAtToOrNow(request);

        orders = cpaOrderRepository.findOrdersByUpdatedAt(
                updatedAtFrom,
                updatedAtTo,
                // TODO(tlg-13, mbobrov): add hotel and avia orders
                Set.of(EDisplayOrderType.DT_TRAIN, EDisplayOrderType.DT_SUBURBAN, EDisplayOrderType.DT_BUS),
                PageRequest.of(0, request.getMaxSnapshots() + 1)
        );

        return orders.stream()
                .map(this::toBoyOrderSnapshot)
                .collect(toList());
    }

    private TBoyOrderSnapshot toBoyOrderSnapshot(Order order) {
        ECpaOrderStatus status = services.stream()
                .filter(service -> service.supports(order))
                .findAny()
                .orElseThrow(() ->
                        new UnsupportedOperationException(String.format("Order %s public type %s is unsupported",
                                order.getId(), order.getPublicType())))
                .getCpaOrderStatus(order);
        TBoyOrderSnapshot.Builder builder = TBoyOrderSnapshot.newBuilder()
                .setAmount(ProtoUtils.toTPrice(order.calculateOriginalTotalCost()))
                .setTravelOrderId(order.getId().toString())
                .setPartnerOrderId(order.getPrettyId())
                .setCreatedAt(ProtoUtils.fromInstant(order.getCreatedAt()))
                .setUpdatedAt(ProtoUtils.fromInstant(order.getUpdatedAt()))
                .setLabel(Strings.nullToEmpty(order.getLabel()))
                .setDiscountAmount(ProtoUtils.toTPrice(order.calculateDiscountAmount()))
                .addAllPromoCode(order.getPromoCodeApplications().stream()
                        .map(p -> p.getPromoCodeActivation().getPromoCode().getCode())
                        .collect(Collectors.toUnmodifiableList()))
                .setUsesDeferredPayment(order.getUseDeferredPayment())
                .setAmountReceiveFromUser(ProtoUtils.toTPrice(order.calculatePaidAmount()))
                .setPostPayEligible(order.isPostPayEligible())
                .setPostPayUsed(order.isFullyPostPaid())
                .setCpaOrderStatus(status)
                .setDisplayType(order.getDisplayType());
        Set<String> actions = order.getPromoCodeApplications().stream()
                .map(p -> p.getPromoCodeActivation().getPromoCode().getPromoAction().getName())
                .collect(Collectors.toSet());
        // TODO(akormushkin): add actions from generated promo codes
        // order.getGeneratedPromoCodes().getPromoCodes().forEach(c -> actions.add(c.getPromoAction().getName()));
        builder.addAllPromoAction(actions);
        // TODO(ganintsev, mbobrov): set profit
        return builder.build();
    }

    private Instant getUpdatedAtToOrNow(TBoyOrdersReq request) {
        Instant updatedAtTo;
        if (request.hasUpdatedAtTo()) {
            updatedAtTo = ProtoUtils.toInstant(request.getUpdatedAtTo());
        } else {
            updatedAtTo = Instant.now();
        }
        return updatedAtTo;
    }
}
