package ru.yandex.travel.orders.grpc;

import java.time.Instant;
import java.util.UUID;

import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.Ticket;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.proto.TDevCrashWorkflowReq;
import ru.yandex.travel.orders.proto.TDevCrashWorkflowRsp;
import ru.yandex.travel.orders.proto.TDevCreateTicketReq;
import ru.yandex.travel.orders.proto.TDevCreateTicketRsp;
import ru.yandex.travel.orders.proto.TDevGetTicketInfoReq;
import ru.yandex.travel.orders.proto.TDevGetTicketInfoRsp;
import ru.yandex.travel.orders.proto.TicketDevInterfaceDoNotUseGrpc;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.TicketRepository;
import ru.yandex.travel.orders.workflow.order.proto.TCreateIssue;
import ru.yandex.travel.orders.workflow.ticket.proto.ETicketState;
import ru.yandex.travel.workflow.TWorkflowCrashed;
import ru.yandex.travel.workflow.WorkflowProcessService;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@GrpcService
@Slf4j
@RequiredArgsConstructor
public class DevTicketService extends TicketDevInterfaceDoNotUseGrpc.TicketDevInterfaceDoNotUseImplBase {
    private final WorkflowProcessService workflowProcessService;
    private final TransactionTemplate transactionTemplate;
    private final TicketRepository ticketRepository;
    private final WorkflowRepository workflowRepository;
    private final OrderRepository orderRepository;

    @Override
    @Transactional
    public void createTicket(TDevCreateTicketReq request, StreamObserver<TDevCreateTicketRsp> responseObserver) {
        ServerUtils.synchronously(log, request, responseObserver, req -> {
            UUID orderId = null;
            if (StringUtils.isNotBlank(req.getOrderId())) {
                orderId = ProtoChecks.checkStringIsUuid("order id", req.getOrderId());
            }
            Ticket ticket = new Ticket();
            ticket.setState(ETicketState.TS_NEW);
            ticket.setId(UUID.randomUUID());
            if (orderId != null) {
                try (var ignored = NestedMdc.forEntity(orderId, null)) {
                    log.info("Creating order ticket via dev API");
                }
                ticket.setOrderId(orderId);
            }
            ticket = ticketRepository.saveAndFlush(ticket);
            Workflow workflowForTicket = Workflow.createWorkflowForEntity(ticket);
            workflowForTicket = workflowRepository.saveAndFlush(workflowForTicket);
            workflowProcessService.scheduleEvent(workflowForTicket.getId(), TCreateIssue.newBuilder()
                    .setTitle(req.getTitle())
                    .setDescription(req.getDescription())
                    .setQueueName("HBSUPTEST")
                    .setIssueType("task")
                    .build());

            return TDevCreateTicketRsp.newBuilder()
                    .setTicketId(ticket.getId().toString())
                    .build();
        });
    }

    @Override
    public void getTicketInfo(TDevGetTicketInfoReq request, StreamObserver<TDevGetTicketInfoRsp> responseObserver) {
        ServerUtils.synchronously(log, request, responseObserver, req -> transactionTemplate.execute(status -> {
            Ticket ticket = ticketRepository.getOne(ProtoChecks.checkStringIsUuid("ticket id", req.getTicketId()));
            TDevGetTicketInfoRsp.Builder builder = TDevGetTicketInfoRsp.newBuilder()
                    .setTicketId(ticket.getId().toString())
                    .setTicketState(ticket.getState().getValueDescriptor().getName())
                    .setCreatedAt(ProtoUtils.fromInstant(ticket.getCreatedAt()))
                    .setUpdatedAt(ProtoUtils.fromInstant(ticket.getUpdatedAt()));
            if (StringUtils.isNotBlank(ticket.getIssueId())) {
                builder.setIssueId(ticket.getIssueId());
            }
            if (ticket.getOrderId() != null) {
                builder.setOrderId(ticket.getOrderId().toString());
            }
            return builder.build();
        }));
    }

    @Override
    @Transactional
    public void crashWorkflow(TDevCrashWorkflowReq request, StreamObserver<TDevCrashWorkflowRsp> responseObserver) {
        ServerUtils.synchronously(log, request, responseObserver, req -> {
            Workflow workflow;
            Order order;
            switch (req.getIdsCase()) {
                case ORDERID:
                    order = orderRepository.getOne(ProtoChecks.checkStringIsUuid("order id", req.getOrderId()));
                    workflow = order.getWorkflow();
                    break;
                case WORKFLOWID:
                    workflow = workflowRepository.getOne(ProtoChecks.checkStringIsUuid("workflow id",
                            req.getWorkflowId()));
                    order = orderRepository.getOne(workflow.getEntityId());
                    break;
                case IDS_NOT_SET:
                default:
                    throw new IllegalArgumentException("One of ids should not be empty");
            }
            try (var ignored = NestedMdc.forEntity(order)) {
                log.info("Crashing workflow {}", workflow.getId());
                workflowProcessService.scheduleEvent(WellKnownWorkflow.ORDER_SUPERVISOR.getUuid(),
                        TWorkflowCrashed.newBuilder()
                                .setEntityId(order.getId().toString())
                                .setEntityType(((WorkflowEntity) order).getEntityType())
                                .setWorkflowId(workflow.getId().toString())
                                .setHappenedAt(ProtoUtils.fromInstant(Instant.now()))
                                .build());
                return TDevCrashWorkflowRsp.newBuilder().setMessage("OK").build();
            }
        });
    }
}
