package ru.yandex.travel.orders.grpc;

import java.io.IOException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import javax.annotation.PreDestroy;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.grpc.stub.StreamObserver;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.RequestBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.entities.StockInfo;
import ru.yandex.travel.orders.entities.StocksResponse;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.proto.ExchangeRateInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TGetExchangeRateReq;
import ru.yandex.travel.orders.proto.TGetExchangeRateRsp;
import ru.yandex.travel.orders.services.ExchangeRateConfigurationProperties;

@GrpcService(authenticateService = true, authenticateUser = false)
@Service
@Slf4j
@EnableConfigurationProperties(ExchangeRateConfigurationProperties.class)
public class ExchangeRateService extends ExchangeRateInterfaceV1Grpc.ExchangeRateInterfaceV1ImplBase {

    private static final ImmutableMap<ECurrency, String> CURRENCY_CODES = new ImmutableMap
            .Builder<ECurrency, String>()
            .put(ECurrency.C_USD, "2002")
            .put(ECurrency.C_RUB, "5002")
            .build();
    private final ExchangeRateConfigurationProperties config;
    private ConcurrentMap<String, RateInfo> exchangeRates;
    private AsyncHttpClientWrapper httpClient;
    private ObjectMapper objectMapper;
    private ScheduledExecutorService refreshService;

    @Autowired
    public ExchangeRateService(ExchangeRateConfigurationProperties config,
                               @Qualifier("exchangeRateAhcClientWrapper") AsyncHttpClientWrapper httpClient) {
        this.exchangeRates = new ConcurrentHashMap<>();
        this.refreshService = Executors.newSingleThreadScheduledExecutor(
                new ThreadFactoryBuilder()
                        .setNameFormat("exchange-rate-refresh-thread-%d")
                        .build());
        this.config = config;
        this.httpClient = httpClient;
        this.objectMapper = new ObjectMapper()
                .setDefaultPropertyInclusion(JsonInclude.Include.NON_EMPTY)
                .registerModule(new JavaTimeModule());

        if (config.getEnabled() == Boolean.TRUE) {
            refreshService.scheduleAtFixedRate(this::runAsyncRefresh,
                    ThreadLocalRandom.current().nextLong(10000L),
                    config.getRefreshInterval().toMillis(),
                    TimeUnit.MILLISECONDS);
        } else {
            log.warn("ExchangeRate updates are disabled; no rates will be provided with the current configuration");
        }
    }

    @Override
    public void getExchangeRate(TGetExchangeRateReq request, StreamObserver<TGetExchangeRateRsp> responseObserver) {
        ServerUtils.synchronously(
                log,
                request,
                responseObserver,
                req -> {
                    ECurrency fromCurrency = ProtoChecks.checkCurrency("fromCurrency", req.getFromCurrency());
                    ECurrency toCurrency = ProtoChecks.checkCurrency("toCurrency", req.getToCurrency());

                    RateInfo rate = doGetExchangeRate(fromCurrency, toCurrency);
                    Instant rateValidUntil = rate.getDateTime().toInstant(ZoneOffset.UTC)
                            .plus(config.getValidityInterval());
                    Error.checkState(Instant.now().isBefore(rateValidUntil),
                            "Exchange rate is stale; last updated at %s", rate.getDateTime().toString());
                    log.debug(String.format("rate for %s -- %s is %s", fromCurrency.toString(), toCurrency.toString()
                            , rate));
                    return TGetExchangeRateRsp.newBuilder()
                            .setRateValue(rate.getValue().floatValue())
                            .setRateValidUntil(ProtoUtils.fromInstant(rateValidUntil))
                            .build();
                },
                ex -> GrpcExceptionHelper.mapStatusException(log, request, ex)
        );
    }

    private RateInfo doGetExchangeRate(ECurrency fromCurrency, ECurrency toCurrency) {
        Error.checkArgument(fromCurrency == ECurrency.C_RUB || toCurrency == ECurrency.C_RUB,
                "Currently service provide only rates that refer to RUB");
        boolean revertRate = fromCurrency == ECurrency.C_RUB;
        ECurrency exchangeCurrency = revertRate ? toCurrency : fromCurrency;
        String exchangeCode = CURRENCY_CODES.get(exchangeCurrency);
        log.debug("ExchangeCode = " + exchangeCode);
        RateInfo result = exchangeRates.getOrDefault(exchangeCode, null);
        Error.checkState(result != null, "No info for currency %s", exchangeCurrency.toString());
        result.setValue(revertRate ? 1.00d / result.getValue() : result.getValue());
        return result;
    }

    private void runAsyncRefresh() {
        RequestBuilder stocksRequestBuilder = new RequestBuilder()
                .setMethod("GET")
                .setUrl(config.getUrl());
        httpClient.executeRequest(stocksRequestBuilder)
                .thenApply(rsp -> {
                    try {
                        return objectMapper.readValue(rsp.getResponseBody(), StocksResponse.class);
                    } catch (IOException e) {
                        log.error("Unable to parse Stocks response", e);
                        throw new RuntimeException("Unable to parse Stocks response", e);
                    }
                }).whenComplete((r, t) -> {
            if (t != null) {
                log.error("Stocks info unavailable. URL: " + config.getUrl(), t);
            }
            if (r != null) {
                r.getStocks().forEach(info -> {
                    String infoId = info.getId();
                    if (info.getDateTime() != null) {
                        exchangeRates.compute(infoId, (id, rate) ->
                                rate == null || info.getDateTime().isAfter(rate.getDateTime()) ?
                                        computeStockRate(info) : rate
                        );
                    }
                });
                log.debug(String.format("Stocks info updated for %s rates", this.exchangeRates.size()));
            }
        });
    }

    private RateInfo computeStockRate(StockInfo info) {
        return new RateInfo(
                info.getBuyValue() + info.getBuyValue() * config.getCommissionPercent() / 100.0,
                info.getDateTime());
    }

    @PreDestroy
    public void closeExecutor() {
        MoreExecutors.shutdownAndAwaitTermination(refreshService, config.getShutdownTimeout().toNanos(),
                TimeUnit.NANOSECONDS);
    }

    @Data
    @AllArgsConstructor
    private static class RateInfo {
        Double value;
        LocalDateTime dateTime;
    }
}

