package ru.yandex.travel.orders.grpc;

import java.math.BigDecimal;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.logging.log4j.util.Strings;
import org.javamoney.moneta.Money;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.orders.promo.AppliedPromoCampaigns;
import ru.yandex.travel.hotels.common.orders.promo.YandexPlusApplication;
import ru.yandex.travel.orders.admin.proto.EFinancialEventPaymentScheme;
import ru.yandex.travel.orders.admin.proto.EPauseExportTransactionStatus;
import ru.yandex.travel.orders.admin.proto.OrdersAdminInsecureInterfaceV1Grpc;
import ru.yandex.travel.orders.admin.proto.TAddExtraChargeReq;
import ru.yandex.travel.orders.admin.proto.TAddExtraChargeRsp;
import ru.yandex.travel.orders.admin.proto.TAddPlusPointsReq;
import ru.yandex.travel.orders.admin.proto.TAddPlusPointsRsp;
import ru.yandex.travel.orders.admin.proto.TBillingTransaction;
import ru.yandex.travel.orders.admin.proto.TCalculateHotelOrderRefundReq;
import ru.yandex.travel.orders.admin.proto.TCalculateHotelOrderRefundRsp;
import ru.yandex.travel.orders.admin.proto.TCalculateMoneyOnlyRefundReq;
import ru.yandex.travel.orders.admin.proto.TCalculateMoneyOnlyRefundRsp;
import ru.yandex.travel.orders.admin.proto.TCancelExtraChargeReq;
import ru.yandex.travel.orders.admin.proto.TCancelExtraChargeRsp;
import ru.yandex.travel.orders.admin.proto.TChangeBillingTransactionDatesReq;
import ru.yandex.travel.orders.admin.proto.TChangeBillingTransactionDatesRsp;
import ru.yandex.travel.orders.admin.proto.TCreateFinancialEventReq;
import ru.yandex.travel.orders.admin.proto.TCreateFinancialEventRsp;
import ru.yandex.travel.orders.admin.proto.TFinancialEvent;
import ru.yandex.travel.orders.admin.proto.TListTransactionsAndFinEventsReq;
import ru.yandex.travel.orders.admin.proto.TListTransactionsAndFinEventsRsp;
import ru.yandex.travel.orders.admin.proto.TManualRefundHotelOrderReq;
import ru.yandex.travel.orders.admin.proto.TManualRefundHotelOrderRsp;
import ru.yandex.travel.orders.admin.proto.TManualRefundMoneyOnlyReq;
import ru.yandex.travel.orders.admin.proto.TManualRefundMoneyOnlyRsp;
import ru.yandex.travel.orders.admin.proto.TModifyHotelOrderDetailsInsecureReq;
import ru.yandex.travel.orders.admin.proto.TModifyHotelOrderDetailsInsecureRsp;
import ru.yandex.travel.orders.admin.proto.TMoveHotelOrderToNewContractReq;
import ru.yandex.travel.orders.admin.proto.TMoveHotelOrderToNewContractRsp;
import ru.yandex.travel.orders.admin.proto.TOrderId;
import ru.yandex.travel.orders.admin.proto.TPatchPartnerCommissionReq;
import ru.yandex.travel.orders.admin.proto.TPatchPartnerCommissionRsp;
import ru.yandex.travel.orders.admin.proto.TPauseExportBillingTransactionsToYTReq;
import ru.yandex.travel.orders.admin.proto.TPauseExportBillingTransactionsToYTRsp;
import ru.yandex.travel.orders.admin.proto.TRegenerateVoucherReq;
import ru.yandex.travel.orders.admin.proto.TRegenerateVoucherRsp;
import ru.yandex.travel.orders.admin.proto.TRestoreDolphinOrderReq;
import ru.yandex.travel.orders.admin.proto.TRestoreDolphinOrderRsp;
import ru.yandex.travel.orders.admin.proto.TResumeExportBillingTransactionToYtReq;
import ru.yandex.travel.orders.admin.proto.TResumeExportBillingTransactionToYtRsp;
import ru.yandex.travel.orders.admin.proto.TResumePaymentsReq;
import ru.yandex.travel.orders.admin.proto.TResumePaymentsRsp;
import ru.yandex.travel.orders.admin.proto.TStartAutoPaymentReq;
import ru.yandex.travel.orders.admin.proto.TStartAutoPaymentRsp;
import ru.yandex.travel.orders.admin.proto.TStopPaymentsReq;
import ru.yandex.travel.orders.admin.proto.TStopPaymentsRsp;
import ru.yandex.travel.orders.cache.BalanceContractDictionary;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.PendingInvoice;
import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreementProvider;
import ru.yandex.travel.orders.grpc.helpers.TxCallWrapper;
import ru.yandex.travel.orders.infrastructure.CallDescriptor;
import ru.yandex.travel.orders.repository.AuthorizedUserRepository;
import ru.yandex.travel.orders.repository.BillingPartnerConfigRepository;
import ru.yandex.travel.orders.repository.BillingTransactionRepository;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.repository.HotelOrderRepository;
import ru.yandex.travel.orders.repository.OrderItemRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.admin.AdminPartnerPaymentsService;
import ru.yandex.travel.orders.services.admin.OrderAdminOperationsService;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.plus.YandexPlusPromoService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.payments.proto.TCancelPayment;
import ru.yandex.travel.orders.workflow.payments.schedule.proto.TStartAutoPayment;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static ru.yandex.travel.orders.infrastructure.CallDescriptor.NO_CALL_ID;

@GrpcService(authenticateService = true)
@Slf4j
@RequiredArgsConstructor
public class OrdersAdminInsecureService extends OrdersAdminInsecureInterfaceV1Grpc.OrdersAdminInsecureInterfaceV1ImplBase {
    private static final Duration exportToYTSafetyPeriod = Duration.ofMinutes(5);
    private static final Map<EFinancialEventPaymentScheme, FinancialEventPaymentScheme> PAYMENT_SCHEME_MAPPING =
            ImmutableMap.<EFinancialEventPaymentScheme, FinancialEventPaymentScheme>builder()
                    .put(EFinancialEventPaymentScheme.FEPS_HOTELS, FinancialEventPaymentScheme.HOTELS)
                    .put(EFinancialEventPaymentScheme.FEPS_SUBURBAN, FinancialEventPaymentScheme.SUBURBAN)
                    .put(EFinancialEventPaymentScheme.FEPS_TRAINS, FinancialEventPaymentScheme.TRAINS)
                    .put(EFinancialEventPaymentScheme.FEPS_AEROEXPRESS, FinancialEventPaymentScheme.AEROEXPRESS)
                    .build();
    private final WorkflowMessageSender workflowMessageSender;
    private final OrderRepository orderRepository;
    private final OrderItemRepository orderItemRepository;
    private final HotelOrderRepository hotelOrderRepository;
    private final BillingTransactionRepository billingTransactionRepository;
    private final FinancialEventRepository financialEventRepository;
    private final BalanceContractDictionary balanceContractDictionary;
    private final BillingPartnerConfigRepository billingPartnerConfigRepository;
    private final Clock clock;
    private final WorkflowRepository workflowRepository;
    private final TxCallWrapper txCallWrapper;
    private final OrderAdminOperationsService orderAdminOperationsService;
    private final AdminPartnerPaymentsService adminPartnerPaymentsService;
    private final YandexPlusPromoService yandexPlusPromoService;
    private final AuthorizedUserRepository authorizedUserRepository;
    private final FinancialEventService financialEventService;

    private static HotelOrderItem checkAndGetHotelOrderItemForMoneyOnlyRefund(HotelOrder order) {
        Error.checkArgument(order.getDisplayType() == EDisplayOrderType.DT_HOTEL,
                "Only hotel order can refund money with this method, got " + order.getDisplayType().toString());
        Error.checkState(order.getOrderItems().size() == 1, "Only 1 order item expected");
        Error.checkState(order.getState() == EHotelOrderState.OS_CONFIRMED || order.getState() == EHotelOrderState.OS_REFUNDED,
                "Order must be confirmed or refunded for money refund");
        return (HotelOrderItem) order.getOrderItems().get(0);
    }

    @Override
    public void restoreDolphinOrder(TRestoreDolphinOrderReq request,
                                    StreamObserver<TRestoreDolphinOrderRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            HotelOrder order = orderAdminOperationsService.getHotelOrder(r.getOrderId());
            return orderAdminOperationsService.restoreDolphinOrder(order.getId(), request.getCancel());
        });
    }

    @Override
    public void calculateHotelOrderRefund(TCalculateHotelOrderRefundReq request,
                                          StreamObserver<TCalculateHotelOrderRefundRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver,
                r -> {
                    Order order = getOrder(request.getOrderId());
                    return orderAdminOperationsService.calculateHotelOrderRefund(order);
                });
    }

    @Override
    public void manualRefundHotelOrder(TManualRefundHotelOrderReq request,
                                       StreamObserver<TManualRefundHotelOrderRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            Order order = getOrder(request.getOrderId());
            orderAdminOperationsService.manualRefundHotelOrder(order, request.getRefundAmount(),
                    request.getGenerateFinEvents(), request.getMoneyRefundMode(), "CLI Refund");

            return TManualRefundHotelOrderRsp.newBuilder().build();
        });
    }

    @Override
    public void calculateMoneyOnlyRefund(TCalculateMoneyOnlyRefundReq request,
                                         StreamObserver<TCalculateMoneyOnlyRefundRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, r -> {
            HotelOrder order = orderAdminOperationsService.getHotelOrder(r.getOrderId());

            return orderAdminOperationsService.calculateMoneyOnlyRefund(order);
        });
    }

    @Override
    public void manualRefundMoneyOnly(TManualRefundMoneyOnlyReq request,
                                      StreamObserver<TManualRefundMoneyOnlyRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            HotelOrder order = orderAdminOperationsService.getHotelOrder(r.getOrderId());
            return orderAdminOperationsService.manualRefundMoneyOnly(order, request.getRefundUserMoney(),
                    request.getGenerateFinEvents(),
                    request.getNewInvoiceAmount(), request.getNewInvoiceAmountMarkup(), request.getRefundAmount(),
                    request.getMoneyRefundMode(), null);
        });
    }

    @Override
    public void regenerateVoucher(TRegenerateVoucherReq request,
                                  StreamObserver<TRegenerateVoucherRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            Order order = getOrder(r.getOrderId());
            try (var ignored = NestedMdc.forEntity(order)) {
                orderAdminOperationsService.regenerateOrderVouchers(order.getId());
                return TRegenerateVoucherRsp.newBuilder()
                        .setOrderId(order.getId().toString())
                        .setOrderPrettyId(order.getPrettyId())
                        .build();
            }
        });
    }

    @Override
    public void addExtraCharge(TAddExtraChargeReq request, StreamObserver<TAddExtraChargeRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            HotelOrder hotelOrder = orderAdminOperationsService.getHotelOrder(r.getOrderId());
            try (var ignored = NestedMdc.forEntity(hotelOrder)) {
                Error.checkState(hotelOrder.getState() == EHotelOrderState.OS_MANUAL_PROCESSING ||
                                hotelOrder.getState() == EHotelOrderState.OS_CONFIRMED,
                        "Order should be in manual processing or confirmed state");
                Error.checkState(hotelOrder.getOrderItems().size() == 1,
                        "Unexpected number of services");
                Error.checkState(hotelOrder.getPayments().stream()
                                .allMatch(pi -> pi.getState() == EPaymentState.PS_FULLY_PAID ||
                                        pi.getState() == EPaymentState.PS_CANCELLED),
                        "Order already has one or more unpaid invoices");

                OrderItem orderItem = hotelOrder.getOrderItems().get(0);
                Money extraMoney = ProtoUtils.fromTPrice(r.getExtraAmount());
                log.info("Changing FiscalPrice of order item");
                HotelItinerary itinerary = ((HotelOrderItem) orderItem).getHotelItinerary();
                itinerary.addExtra(extraMoney);
                if (!orderItem.isPostPaid()) {
                    Error.checkState(orderItem.getFiscalItems().size() > 0, "Order item should have at least one fiscal " +
                            "item");
                    FiscalItem firstFiscalItem = orderItem.getFiscalItems().get(0);
                    String title;
                    if (StringUtils.isNotBlank(r.getExtraDescription())) {
                        title = r.getExtraDescription();
                    } else {
                        title = firstFiscalItem.getTitle();
                    }
                    FiscalItem newFiscalItem = FiscalItem.builder()
                            .moneyAmount(extraMoney)
                            .type(firstFiscalItem.getType())
                            .title(title)
                            .vatType(firstFiscalItem.getVatType())
                            .inn(firstFiscalItem.getInn())
                            .build();
                    log.info("Adding a new fiscal item to service: {}", newFiscalItem.toString());
                    orderItem.addFiscalItem(newFiscalItem);
                    OrderDetails details = ((HotelOrderItem) orderItem).getHotelItinerary().getOrderDetails();
                    ZoneId hotelZoneId = details.getHotelTimeZoneId();
                    if (hotelZoneId == null) {
                        hotelZoneId = ZoneId.systemDefault();
                    }
                    Instant expiresAt = details.getCheckinDate().atStartOfDay(hotelZoneId).toInstant();
                    Instant now = Instant.now(clock);
                    if (expiresAt.isBefore(now)) {
                        expiresAt = now.plus(3, ChronoUnit.DAYS);
                    }

                    log.info("Creating pending invoice");
                    PendingInvoice pendingInvoice = PendingInvoice.builder()
                            .id(UUID.randomUUID())
                            .title(r.getExtraDescription())
                            .order(hotelOrder)
                            .state(EPaymentState.PS_INVOICE_PENDING)
                            .expiresAt(expiresAt)
                            .build();
                    pendingInvoice.addItemForFiscalItem(newFiscalItem, extraMoney, Money.zero(extraMoney.getCurrency()));
                    hotelOrder.addPendingInvoice(pendingInvoice);

                    Workflow invoiceWorkflow = Workflow.createWorkflowForEntity(pendingInvoice,
                            hotelOrder.getWorkflow().getId());
                    pendingInvoice.setWorkflow(invoiceWorkflow);

                    workflowRepository.save(invoiceWorkflow);
                    hotelOrder.setActivePaymentWorkflowId(invoiceWorkflow.getId());
                }
                log.info("Moving order to WAITING_EXTRA_PAYMENT");
                hotelOrder.setState(EHotelOrderState.OS_WAITING_EXTRA_PAYMENT);
                if (orderItem.isPostPaid()) {
                    workflowMessageSender.scheduleEvent(
                        hotelOrder.getWorkflow().getId(),
                        TMoneyAcquired.newBuilder().build()
                    );
                }
            }
            return TAddExtraChargeRsp.newBuilder()
                    .setOrderId(hotelOrder.getId().toString())
                    .setOrderPrettyId(hotelOrder.getPrettyId())
                    .build();
        });
    }

    @Override
    public void cancelExtraCharge(TCancelExtraChargeReq request,
                                  StreamObserver<TCancelExtraChargeRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            HotelOrder hotelOrder = orderAdminOperationsService.getHotelOrder(r.getOrderId());
            try (var ignored = NestedMdc.forEntity(hotelOrder)) {
                Error.checkState(hotelOrder.getState() == EHotelOrderState.OS_WAITING_EXTRA_PAYMENT,
                        "Order should be in waiting extra payment state");
                Error.checkState(hotelOrder.getOrderItems().size() == 1,
                        "Unexpected number of services");
                OrderItem orderItem = hotelOrder.getOrderItems().get(0);
                Set<EPaymentState> stateSet = Set.of(EPaymentState.PS_INVOICE_PENDING, EPaymentState.PS_PAYMENT_IN_PROGRESS);
                PendingInvoice invoiceToCancel = hotelOrder.getPayments().stream()
                        .filter(pi -> stateSet.contains(pi.getState()) && pi instanceof PendingInvoice)
                        .collect(Collectors.collectingAndThen(Collectors.toList(),
                                list -> {
                                    Preconditions.checkState(list.size() == 1, "Single pending invoice expected");
                                    return (PendingInvoice) list.get(0);
                                }
                        ));
                log.info("Cancelling pending extra payment {}", invoiceToCancel.getId());

                Money toDeduct = invoiceToCancel.getPendingInvoiceItems().stream().filter(pii -> pii.getFiscalItem() != null).map(
                        pii -> {
                            log.info("Removing extra fiscal item {} from order item {}", pii.getFiscalItem().getId(),
                                    pii.getFiscalItem().getOrderItem().getId());
                            orderItem.removeFiscalItem(pii.getFiscalItem());
                            return pii.getFiscalItem().getMoneyAmount();
                        }
                ).reduce(Money.zero(hotelOrder.getCurrency()), Money::add);
                log.info("Will reduce fiscal price by {}", toDeduct);
                HotelItinerary itinerary = ((HotelOrderItem) orderItem).getHotelItinerary();
                itinerary.removeExtra(toDeduct);

                workflowMessageSender.scheduleEvent(invoiceToCancel.getWorkflow().getId(),
                        TCancelPayment.newBuilder().setRefundIfAlreadyPaid(true).build());
            }
            return TCancelExtraChargeRsp.newBuilder()
                    .setOrderId(hotelOrder.getId().toString())
                    .setOrderPrettyId(hotelOrder.getPrettyId())
                    .build();
        });
    }

    @Override
    public void pauseExportBillingTransactionsToYt(TPauseExportBillingTransactionsToYTReq request,
                                                   StreamObserver<TPauseExportBillingTransactionsToYTRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            Order order = getOrder(r.getOrderId());
            List<BillingTransaction> billingTransactions =
                    billingTransactionRepository.findAllByServiceOrderId(order.getPrettyId());
            TPauseExportBillingTransactionsToYTRsp.Builder responseBuilder =
                    TPauseExportBillingTransactionsToYTRsp.newBuilder();
            if (billingTransactions.isEmpty()) {
                responseBuilder.setStatus(EPauseExportTransactionStatus.ER_NOTHING_TO_PAUSE);
                return responseBuilder.build();
            }

            boolean isAboutToExport = false;
            for (BillingTransaction transaction : billingTransactions) {
                if (transaction.isExportedToYt() || transaction.getYtId() != null) {
                    responseBuilder.setStatus(EPauseExportTransactionStatus.EP_ALREADY_EXPORTED);
                    return responseBuilder.build();
                }
                if (transaction.getPayoutAt().isBefore(Instant.now(clock).plus(exportToYTSafetyPeriod))) {
                    isAboutToExport = true;
                }
            }
            if (isAboutToExport) {
                responseBuilder.setStatus(EPauseExportTransactionStatus.EP_ABOUT_TO_EXPORT);
                return responseBuilder.build();
            }
            responseBuilder.setStatus(EPauseExportTransactionStatus.EP_SUCCESS);
            for (BillingTransaction transaction : billingTransactions) {
                transaction.setPaused(true);
            }
            responseBuilder.setPausedTransactions(billingTransactions.size());
            return responseBuilder.build();
        });
    }

    @Override
    public void resumeExportBillingTransactionToYt(TResumeExportBillingTransactionToYtReq request,
                                                   StreamObserver<TResumeExportBillingTransactionToYtRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            BillingTransaction billingTransaction = billingTransactionRepository.findById(r.getBillingTransactionId())
                    .orElse(null);
            if (billingTransaction == null) {
                return TResumeExportBillingTransactionToYtRsp.newBuilder()
                        .setResumeResult("BillingTransaction " + r.getBillingTransactionId() + " не найдена!")
                        .build();
            }
            if (!billingTransaction.isPaused()) {
                return TResumeExportBillingTransactionToYtRsp.newBuilder()
                        .setResumeResult("BillingTransaction " + r.getBillingTransactionId()
                                + " уже находится в запущенном состоянии")
                        .build();
            }
            if (billingTransaction.getPayoutAt().isBefore(Instant.now(clock))) {
                return TResumeExportBillingTransactionToYtRsp.newBuilder()
                        .setResumeResult("BillingTransaction " + r.getBillingTransactionId()
                                + " не возобновлена, так как дата выплаты уже в прошлом: "
                                + billingTransaction.getPayoutAt().toString())
                        .build();
            }
            if (billingTransaction.getAccountingActAt().isBefore(Instant.now(clock))) {
                return TResumeExportBillingTransactionToYtRsp.newBuilder()
                        .setResumeResult("BillingTransaction " + r.getBillingTransactionId()
                                + " не возобновлена, так как дата актирования уже в прошлом: "
                                + billingTransaction.getAccountingActAt().toString())
                        .build();
            }
            billingTransaction.setPaused(false);
            return TResumeExportBillingTransactionToYtRsp.newBuilder()
                    .setResumeResult("BillingTransaction " + r.getBillingTransactionId()
                            + " успешно возобновлена")
                    .build();
        });
    }

    @Override
    public void changeBillingTransactionDates(TChangeBillingTransactionDatesReq request,
                                              StreamObserver<TChangeBillingTransactionDatesRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            if (!r.hasAccountingActAt() && !r.hasPayoutAt()) {
                return TChangeBillingTransactionDatesRsp.newBuilder()
                        .setChangeDatesResult("BillingTransaction " + r.getBillingTransactionId()
                                + ": не задано ни одной даты для смены")
                        .build();
            }
            BillingTransaction billingTransaction = billingTransactionRepository.findById(r.getBillingTransactionId())
                    .orElse(null);
            if (billingTransaction == null) {
                return TChangeBillingTransactionDatesRsp.newBuilder()
                        .setChangeDatesResult("BillingTransaction " + r.getBillingTransactionId() + " не найдена!")
                        .build();
            }
            if (!billingTransaction.isPaused()) {
                return TChangeBillingTransactionDatesRsp.newBuilder()
                        .setChangeDatesResult("BillingTransaction " + r.getBillingTransactionId() + ": нельзя менять " +
                                "даты у неостановленной транзакции")
                        .build();
            }
            if (r.hasPayoutAt()) {
                billingTransaction.setPayoutAt(ProtoUtils.toInstant(r.getPayoutAt()));
            }
            if (r.hasAccountingActAt()) {
                billingTransaction.setAccountingActAt(ProtoUtils.toInstant(r.getAccountingActAt()));
            }
            return TChangeBillingTransactionDatesRsp.newBuilder()
                    .setChangeDatesResult("BillingTransaction " + r.getBillingTransactionId()
                            + ": даты были успешно обновлены")
                    .build();
        });
    }

    @Override
    public void listTransactionsAndFinEventsReq(TListTransactionsAndFinEventsReq request,
                                                StreamObserver<TListTransactionsAndFinEventsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            Order order = getOrder(r.getOrderId());
            List<BillingTransaction> billingTransactions =
                    billingTransactionRepository.findAllByServiceOrderId(order.getPrettyId());
            List<FinancialEvent> financialEvents = financialEventRepository.findAllByOrder(order);
            TListTransactionsAndFinEventsRsp.Builder responseBuilder = TListTransactionsAndFinEventsRsp.newBuilder();
            responseBuilder.setOrderId(order.getId().toString());
            responseBuilder.setOrderPrettyId(order.getPrettyId());
            for (BillingTransaction transaction : billingTransactions) {
                responseBuilder.addBillingTransaction(mapToProtoBillingTransaction(transaction));
            }
            for (FinancialEvent event : financialEvents) {
                responseBuilder.addFinancialEvent(mapToProtoFinancialEvent(event));
            }
            return responseBuilder.build();
        });
    }

    @Override
    public void stopPayments(TStopPaymentsReq request, StreamObserver<TStopPaymentsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log,
                req -> adminPartnerPaymentsService.stopPayments(req.getBillingClientId(), req.getBillingContractId()));
    }

    @Override
    public void resumePayments(TResumePaymentsReq request, StreamObserver<TResumePaymentsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log,
                req -> adminPartnerPaymentsService.resumePayments(req.getBillingClientId(),
                        req.getBillingContractId()));
    }

    @Override
    public void createFinancialEvent(TCreateFinancialEventReq request,
                                     StreamObserver<TCreateFinancialEventRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            Instant payoutAt = ProtoUtils.toInstant(r.getPayoutAt());
            Instant accountingActAt = ProtoUtils.toInstant(r.getAccountingActAt());
            if (payoutAt.isBefore(Instant.now(clock))) {
                return TCreateFinancialEventRsp.newBuilder()
                        .setCreateResult("Фин. эвент не создан - дата выплаты (payout_at) находится в прошлом")
                        .build();
            }
            if (!payoutAt.isBefore(accountingActAt)) {
                return TCreateFinancialEventRsp.newBuilder()
                        .setCreateResult("Фин. эвент не создан - дата выплаты (payout_at) должна быть до даты " +
                                "актирования (accounting_act_at)")
                        .build();
            }
            OrderItem orderItem = orderItemRepository.findById(UUID.fromString(r.getOrderItemId())).orElse(null);
            if (orderItem == null) {
                return TCreateFinancialEventRsp.newBuilder()
                        .setCreateResult("Фин. эвент не создан - OrderItem " + r.getOrderItemId() + " не найден")
                        .build();
            }

            FinancialEventPaymentScheme paymentScheme =
                    PAYMENT_SCHEME_MAPPING.get(r.getPaymentScheme());
            if (paymentScheme == null) {
                return TCreateFinancialEventRsp.newBuilder()
                        .setCreateResult(String.format("Фин. эвент не создан, неизвестный PaymentScheme: %s",
                                r.getPaymentScheme()))
                        .build();
            }

            FinancialEvent newEvent = FinancialEvent.builder()
                    .order(orderItem.getOrder())
                    .orderPrettyId(orderItem.getOrder().getPrettyId())
                    .orderItem(orderItem)
                    .billingClientId(r.getBillingClientId())   //maybe check, that it matches to
                    // BalanceContractDictionary?
                    .billingContractId(r.getBillingContractId())
                    .type(FinancialEventType.convertFromProto(r.getType()))
                    .accrualAt(Instant.now(clock))
                    .paymentScheme(paymentScheme)
                    .payoutAt(payoutAt)
                    .accountingActAt(accountingActAt)
                    .build();
            if (r.getOriginalEventId() != 0) {
                FinancialEvent originalEvent = financialEventRepository.findById(r.getOriginalEventId()).orElse(null);
                if (originalEvent == null) {
                    return TCreateFinancialEventRsp.newBuilder()
                            .setCreateResult("Фин. эвент не создан - был указан OriginalEventId " + r.getOriginalEventId() + ", но такого события не найдено")
                            .build();
                }
                newEvent.setOriginalEvent(originalEvent);
            }
            if (r.hasPartnerAmount()) {
                newEvent.setPartnerAmount(ProtoUtils.fromTPrice(r.getPartnerAmount()));
            }
            if (r.hasPartnerRefundAmount()) {
                newEvent.setPartnerRefundAmount(ProtoUtils.fromTPrice(r.getPartnerRefundAmount()));
            }
            if (r.hasFeeAmount()) {
                newEvent.setFeeAmount(ProtoUtils.fromTPrice(r.getFeeAmount()));
            }
            if (r.hasFeeRefundAmount()) {
                newEvent.setFeeRefundAmount(ProtoUtils.fromTPrice(r.getFeeRefundAmount()));
            }
            newEvent = financialEventRepository.save(newEvent);
            return TCreateFinancialEventRsp.newBuilder()
                    .setCreateResult("Фин. эвент " + newEvent.getId() + " успешно создан")
                    .build();
        });
    }

    private TFinancialEvent mapToProtoFinancialEvent(FinancialEvent event) {
        TFinancialEvent.Builder eventBuilder = TFinancialEvent.newBuilder()
                .setId(event.getId())
                .setOrderPrettyId(event.getOrderPrettyId())
                .setPaymentScheme(event.getPaymentScheme().toString())
                .setBillingClientId(event.getBillingClientId())
                .setBillingContractId(event.getBillingContractId())
                .setType(event.getType().toString())
                .setAccrualAt(event.getAccrualAt().toString())
                .setPayoutAt(event.getPayoutAt().toString())
                .setProcessed(event.isProcessed())
                .setAccountingActAt(event.getAccountingActAt().toString());
        if (event.getOrder() != null) {
            eventBuilder.setOrderId(event.getOrder().getId().toString());
        }
        if (event.getOrderItem() != null) {
            eventBuilder.setOrderItemId(event.getOrderItem().getId().toString());
        }
        if (event.getOriginalEvent() != null) {
            eventBuilder.setOriginalEventId(event.getOriginalEvent().getId());
        }
        if (event.getPartnerAmount() != null) {
            eventBuilder.setPartnerAmount(event.getPartnerAmount().toString());
        }
        if (event.getPartnerRefundAmount() != null) {
            eventBuilder.setPartnerRefundAmount(event.getPartnerRefundAmount().toString());
        }
        if (event.getFeeAmount() != null) {
            eventBuilder.setFeeAmount(event.getFeeAmount().toString());
        }
        if (event.getFeeRefundAmount() != null) {
            eventBuilder.setFeeRefundAmount(event.getFeeRefundAmount().toString());
        }
        return eventBuilder.build();
    }

    private TBillingTransaction mapToProtoBillingTransaction(BillingTransaction transaction) {
        TBillingTransaction.Builder transactionBuilder = TBillingTransaction.newBuilder()
                .setId(transaction.getId())
                .setServiceId(transaction.getServiceId())
                .setTransactionType(transaction.getTransactionType().toString())
                .setPaymentType(transaction.getPaymentType().toString())
                .setPartnerId(transaction.getPartnerId())
                .setClientId(transaction.getClientId())
                .setTrustPaymentId(transaction.getTrustPaymentId())
                .setServiceOrderId(transaction.getServiceOrderId())
                .setValue(transaction.getValue().toString())
                .setCreatedAt(transaction.getCreatedAt().toString())
                .setPayoutAt(transaction.getPayoutAt().toString())
                .setAccountingActAt(transaction.getAccountingActAt().toString())
                .setExportedToYt(transaction.isExportedToYt())
                .setActCommitted(transaction.isActCommitted())
                .setPaused(transaction.isPaused())
                .setPaymentSystemType(transaction.getPaymentSystemType().toString());
        if (transaction.getOriginalTransaction() != null) {
            transactionBuilder.setOriginalTransactionId(transaction.getOriginalTransaction().getId());
        }
        if (transaction.getSourceFinancialEvent() != null) {
            transactionBuilder.setSourceFinancialEventId(transaction.getSourceFinancialEvent().getId());
        }
        if (transaction.getYtId() != null) {
            transactionBuilder.setYtId(transaction.getYtId());
        }
        if (transaction.getExportedToYtAt() != null) {
            transactionBuilder.setExportedToYtAt(transaction.getExportedToYtAt().toString());
        }
        if (transaction.getActCommittedAt() != null) {
            transactionBuilder.setActCommittedAt(transaction.getActCommittedAt().toString());
        }
        return transactionBuilder.build();
    }

    @Override
    public void startAutoPayment(TStartAutoPaymentReq request, StreamObserver<TStartAutoPaymentRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            Order order = getOrder(r.getOrderId());
            try (var ignored = NestedMdc.forEntity(order)) {
                Preconditions.checkState(order.getPaymentSchedule() != null, "Order does not have any deferred " +
                        "payments");
                Preconditions.checkState(order.getPaymentSchedule().getInitialPendingInvoice().getState() == EPaymentState.PS_FULLY_PAID,
                        "Initial payment is " + order.getPaymentSchedule().getInitialPendingInvoice().getState());
                Preconditions.checkState(order.getPaymentSchedule().getState() == EPaymentState.PS_PARTIALLY_PAID,
                        "Schedule is in " + order.getPaymentSchedule().getState());
                var itemsToStart = order.getPaymentSchedule().getItems()
                        .stream()
                        .filter(i -> i.getPendingInvoice().getState() == EPaymentState.PS_INVOICE_PENDING
                                && i.getBoundPaymentMethod() != null)
                        .collect(Collectors.toList());
                Preconditions.checkState(!itemsToStart.isEmpty(), "No schedule items in appropriate state");
                itemsToStart.forEach(i -> workflowMessageSender.scheduleEvent(order.getPaymentSchedule().getWorkflow().getId(),
                        TStartAutoPayment.newBuilder().setItemId(i.getId().toString()).build()));
                log.info("Manually scheduled auto payment for {} schedule items", itemsToStart.size());
                return TStartAutoPaymentRsp.newBuilder().build();
            }
        });
    }

    @Override
    public void moveHotelOrderToNewContract(TMoveHotelOrderToNewContractReq request,
                                            StreamObserver<TMoveHotelOrderToNewContractRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver,
                orderAdminOperationsService::moveHotelOrderToNewContract);
    }

    @Override
    public void addPlusPoints(TAddPlusPointsReq request, StreamObserver<TAddPlusPointsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, r -> {
            HotelOrder order = orderAdminOperationsService.getHotelOrder(r.getOrderId());
            try (var ignored = NestedMdc.forEntity(order)) {
                Preconditions.checkArgument(request.getPoints() > 0, "Illegal plus amount");
                String passportId;
                if (Strings.isNotBlank(request.getLogin())) {
                    Preconditions.checkArgument(Strings.isBlank(request.getPassportID()),
                            "Passport id should not be specified when passing login");
                    passportId =
                            authorizedUserRepository.findByIdOrderId(order.getId()).stream().filter(au -> StringUtils.equalsIgnoreCase(au.getLogin(), request.getLogin()) && Strings.isNotBlank(au.getPassportId())).map(AuthorizedUser::getPassportId).findFirst().orElseThrow(() -> new IllegalStateException("Order does not have any authorized user with login=" + request.getLogin()));
                    log.info("Detected passportId '{}' by login '{}'", passportId, request.getLogin());
                } else if (request.getAllowPassportIdAutoDetection()) {
                    if (Strings.isNotBlank(request.getPassportID())) {
                        passportId = request.getPassportID();
                    } else {
                        passportId = null;
                    }
                } else {
                    Preconditions.checkArgument(Strings.isNotBlank(request.getPassportID()),
                            "No passport id specified");
                    passportId = request.getPassportID();
                }

                int points = (int) request.getPoints();
                HotelOrderItem item = request.getIgnoreOrderStatus()
                        ? checkAndGetItemWithoutState(order)
                        : orderAdminOperationsService.checkAndGetItem(order);
                Instant topupInstant = request.getNow() ?
                        Instant.now(clock) : yandexPlusPromoService.getTopupAt(item, true);

                log.info("Manually scheduling plus topup: {} points will be added to user {} after {}",
                        points, Objects.requireNonNullElse(passportId, "<will be auto-detected>"), topupInstant);
                yandexPlusPromoService.scheduleTopupOperationForOrder(points, ProtoCurrencyUnit.RUB,
                        item.getId(), passportId, topupInstant, request.getIgnoreOrderStatus());
                if (!request.getSkipPayloadUpdate()) {
                    YandexPlusApplication plusCampaign = null;
                    if (item.getHotelItinerary().getAppliedPromoCampaigns() != null) {
                        plusCampaign = item.getHotelItinerary().getAppliedPromoCampaigns().getYandexPlus();
                    }
                    if (plusCampaign != null) {
                        if (plusCampaign.getMode() == YandexPlusApplication.Mode.TOPUP) {
                            log.info("Itinerary already contains a plus-topup payload for {} points, will add {} more",
                                    plusCampaign.getPoints(), points);
                            if (request.getOverridePayloadPointsInsteadOfSum()) {
                                plusCampaign.setPoints(points);
                            } else {
                                plusCampaign.setPoints(plusCampaign.getPoints() + points);
                            }
                        } else {
                            log.warn("Itinerary already contains a plus-withdraw payload. " +
                                    "Combined payloads are not supported, will NOT update the payload " +
                                    "({} points will be added silently)", points);
                        }
                    } else {
                        plusCampaign = YandexPlusApplication.builder()
                                .mode(YandexPlusApplication.Mode.TOPUP)
                                .points(points)
                                .build();
                        if (item.getHotelItinerary().getAppliedPromoCampaigns() == null) {
                            item.getHotelItinerary().setAppliedPromoCampaigns(AppliedPromoCampaigns.builder().yandexPlus(plusCampaign).build());
                        } else {
                            item.getHotelItinerary().getAppliedPromoCampaigns().setYandexPlus(plusCampaign);
                        }
                    }
                }
            }
            return TAddPlusPointsRsp.newBuilder()
                    .setOrderId(order.getId().toString())
                    .setOrderPrettyId(order.getPrettyId())
                    .build();
        });
    }

    @Override
    public void modifyHotelOrderDetails(TModifyHotelOrderDetailsInsecureReq request,
                                        StreamObserver<TModifyHotelOrderDetailsInsecureRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            HotelOrder order = orderAdminOperationsService.getHotelOrder(r.getParams().getOrderId());
            orderAdminOperationsService.modifyHotelOrderDetails(order, r.getParams());
            return TModifyHotelOrderDetailsInsecureRsp.newBuilder().build();
        });
    }

    @Override
    public void patchPartnerCommission(TPatchPartnerCommissionReq request,
                                       StreamObserver<TPatchPartnerCommissionRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, r -> {
            HotelOrder order = orderAdminOperationsService.getHotelOrder(r.getOrderId());
            try (var ignored = NestedMdc.forEntity(order)) {
                var orderItem = checkAndGetItemWithoutState(order);
                Preconditions.checkArgument(orderItem instanceof DirectHotelBillingPartnerAgreementProvider,
                        "Unexpected order item type");
                var agreementProvider = ((DirectHotelBillingPartnerAgreementProvider) orderItem);
                var agreement = agreementProvider.getAgreement();
                agreement.setOrderConfirmedRate(BigDecimal.valueOf(request.getTargetConfirmedRatePct(), 2));
                agreement.setOrderRefundedRate(BigDecimal.valueOf(request.getTargetRefundedRatePct(), 2));
                agreementProvider.setAgreement(agreement);

                var registeredNewEvents = financialEventService.updateEventsWithoutBalanceChanges(orderItem,
                        request.getInferBillingIdsFromEvents());
                return TPatchPartnerCommissionRsp.newBuilder().setRegisteredNewEvents(registeredNewEvents).build();
            }
        });
    }

    private HotelOrderItem checkAndGetItemWithoutState(Order order) {
        Error.checkArgument(order.getDisplayType() == EDisplayOrderType.DT_HOTEL,
                "Only hotel order can refund money with this method, got " + order.getDisplayType().toString());
        Error.checkState(order.getOrderItems().size() == 1, "Only 1 order item expected");
        return (HotelOrderItem) order.getOrderItems().get(0);
    }

    private Order getOrder(TOrderId orderId) {
        Order order = null;
        switch (orderId.getOneOfOrderIdsCase()) {
            case ORDERID:
                order = orderRepository.getOne(UUID.fromString(orderId.getOrderId()));
                break;
            case PRETTYID:
                order = orderRepository.getOrderByPrettyId(orderId.getPrettyId());
                break;
        }
        Error.checkArgument(order != null, "Order not found");
        return order;
    }
}
