package ru.yandex.travel.orders.grpc;

import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.google.protobuf.util.JsonFormat;
import io.grpc.stub.StreamObserver;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.function.forhuman.Comparator;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.commons.lang.ComparatorUtils;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.logging.ydb.YdbLogService;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.BronevikHotelItinerary;
import ru.yandex.travel.hotels.common.orders.DolphinHotelItinerary;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.logging.ydb.TOrderLogRecord;
import ru.yandex.travel.orders.admin.proto.ENotificationTransportType;
import ru.yandex.travel.orders.admin.proto.ENotificationType;
import ru.yandex.travel.orders.admin.proto.OrdersAdminInterfaceV1Grpc;
import ru.yandex.travel.orders.admin.proto.TCalculateHotelOrderRefundReq;
import ru.yandex.travel.orders.admin.proto.TCalculateHotelOrderRefundRsp;
import ru.yandex.travel.orders.admin.proto.TCalculateMoneyOnlyRefundReq;
import ru.yandex.travel.orders.admin.proto.TCalculateMoneyOnlyRefundRsp;
import ru.yandex.travel.orders.admin.proto.TChangeEmailReq;
import ru.yandex.travel.orders.admin.proto.TChangeEmailRsp;
import ru.yandex.travel.orders.admin.proto.TChangeEventStateReq;
import ru.yandex.travel.orders.admin.proto.TChangeEventStateRsp;
import ru.yandex.travel.orders.admin.proto.TChangePhoneReq;
import ru.yandex.travel.orders.admin.proto.TChangePhoneRsp;
import ru.yandex.travel.orders.admin.proto.TFilterValues;
import ru.yandex.travel.orders.admin.proto.TGetLogRecordsReq;
import ru.yandex.travel.orders.admin.proto.TGetLogRecordsRsp;
import ru.yandex.travel.orders.admin.proto.TGetOrderPayloadsReq;
import ru.yandex.travel.orders.admin.proto.TGetOrderPayloadsRsp;
import ru.yandex.travel.orders.admin.proto.TGetOrderReq;
import ru.yandex.travel.orders.admin.proto.TGetOrderRsp;
import ru.yandex.travel.orders.admin.proto.TGetPaymentsStateReq;
import ru.yandex.travel.orders.admin.proto.TGetPaymentsStateRsp;
import ru.yandex.travel.orders.admin.proto.TGetWorkflowReq;
import ru.yandex.travel.orders.admin.proto.TGetWorkflowRsp;
import ru.yandex.travel.orders.admin.proto.TLinkStartrekIssueReq;
import ru.yandex.travel.orders.admin.proto.TLinkStartrekIssueRsp;
import ru.yandex.travel.orders.admin.proto.TListOrdersReq;
import ru.yandex.travel.orders.admin.proto.TListOrdersRsp;
import ru.yandex.travel.orders.admin.proto.TManualMoneyRefundReq;
import ru.yandex.travel.orders.admin.proto.TManualMoneyRefundRsp;
import ru.yandex.travel.orders.admin.proto.TModifyHotelOrderDetailsReq;
import ru.yandex.travel.orders.admin.proto.TModifyHotelOrderDetailsRsp;
import ru.yandex.travel.orders.admin.proto.TModifyOrderPayloadReq;
import ru.yandex.travel.orders.admin.proto.TModifyOrderPayloadRsp;
import ru.yandex.travel.orders.admin.proto.TMoveHotelOrderToNewContractReq;
import ru.yandex.travel.orders.admin.proto.TMoveHotelOrderToNewContractRsp;
import ru.yandex.travel.orders.admin.proto.TOrderId;
import ru.yandex.travel.orders.admin.proto.TPauseWorkflowReq;
import ru.yandex.travel.orders.admin.proto.TPauseWorkflowRsp;
import ru.yandex.travel.orders.admin.proto.TPayloadInfo;
import ru.yandex.travel.orders.admin.proto.TRefundCancelledHotelOrderReq;
import ru.yandex.travel.orders.admin.proto.TRefundCancelledHotelOrderRsp;
import ru.yandex.travel.orders.admin.proto.TRefundHotelMoneyOnlyReq;
import ru.yandex.travel.orders.admin.proto.TRefundHotelMoneyOnlyRsp;
import ru.yandex.travel.orders.admin.proto.TRefundHotelOrderReq;
import ru.yandex.travel.orders.admin.proto.TRefundHotelOrderRsp;
import ru.yandex.travel.orders.admin.proto.TRegenerateVoucherReq;
import ru.yandex.travel.orders.admin.proto.TRegenerateVoucherRsp;
import ru.yandex.travel.orders.admin.proto.TRestoreDolphinOrderSecureReq;
import ru.yandex.travel.orders.admin.proto.TRestoreDolphinOrderSecureRsp;
import ru.yandex.travel.orders.admin.proto.TResumePaymentsReq;
import ru.yandex.travel.orders.admin.proto.TResumePaymentsRsp;
import ru.yandex.travel.orders.admin.proto.TRetryMoneyRefundReq;
import ru.yandex.travel.orders.admin.proto.TRetryMoneyRefundRsp;
import ru.yandex.travel.orders.admin.proto.TSendEventToWorkflowReq;
import ru.yandex.travel.orders.admin.proto.TSendEventToWorkflowRsp;
import ru.yandex.travel.orders.admin.proto.TSendUserNotificationReq;
import ru.yandex.travel.orders.admin.proto.TSendUserNotificationRsp;
import ru.yandex.travel.orders.admin.proto.TStopPaymentsReq;
import ru.yandex.travel.orders.admin.proto.TStopPaymentsRsp;
import ru.yandex.travel.orders.admin.proto.TStopWorkflowReq;
import ru.yandex.travel.orders.admin.proto.TStopWorkflowRsp;
import ru.yandex.travel.orders.admin.proto.TUnpauseWorkflowReq;
import ru.yandex.travel.orders.admin.proto.TUnpauseWorkflowRsp;
import ru.yandex.travel.orders.admin.proto.TUpdateTrainTicketsReq;
import ru.yandex.travel.orders.admin.proto.TUpdateTrainTicketsRsp;
import ru.yandex.travel.orders.commons.proto.EAdminAction;
import ru.yandex.travel.orders.commons.proto.EAdminRole;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.ESnippet;
import ru.yandex.travel.orders.entities.AdminFilterValues;
import ru.yandex.travel.orders.entities.AdminListOrdersParams;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.entities.BronevikOrderItem;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.GenericOrderUserRefund;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.HotelOrderItem;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.Ticket;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainOrderRefund;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.TrustRefund;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.grpc.helpers.TxCallWrapper;
import ru.yandex.travel.orders.infrastructure.CallDescriptor;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.repository.AuthorizedUserRepository;
import ru.yandex.travel.orders.repository.BusTicketRefundRepository;
import ru.yandex.travel.orders.repository.NotificationRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.TicketRepository;
import ru.yandex.travel.orders.repository.TrainInsuranceRefundRepository;
import ru.yandex.travel.orders.repository.TrainTicketRefundRepository;
import ru.yandex.travel.orders.repository.TrustRefundRepository;
import ru.yandex.travel.orders.repository.VoucherRepository;
import ru.yandex.travel.orders.repository.YandexPlusTopupRepository;
import ru.yandex.travel.orders.services.AdminUserActionAuditService;
import ru.yandex.travel.orders.services.AuthorizationAdminService;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.OrderInfoMapper;
import ru.yandex.travel.orders.services.OrdersAdminMapper;
import ru.yandex.travel.orders.services.RefundCalculationService;
import ru.yandex.travel.orders.services.VoucherInfoMapper;
import ru.yandex.travel.orders.services.admin.AdminActionTokenService;
import ru.yandex.travel.orders.services.admin.AdminPartnerPaymentsService;
import ru.yandex.travel.orders.services.admin.OrderAdminOperationsService;
import ru.yandex.travel.orders.services.buses.BusNotificationHelper;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.orders.services.indexing.OrdersIndexingService;
import ru.yandex.travel.orders.services.orders.CheckMoneyRefundsService;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.promo.campaigns.PromoCampaignsService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentRefund;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.orders.workflow.order.proto.TManualRefund;
import ru.yandex.travel.orders.workflow.order.proto.TStartManualServiceRefund;
import ru.yandex.travel.orders.workflow.train.proto.TUpdateTrainTickets;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMaintenanceService;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEvent;
import ru.yandex.travel.workflow.repository.WorkflowEventRepository;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static java.util.stream.Collectors.toMap;
import static ru.yandex.travel.orders.infrastructure.CallDescriptor.NO_CALL_ID;
import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.calculateDefaultTargetMoneyMarkup;
import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsFromProto;
import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsMarkupToProto;
import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsToProto;

/**
 * @implNote practically every method call is being logged (#logUserAction) to the db,
 * therefore all calls are read/write.
 */
@GrpcService(authenticateUser = true, authenticateService = true)
@Slf4j
@RequiredArgsConstructor
public class OrdersAdminService extends OrdersAdminInterfaceV1Grpc.OrdersAdminInterfaceV1ImplBase {
    private final OrderInfoMapper orderInfoMapper;
    private final OrderRepository orderRepository;
    private final WorkflowRepository workflowRepository;
    private final NotificationRepository notificationRepository;
    private final VoucherRepository voucherRepository;
    private final VoucherInfoMapper voucherInfoMapper;
    private final TicketRepository ticketRepository;
    private final TrustRefundRepository trustRefundRepository;
    private final WorkflowEventRepository workflowEventRepository;
    private final AuthorizationAdminService authAdminService;
    private final AuthorizationService authService;
    private final OrdersIndexingService indexingService;
    private final WorkflowMessageSender workflowMessageSender;
    private final WorkflowMaintenanceService workflowMaintenanceService;
    private final NotificationHelper notificationHelper;
    private final BusNotificationHelper busNotificationHelper;
    private final AuthorizedUserRepository authRepository;
    private final AdminUserActionAuditService actionAuditService;
    private final CheckMoneyRefundsService checkMoneyRefundsService;
    private final RefundCalculationService refundCalculationService;
    private final YdbLogService ydbLogService;
    private final OrdersAdminMapper ordersAdminMapper;
    private final PromoCampaignsService promoCampaignsService;
    private final StarTrekService starTrekService;
    private final OrderAdminOperationsService orderAdminOperationsService;
    private final AdminPartnerPaymentsService adminPartnerPaymentsService;
    private final AdminActionTokenService adminActionTokenService;
    private final YandexPlusTopupRepository plusTopupRepository;

    private final BusTicketRefundRepository busTicketRefundRepository;
    private final TrainTicketRefundRepository trainTicketRefundRepository;
    private final TrainInsuranceRefundRepository trainInsuranceRefundRepository;
    private final TxCallWrapper txCallWrapper;
    private final Logger actionLogger = LoggerFactory.getLogger("ru.yandex.travel.AdminUserActionLogger");

    private static final Counter personalDataRequestedCounter =
            Counter.builder("admin.personalData").tag("requested", "true").register(Metrics.globalRegistry);
    private static final Counter personalDataOmitedCounter =
            Counter.builder("admin.personalData").tag("requested", "false").register(Metrics.globalRegistry);


    @Override
    public void listOrders(TListOrdersReq request, StreamObserver<TListOrdersRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            EAdminAction adminAction =
                    req.getSnippetCount() != 0 && req.getSnippetList().contains(ESnippet.S_PRIVATE_INFO) ?
                            EAdminAction.AA_GET_PRIVATE_INFO : EAdminAction.AA_GET_INFO;
            authorizeUserAction(adminAction, EDisplayOrderType.DT_UNKNOWN, "User is not authorized to list orders");
            logUserAction(adminAction, "Listing orders", null,
                    String.format("snippets=%s", req.getSnippetList()));

            Error.checkArgument(req.getPage().getLimit() <= 50, "Limit must be <= 50 due to security reasons");

            TListOrdersRsp.Builder responseBuilder = TListOrdersRsp.newBuilder();
            List<UUID> listOfOrdersWithAdminFilters = indexingService.getListOfOrdersWithAdminFilters(req);
            AdminListOrdersParams requestParams = new AdminListOrdersParams(listOfOrdersWithAdminFilters,
                    req.getSorterList());

            orderRepository.findOrdersWithAdminFilters(requestParams).forEach((order) ->
                    responseBuilder.addOrder(ordersAdminMapper.mapAdminListItemFromOrder(order))
            );
            responseBuilder.setQuantity(indexingService.countOrdersWithAdminFilters(req));
            responseBuilder.setPage(req.getPage());
            AdminFilterValues actualStateValues = indexingService.getActualStateValues(req);
            responseBuilder.setFilterValues(TFilterValues.newBuilder()
                    .setOrderIdFilter(req.getOrderIdFilter())
                    .setPrettyIdFilter(req.getPrettyIdFilter())
                    .setProviderIdFilter(req.getProviderIdFilter())
                    .setEmailFilter(req.getEmailFilter())
                    .setPhoneFilter(req.getPhoneFilter())
                    .setOrderTypeFilter(req.getOrderTypeFilter())
                    .setOrderStateFilter(req.getOrderStateFilter())
                    .setPartnerTypeFilter(req.getPartnerTypeFilter())
                    .setCreatedAtFromFilter(req.getCreatedAtFromFilter())
                    .setCreatedAtToFilter(req.getCreatedAtToFilter())
                    .setPurchaseTokenFilter(req.getPurchaseTokenFilter())
                    .setCardMaskFilter(req.getCardMaskFilter())
                    .setRrnFilter(req.getRrnFilter())
                    .setPassengerNamesFilter(req.getPassengerNamesFilter())
                    .setBrokenFlagFilter(req.getBrokenFlagFilter())
                    .addAllOrderType(actualStateValues.getTypes())
                    .addAllDisplayOrderState(actualStateValues.getStates())
                    .addAllPartner(actualStateValues.getPartners())
                    .setPaymentScheduleTypeFilter(req.getPaymentScheduleTypeFilter())
                    .setTicketNumberFilter(req.getTicketNumberFilter())
                    .setYandexUidFilter(req.getYandexUidFilter())
                    .setCarrierFilter(req.getCarrierFilter())
                    .setReferralPartnerIdFilter(req.getReferralPartnerIdFilter()));
            return responseBuilder.build();
        });
    }

    @Override
    public void getOrder(TGetOrderReq request, StreamObserver<TGetOrderRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            EAdminAction adminAction =
                    req.getSnippetCount() != 0 && req.getSnippetList().contains(ESnippet.S_PRIVATE_INFO) ?
                            EAdminAction.AA_GET_PRIVATE_INFO : EAdminAction.AA_GET_INFO;
            Order order = getOrderOrThrow(req);
            authorizeUserAction(adminAction, order.getDisplayType(), "User is not authorized to get order info");
            try (var ignored = NestedMdc.forEntity(order)) {
                try {
                    logUserAction(adminAction, "Getting order info", order.getId(),
                            String.format("snippets=%s", req.getSnippetList()));

                    String userLogin = UserCredentials.get().getLogin();
                    Map<UUID, List<TrustRefund>> refundMap = order.getInvoices().stream()
                            .map(invoice -> Tuple2.tuple(invoice.getId(),
                                    trustRefundRepository.findAllByInvoiceId(invoice.getId())))
                            .collect(toMap(Tuple2::get1, Tuple2::get2));
                    TGetOrderRsp.Builder responseBuilder = TGetOrderRsp.newBuilder()
                            .setOrderInfo(orderInfoMapper.buildAdminOrderInfoFor(order, refundMap,
                                    authService.getOrderOwner(order.getId()), req.getSnippetList(),
                                    authAdminService.getAdminUserCapabilities(userLogin),
                                    adminActionTokenService.buildOrderToken(order), false))
                            .addAllAuthorizedUsers(authService.getOrderAuthorizedUsers(order.getId()).stream()
                                    .map(user -> {
                                        var userBuilder = TGetOrderRsp.TAuthorizedUser.newBuilder();
                                        if (user.getRole() != null) {
                                            userBuilder.setRole(user.getRole().toString());
                                        }
                                        userBuilder.setLogin(Strings.nullToEmpty(user.getLogin()));
                                        userBuilder.setYandexUid(Strings.nullToEmpty(user.getYandexUid()));
                                        userBuilder.setLoggedIn(user.getPassportId()!=null);
                                        return userBuilder.build();
                                    })
                                    .collect(Collectors.toList()))
                            .addAllStarTrekTicket(ticketRepository.findAllByOrderId(order.getId()).stream()
                                    .map(Ticket::getIssueId)
                                    .filter(Objects::nonNull)
                                    .collect(Collectors.toList()))
                            .setPromoCampaigns(orderInfoMapper.buildPromoCampaignsFor(
                                    promoCampaignsService.getOrderPromoCampaignsParticipationInfo(order), order))
                            .addAllVouchers(
                                    voucherInfoMapper.getVouchersInfoFor(
                                            voucherRepository.findAllByOrderId(order.getId()),
                                            req.getSnippetList()
                                    )
                            );
                    return responseBuilder.build();
                } catch (Exception ex) {
                    log.error("Getting order error", ex);
                    throw ex;
                }
            }
        });
    }

    @Override
    public void getWorkflow(TGetWorkflowReq request, StreamObserver<TGetWorkflowRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            UUID workflowId = ProtoChecks.checkStringIsUuid("workflow id", req.getWorkflowId());
            Workflow workflow = workflowRepository.findById(workflowId).orElseThrow(() ->
                    Error.with(EErrorCode.EC_NOT_FOUND, "Workflow not found").toEx());
            Order order = getOrderByUUIDOrThrow(workflow.getEntityId());
            authorizeUserAction(EAdminAction.AA_GET_WORKFLOW_INFO, order.getDisplayType(), "User is not authorized to get workflow info");
            try (var ignored = NestedMdc.forEntityId(workflow.getEntityId())) {
                logUserAction(EAdminAction.AA_GET_WORKFLOW_INFO, "Getting workflow info", null,
                        String.format("workflowId=%s, entityId=%s", workflow.getId(), workflow.getEntityId()));

                return ordersAdminMapper.mapWorkflowWithSupervised(workflow);
            }
        });
    }

    @Override
    public void getLogRecords(TGetLogRecordsReq request, StreamObserver<TGetLogRecordsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            UUID orderId = ProtoChecks.checkStringIsUuid("Order Id", req.getOrderId());
            Order order = getOrderByUUIDOrThrow(orderId);
            authorizeUserAction(EAdminAction.AA_GET_ORDER_LOGS, order.getDisplayType(), "User is not authorized to get order logs");
            try (var ignored = NestedMdc.forEntityId(orderId)) {
                logUserAction(EAdminAction.AA_GET_ORDER_LOGS, "Getting order logs", orderId);

                // TODO akormushkin, mbobrov: think of moving this logic away
                Set<UUID> recordsUuids = new HashSet<>();
                recordsUuids.add(order.getId());
                order.getOrderItems().forEach(oi -> addUidsForOrderItem(oi, recordsUuids));
                order.getArchivedOrderItems().forEach(aoi -> addUidsForOrderItem(aoi, recordsUuids));
                order.getInvoices().forEach(i -> recordsUuids.add(i.getId()));
                order.getInvoices().stream()
                        .flatMap(invoice -> trustRefundRepository.findAllByInvoiceId(invoice.getId()).stream())
                        .forEach(tr -> recordsUuids.add(tr.getId()));
                order.getPayments().stream().flatMap(p -> p.getLogEntityIds().stream()).forEach(recordsUuids::add);

                voucherRepository.findAllByOrderId(orderId).forEach(v -> recordsUuids.add(v.getId()));
                notificationRepository.findAllByOrderId(orderId).forEach(n -> {
                    recordsUuids.add(n.getId());
                    recordsUuids.addAll(n.getAttachments().stream().map(Attachment::getId).collect(Collectors.toList()));
                });
                ticketRepository.findAllByOrderId(orderId).forEach(t -> recordsUuids.add(t.getId()));

                List<TOrderLogRecord> acquiredLogs = ydbLogService.getLogsForIds(recordsUuids, req.getLevel(),
                        req.getLogger(), req.getOffset(), req.getLimit(), req.getSearchText());
                Long countAcquiredLogs = ydbLogService.countLogsForIds(recordsUuids, req.getLevel(),
                        req.getLogger(), req.getSearchText());
                return TGetLogRecordsRsp.newBuilder()
                        .setCount(countAcquiredLogs)
                        .addAllLogRecord(acquiredLogs)
                        .build();
            }
        });
    }

    private void addUidsForOrderItem(OrderItem orderItem, Set<UUID> uuids) {
        uuids.add(orderItem.getId());
        if (orderItem instanceof TrainOrderItem) {
            trainTicketRefundRepository.findAllByOrderItemId(orderItem.getId()).forEach(ttr -> uuids.add(ttr.getId()));
            trainInsuranceRefundRepository.findAllByOrderItemId(orderItem.getId()).forEach(tir -> uuids.add(tir.getId()));
        } else if (orderItem instanceof BusOrderItem) {
            busTicketRefundRepository.findAllByOrderItemId(orderItem.getId()).forEach(btr -> uuids.add(btr.getId()));
        }
        var plusTopup = plusTopupRepository.findByOrderItemId(orderItem.getId());
        if (plusTopup != null) {
            uuids.add(plusTopup.getId());
        }
    }

    @Override
    public void sendUserNotification(TSendUserNotificationReq request,
                                     StreamObserver<TSendUserNotificationRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            UUID orderId = ProtoChecks.checkStringIsUuid("Order Id", req.getOrderId());
            Order order = getOrderByUUIDOrThrow(orderId);
            authorizeUserAction(EAdminAction.AA_SEND_USER_NOTIFICATION, order.getDisplayType(),
                    "User is not authorized to send notifications to user");
            try (var ignored = NestedMdc.forEntity(order)) {
                logUserAction(EAdminAction.AA_SEND_USER_NOTIFICATION, "Sending user notification", order.getId());

                switch (order.getPublicType()) {
                    case OT_HOTEL_EXPEDIA:
                        var hotelOrder = (HotelOrder) order;
                        if (req.getTransport() == ENotificationTransportType.NTT_EMAIL) {
                            if (req.getType() == ENotificationType.NT_SUCCESS) {
                                workflowMessageSender.scheduleEvent(
                                        notificationHelper.createWorkflowForSuccessfulHotelNotification(hotelOrder),
                                        TSend.newBuilder().build());
                            } else if (req.getType() == ENotificationType.NT_REFUND) {
                                workflowMessageSender.scheduleEvent(
                                        notificationHelper.createWorkflowForRefundHotelNotification(hotelOrder),
                                        TSend.newBuilder().build());
                            } else {
                                throw Error.with(EErrorCode.EC_FAILED_PRECONDITION,
                                        "Order is not in desired state. Order state is " + hotelOrder.getState()).toEx();
                            }
                        } else {
                            throw Error.with(EErrorCode.EC_INVALID_ARGUMENT,
                                    "Notification transport type is not supported for this order").toEx();
                        }
                        break;
                    case OT_TRAIN:
                        sendTrainNotification(req, order);
                        break;
                    case OT_GENERIC:
                        if (OrderCompatibilityUtils.isTrainOrder(order)) {
                            sendTrainNotification(req, order);
                        } else if (OrderCompatibilityUtils.isBusOrder(order)) {
                            sendBusNotification(req, order);
                        } else {
                            throw Error.with(EErrorCode.EC_FAILED_PRECONDITION,
                                    "Unsupported order type: " + order.getPublicType()).toEx();
                        }
                        break;
                    case OT_AVIA_AEROFLOT:
                    case OT_BUS:
                    default:
                        throw Error.with(EErrorCode.EC_FAILED_PRECONDITION,
                                "Unsupported order type: " + order.getPublicType()).toEx();
                }
            }

            return TSendUserNotificationRsp.newBuilder()
                    .build();
        });
    }

    private void sendTrainNotification(TSendUserNotificationReq req, Order trainOrder) {
        Preconditions.checkState(trainOrder.getPublicType() == EOrderType.OT_GENERIC, "Train order must be GENERIC");
        if (req.getType() == ENotificationType.NT_SUCCESS) {
            if (req.getTransport() == ENotificationTransportType.NTT_EMAIL) {
                OrderRefund insuranceAutoRefund = null;
                for (var refund : trainOrder.getOrderRefunds()) {
                    if (refund.getRefundType() == EOrderRefundType.RT_TRAIN_INSURANCE_AUTO_RETURN) {
                        insuranceAutoRefund = refund;
                    }
                }
                UUID mailNotificationWorkflowId = notificationHelper.createWorkflowForSuccessfulTrainEmailV2(
                        (GenericOrder) trainOrder, insuranceAutoRefund);
                workflowMessageSender.scheduleEvent(mailNotificationWorkflowId, TSend.newBuilder().build());
            } else if (req.getTransport() == ENotificationTransportType.NTT_SMS) {
                workflowMessageSender.scheduleEvent(
                        notificationHelper.createWorkflowForSuccessfulTrainSms(trainOrder),
                        TSend.newBuilder().build());
            } else {
                throw Error.with(EErrorCode.EC_INVALID_ARGUMENT,
                        "Notification transport type is not supported for this order").toEx();
            }
        } else if (req.getType() == ENotificationType.NT_REFUND) {
            if (req.getTransport() == ENotificationTransportType.NTT_EMAIL) {
                Error.checkArgument(!Strings.isNullOrEmpty(req.getOrderRefundId()),
                        "Order refund ID should be present for train refund mail");
                var orderRefundId = ProtoChecks.checkStringIsUuid("order refund id",
                        req.getOrderRefundId());
                var orderRefund = trainOrder.getOrderRefunds().stream()
                        .filter(refund -> orderRefundId.equals(refund.getId()))
                        .findFirst()
                        .orElseThrow(() -> {
                            throw Error.with(EErrorCode.EC_NOT_FOUND,
                                    "Order refund not found for this order").toEx();
                        });
                Error.checkArgument(orderRefund instanceof TrainOrderRefund,
                        "Refund mail is supported only for user refunds but got %s", orderRefund.getRefundType());
                List<TrainTicketRefund> ticketRefunds = ((TrainOrderRefund) orderRefund).getTrainTicketRefunds();
                UUID mailNotificationWorkflowId = notificationHelper.createWorkflowForTrainRefundEmailV2(
                        (GenericOrder) trainOrder, orderRefund, ticketRefunds);
                workflowMessageSender.scheduleEvent(mailNotificationWorkflowId, TSend.newBuilder().build());
            } else {
                throw Error.with(EErrorCode.EC_INVALID_ARGUMENT,
                        "Notification transport type is not supported for this order").toEx();
            }
        } else {
            throw Error.with(EErrorCode.EC_FAILED_PRECONDITION,
                    "Order is not in desired state. Order state is " + trainOrder.getEntityState()).toEx();
        }
    }

    private void sendBusNotification(TSendUserNotificationReq req, Order order) {
        Preconditions.checkState(order.getPublicType() == EOrderType.OT_GENERIC, "Bus order must be GENERIC");
        if (req.getType() == ENotificationType.NT_SUCCESS) {
            if (req.getTransport() == ENotificationTransportType.NTT_EMAIL) {
                UUID mailNotificationWorkflowId = busNotificationHelper.createWorkflowForConfirmedOrderEmail(
                        (GenericOrder) order);
                workflowMessageSender.scheduleEvent(mailNotificationWorkflowId, TSend.newBuilder().build());
            } else if (req.getTransport() == ENotificationTransportType.NTT_SMS) {
                workflowMessageSender.scheduleEvent(
                        busNotificationHelper.createWorkflowForConfirmedOrderSms((GenericOrder) order),
                        TSend.newBuilder().build());
            } else {
                throw Error.with(EErrorCode.EC_INVALID_ARGUMENT,
                        "Notification transport type is not supported for this order").toEx();
            }
        } else if (req.getType() == ENotificationType.NT_REFUND) {
            if (req.getTransport() == ENotificationTransportType.NTT_EMAIL) {
                Error.checkArgument(!Strings.isNullOrEmpty(req.getOrderRefundId()),
                        "Order refund ID should be present for bus refund mail");
                var orderRefundId = ProtoChecks.checkStringIsUuid("order refund id",
                        req.getOrderRefundId());
                var orderRefund = order.getOrderRefunds().stream()
                        .filter(refund -> orderRefundId.equals(refund.getId()))
                        .findFirst()
                        .orElseThrow(() -> {
                            throw Error.with(EErrorCode.EC_NOT_FOUND,
                                    "Order refund not found for this order").toEx();
                        });
                Error.checkArgument(orderRefund instanceof GenericOrderUserRefund,
                        "Refund mail is supported only for user refunds but got %s", orderRefund.getRefundType());
                UUID mailNotificationWorkflowId = busNotificationHelper.createWorkflowForRefundEmail(
                        (GenericOrder) order, (GenericOrderUserRefund) orderRefund);
                workflowMessageSender.scheduleEvent(mailNotificationWorkflowId, TSend.newBuilder().build());
            } else {
                throw Error.with(EErrorCode.EC_INVALID_ARGUMENT,
                        "Notification transport type is not supported for this order").toEx();
            }
        } else {
            throw Error.with(EErrorCode.EC_FAILED_PRECONDITION,
                    "Order is not in desired state. Order state is " + order.getEntityState()).toEx();
        }
    }

    @Override
    public void getOrderPayloads(TGetOrderPayloadsReq request, StreamObserver<TGetOrderPayloadsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            UUID orderId = ProtoChecks.checkStringIsUuid("order id", req.getOrderId());
            Order order = getOrderByUUIDOrThrow(orderId);
            authorizeUserAction(EAdminAction.AA_GET_PRIVATE_INFO, order.getDisplayType(), "User is not authorized to get payload info");

            try (var ignored = NestedMdc.forEntity(order)) {
                logUserAction(EAdminAction.AA_GET_PRIVATE_INFO, "Getting payload info for order", order.getId());

                return TGetOrderPayloadsRsp.newBuilder()
                        .setOrderid(orderId.toString())
                        .addAllPayloads(order.getOrderItems().stream()
                                .filter(item -> item.getPayload() != null)
                                .map(item -> TPayloadInfo.newBuilder()
                                        .setOrderItemId(item.getId().toString())
                                        .setVersion(item.getVersion())
                                        .setPayload(ProtoUtils.toTJson(item.getPayload()))
                                        .build())
                                .collect(Collectors.toList()))
                        .build();
            }
        });
    }

    @Override
    public void modifyOrderPayload(TModifyOrderPayloadReq request,
                                   StreamObserver<TModifyOrderPayloadRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, req -> {
            Error.checkArgument(req.hasPayload(), "Modified payload is not provided");
            UUID orderId = ProtoChecks.checkStringIsUuid("order id", req.getOrderId());
            UUID orderItemId = ProtoChecks.checkStringIsUuid("order item id", req.getPayload().getOrderItemId());

            Order order = getOrderByUUIDOrThrow(orderId);
            authorizeUserAction(EAdminAction.AA_MODIFY, order.getDisplayType(), "User is not authorized to modify payload info");

            OrderItem orderItem = order.getOrderItems().stream()
                    .filter(item -> item.getId().equals(orderItemId))
                    .findFirst().orElseThrow(() -> Error.with(EErrorCode.EC_NOT_FOUND,
                            "No order item with id = " + orderItemId + " found for order " + orderId).toEx());
            if (request.getCheckVersion()) {
                Error.checkState(request.getPayload().getVersion() == orderItem.getVersion(),
                        "Payload version mismatch, possible concurrent modification");
            }

            try (var ignored = NestedMdc.forEntity(order)) {
                logUserAction(EAdminAction.AA_MODIFY, "Modifying payload info for order", order.getId(),
                        String.format("orderItemId=%s, comment=%s", orderItemId, req.getComment()));

                switch (orderItem.getPublicType()) {
                    case PT_EXPEDIA_HOTEL:
                        var expedia = (ExpediaOrderItem) orderItem;
                        expedia.setItinerary(ProtoUtils.fromTJson(req.getPayload().getPayload(),
                                ExpediaHotelItinerary.class));
                        break;
                    case PT_DOLPHIN_HOTEL:
                        var dolphin = (DolphinOrderItem) orderItem;
                        dolphin.setItinerary(ProtoUtils.fromTJson(req.getPayload().getPayload(),
                                DolphinHotelItinerary.class));
                        break;
                    case PT_TRAVELLINE_HOTEL:
                        var travelline = (TravellineOrderItem) orderItem;
                        travelline.setItinerary(ProtoUtils.fromTJson(req.getPayload().getPayload(),
                                TravellineHotelItinerary.class));
                        break;
                    case PT_BNOVO_HOTEL:
                        var bnovo = (BNovoOrderItem) orderItem;
                        bnovo.setItinerary(ProtoUtils.fromTJson(req.getPayload().getPayload(),
                                BNovoHotelItinerary.class));
                        break;
                    case PT_BRONEVIK_HOTEL:
                        var bronevik = (BronevikOrderItem) orderItem;
                        bronevik.setItinerary(ProtoUtils.fromTJson(req.getPayload().getPayload(),
                                BronevikHotelItinerary.class));
                        break;
                    case PT_FLIGHT:
                        var flight = (AeroflotOrderItem) orderItem;
                        flight.setPayload(ProtoUtils.fromTJson(req.getPayload().getPayload(),
                                AeroflotServicePayload.class));
                        break;
                    case PT_TRAIN:
                        var train = (TrainOrderItem) orderItem;
                        train.setReservation(ProtoUtils.fromTJson(req.getPayload().getPayload(),
                                TrainReservation.class));
                        break;
                    case PT_BUS:
                        var bus = (BusOrderItem) orderItem;
                        bus.setReservation(ProtoUtils.fromTJson(req.getPayload().getPayload(), BusReservation.class));
                        break;
                    default:
                        log.warn("OrderItemType {} not supported", orderItem.getPublicType());
                        break;
                }
            }

            return TModifyOrderPayloadRsp.newBuilder()
                    .setOrderId(orderId.toString())
                    .setOrderItemId(orderItem.getId().toString())
                    .setVersion(orderItem.getVersion())
                    .build();
        });
    }

    @Override
    public void changeEmail(TChangeEmailReq request, StreamObserver<TChangeEmailRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            var orderId = ProtoChecks.checkStringIsUuid("order id", req.getOrderId());
            var order = getOrderByUUIDOrThrow(orderId);
            authorizeUserAction(EAdminAction.AA_CHANGE_EMAIL, order.getDisplayType(), "User is not authorized to change email");
            try (var ignored = NestedMdc.forEntity(order)) {
                logUserAction(EAdminAction.AA_CHANGE_EMAIL, String.format(
                        "Change email for order %s. Old: %s, new: %s",
                        orderId, order.getEmail(), req.getNewEmail()), order.getId());

                order.setEmail(req.getNewEmail());
                authRepository.deleteAllByIdOrderIdAndRole(orderId, AuthorizedUser.OrderUserRole.VIEWER);
            }

            return TChangeEmailRsp.newBuilder()
                    .build();
        });
    }

    @Override
    public void changePhone(TChangePhoneReq request, StreamObserver<TChangePhoneRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            var orderId = ProtoChecks.checkStringIsUuid("order id", req.getOrderId());
            var order = getOrderByUUIDOrThrow(orderId);
            authorizeUserAction(EAdminAction.AA_CHANGE_PHONE, order.getDisplayType(), "User is not authorized to change phone");
            try (var ignored = NestedMdc.forEntity(order)) {
                logUserAction(EAdminAction.AA_CHANGE_PHONE, String.format(
                        "Change phone for order %s. Old: %s, new: %s",
                        orderId, order.getPhone(), req.getNewPhone()), order.getId());

                order.setPhone(req.getNewPhone());
                authRepository.deleteAllByIdOrderIdAndRole(orderId, AuthorizedUser.OrderUserRole.VIEWER);
            }

            return TChangePhoneRsp.newBuilder()
                    .build();
        });
    }

    @Override
    public void updateTrainTickets(TUpdateTrainTicketsReq request,
                                   StreamObserver<TUpdateTrainTicketsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            var orderId = ProtoChecks.checkStringIsUuid("order id", req.getOrderId());
            var order = getOrderByUUIDOrThrow(orderId);
            authorizeUserAction(EAdminAction.AA_UPDATE_TRAIN_TICKET_STATUS_WITH_IM, order.getDisplayType(),
                    "User is not authorized to update train tickets status");
            try (var ignored = NestedMdc.forEntity(order)) {
                logUserAction(EAdminAction.AA_UPDATE_TRAIN_TICKET_STATUS_WITH_IM,
                        "Scheduling update train tickets", order.getId());

                if (order.getDisplayType() == EDisplayOrderType.DT_TRAIN) {
                    if (OrderCompatibilityUtils.isConfirmed(order) && !order.isUserActionScheduled()) {
                        order.setUserActionScheduled(true);
                        workflowMessageSender.scheduleEvent(order.getWorkflow().getId(),
                                TUpdateTrainTickets.newBuilder().build());
                    }
                } else {
                    throw Error.with(EErrorCode.EC_FAILED_PRECONDITION,
                            "Only train orders are expected but got " + order.getDisplayType()).toEx();
                }
            }

            return TUpdateTrainTicketsRsp.newBuilder()
                    .build();
        });
    }

    @Override
    public void retryMoneyRefund(TRetryMoneyRefundReq request, StreamObserver<TRetryMoneyRefundRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            var orderId = ProtoChecks.checkStringIsUuid("order id", req.getOrderId());
            try (var ignored = NestedMdc.forEntityId(orderId)) {
                Order order = getOrderByUUIDOrThrow(orderId);
                authorizeUserAction(EAdminAction.AA_REFUND_USER_MONEY, order.getDisplayType(), "User is not authorized to refund money");
                //  TODO(ganintsev): check request.getAdminActionToken()
                if (!(order.getCurrentInvoice() instanceof TrustInvoice)) {
                    Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Current invoice type not supported").andThrow();
                }
                TrustInvoice invoice = (TrustInvoice) order.getCurrentInvoice();
                if (invoice.getWorkflow().getState() != EWorkflowState.WS_RUNNING) {
                    Error.with(EErrorCode.EC_FAILED_PRECONDITION, "TPaymentRefund event not found").andThrow();
                }
                Stream<WorkflowEvent> paymentRefundEvents = workflowEventRepository
                        .findAllByWorkflowId(invoice.getWorkflow().getId())
                        .stream()
                        .filter(e -> e.getData() instanceof TPaymentRefund);
                if (!Strings.isNullOrEmpty(request.getOrderRefundId())) {
                    paymentRefundEvents = paymentRefundEvents.filter(e ->
                            request.getOrderRefundId().equals(((TPaymentRefund) e.getData()).getOrderRefundId()));
                }
                WorkflowEvent paymentRefundEvent = paymentRefundEvents
                        .max(Comparator.naturalComparatorBy(WorkflowEvent::getId))
                        .orElse(null);
                if (paymentRefundEvent == null) {
                    Error.with(EErrorCode.EC_FAILED_PRECONDITION, "TPaymentRefund event not found").andThrow();
                }

                logUserAction(EAdminAction.AA_REFUND_USER_MONEY, "Refunding money", order.getId());

                if (invoice.getInvoiceState() == ETrustInvoiceState.IS_REFUNDING) {
                    invoice.setState(ETrustInvoiceState.IS_CLEARED);
                    workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(), paymentRefundEvent.getData());
                    log.info("Payment refund restarted, source event id = " + paymentRefundEvent.getId().toString());
                    return TRetryMoneyRefundRsp.getDefaultInstance();
                }
                if (invoice.getInvoiceState() == ETrustInvoiceState.IS_CLEARED &&
                        order.getWorkflow().getState() == EWorkflowState.WS_PAUSED) {
                    order.getWorkflow().setState(EWorkflowState.WS_RUNNING);
                    workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(), paymentRefundEvent.getData());
                    log.info("Payment refund restarted, source event id = " + paymentRefundEvent.getId().toString());
                    return TRetryMoneyRefundRsp.getDefaultInstance();
                }
                throw Error.with(EErrorCode.EC_UNAVAILABLE, "Dont know how to retry trust refund").toEx();
            }
        });
    }

    @Override
    public void refundCancelledHotelOrder(TRefundCancelledHotelOrderReq request,
                                          StreamObserver<TRefundCancelledHotelOrderRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            Error.checkArgument(req.hasRefundAmount(), "Refund amount should be specified");

            var orderId = ProtoChecks.checkStringIsUuid("order id", req.getOrderId());
            var order = getOrderByUUIDOrThrow(orderId);
            authorizeUserAction(EAdminAction.AA_REFUND_USER_MONEY, order.getDisplayType(),
                    "User is not authorized to refund money for cancelled hotel order");
            try (var ignored = NestedMdc.forEntity(order)) {
                Error.checkArgument(order.getDisplayType() == EDisplayOrderType.DT_HOTEL,
                        "Only hotel order can refund money with this method, got " + order.getDisplayType().toString());
                HotelOrder hotelOrder = (HotelOrder) order;
                Error.checkArgument(hotelOrder.getState() == EHotelOrderState.OS_CONFIRMED,
                        "Hotel order should be in CONFIRMED state, got " + hotelOrder.getState());
                Error.checkState(order.getOrderItems().size() == 1, "Only 1 order item expected");

                logUserAction(EAdminAction.AA_REFUND_USER_MONEY, "Scheduling refund for hotel", order.getId(),
                        String.format("amount: %s", req.getRefundAmount()));

                HotelOrderItem orderItem = (HotelOrderItem) hotelOrder.getOrderItems().get(0);

                order.setUserActionScheduled(true);
                var refundTokenWrapper = refundCalculationService.calculateRefundForHotelItemFromRefundAmount(
                        orderItem,
                        ProtoUtils.fromTPrice(req.getRefundAmount())
                );
                TStartManualServiceRefund serviceRefundMessage = TStartManualServiceRefund.newBuilder()
                        .setToken(refundTokenWrapper.getRefundToken())
                        .build();
                workflowMessageSender.scheduleEvent(order.getWorkflow().getId(), serviceRefundMessage);
                return TRefundCancelledHotelOrderRsp.getDefaultInstance();
            }
        });
    }

    @Override
    public void manualMoneyRefund(TManualMoneyRefundReq request,
                                  StreamObserver<TManualMoneyRefundRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, req -> {
            var orderId = ProtoChecks.checkStringIsUuid("order id", req.getOrderId());
            var order = getOrderByUUIDOrThrow(orderId);
            authorizeUserAction(EAdminAction.AA_REFUND_USER_MONEY, order.getDisplayType(), "User is not authorized to refund money");
            try (var ignored = NestedMdc.forEntity(order)) {
                if (checkMoneyRefundsService.checkManualRefundAllowed(order)) {
                    logUserAction(EAdminAction.AA_REFUND_USER_MONEY, "Scheduling refund user money", order.getId(),
                            String.format("reason: %s", req.getReason()));

                    Error.checkArgument(!Strings.isNullOrEmpty(req.getReason()), "Empty refund reason");
                    var invoice = order.getCurrentInvoice(); // TODO(tivelkov): TRAVELBACK-1398 - support multiple
                    // invoices
                    switch (req.getOneOfRefundTypeCase()) {
                        case BYFISCALITEMREFUND: {
                            if (!checkMoneyRefundsService.checkByFiscalItemManualRefundAllowed(order)) {
                                Error.with(EErrorCode.EC_FAILED_PRECONDITION,
                                        "Refund by fiscalItem not allowed at the moment").andThrow();
                            }
                            Map<Long, Money> fiscalItems = new HashMap<>();
                            Map<Long, MoneyMarkup> fiscalItemsMarkup = new HashMap<>();
                            convertTargetFiscalItemsFromProto(req.getByFiscalItemRefund().getTargetFiscalItemsMap())
                                    .forEach((fiscalItemId, refundMoney) -> {
                                        if (refundMoney.isZero()) {
                                            Error.with(EErrorCode.EC_INVALID_ARGUMENT,
                                                    "Zero refund amount for fiscal item " + fiscalItemId).andThrow();
                                        }
                                        var invoiceItem = invoice.getInvoiceItems().stream()
                                                .filter(item -> item.getFiscalItemId().equals(fiscalItemId))
                                                .findFirst().orElseThrow(() -> Error.with(EErrorCode.EC_NOT_FOUND,
                                                        "Invoice item not found").toEx());
                                        if (!order.getCurrency().equals(refundMoney.getCurrency()) ||
                                                ComparatorUtils.isLessThan(invoiceItem.getPriceMoney(), refundMoney)) {
                                            Error.with(EErrorCode.EC_INVALID_ARGUMENT,
                                                    "Incorrect refund amount for fiscal item " + fiscalItemId).andThrow();
                                        }
                                        Money finalFiscalItemAmount = invoiceItem.getPriceMoney().subtract(refundMoney);
                                        fiscalItems.put(fiscalItemId, finalFiscalItemAmount);
                                        // actually, the refund markup value should be passed from the UI and the
                                        // total amount
                                        fiscalItemsMarkup.put(fiscalItemId, calculateDefaultTargetMoneyMarkup(
                                                invoiceItem.getPriceMarkup(), finalFiscalItemAmount));
                                    });
                            order.toggleUserActionScheduled(true);
                            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(),
                                    TManualRefund.newBuilder()
                                            .putAllTargetFiscalItems(convertTargetFiscalItemsToProto(fiscalItems))
                                            .putAllTargetFiscalItemsMarkup(convertTargetFiscalItemsMarkupToProto(fiscalItemsMarkup))
                                            .setReason(req.getReason())
                                            .build());
                            break;
                        }
                        case ALLMONEYREFUND: {
                            if (!checkMoneyRefundsService.checkAllMoneyRefundAllowed(order)) {
                                Error.with(EErrorCode.EC_FAILED_PRECONDITION,
                                        "All money refund not allowed at the moment").andThrow();
                            }
                            Map<Long, Money> targetFiscalItems =
                                    InvoiceUtils.buildFullRefundTargetFiscalItems(invoice.getInvoiceItems());
                            order.toggleUserActionScheduled(true);
                            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(),
                                    TManualRefund.newBuilder()
                                            .putAllTargetFiscalItems(convertTargetFiscalItemsToProto(targetFiscalItems))
                                            .setReason(req.getReason())
                                            .build());
                            break;
                        }
                        case ALLTRAINFEEREFUND: {
                            if (!checkMoneyRefundsService.checkFeeRefundAllowed(order)) {
                                Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Train fee refund not allowed at the " +
                                        "moment").andThrow();
                            }
                            Map<Long, Money> fiscalItems = InvoiceUtils.buildFullRefundTargetFiscalItems(
                                    invoice.getInvoiceItems().stream()
                                            .filter(item -> item.getFiscalItemType() == FiscalItemType.TRAIN_FEE)
                                            .collect(Collectors.toList()));
                            // TODO(ganintsev): refund buses fee BUSES-1720
                            if (fiscalItems.size() == 0) {
                                Error.with(EErrorCode.EC_INVALID_ARGUMENT, "There is no fee items").andThrow();
                            }
                            order.toggleUserActionScheduled(true);
                            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(),
                                    TManualRefund.newBuilder()
                                            .putAllTargetFiscalItems(convertTargetFiscalItemsToProto(fiscalItems))
                                            .setReason(req.getReason())
                                            .build());
                            break;
                        }
                        case ONEOFREFUNDTYPE_NOT_SET:
                        default:
                            break;
                    }
                } else {
                    Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Manual refunds not allowed at the moment").andThrow();
                }
            }
            return TManualMoneyRefundRsp.newBuilder().build();
        });
    }

    private Order getOrderOrThrow(TGetOrderReq req) {
        TOrderId.Builder orderId = TOrderId.newBuilder();
        if (req.hasOrderId()) {
            orderId.setOrderId(req.getOrderId());
        }
        if (req.hasPrettyId()) {
            orderId.setPrettyId(req.getPrettyId());
        }
        return getOrderOrThrow(orderId.build());
    }

    private Order getOrderOrThrow(TOrderId orderId) {
        Optional<Order> optOrder = Optional.empty();
        switch (orderId.getOneOfOrderIdsCase()) {
            case ORDERID:
                UUID orderIdUuid = ProtoChecks.checkStringIsUuid("Order id", orderId.getOrderId());
                optOrder = orderRepository.findById(orderIdUuid);
                break;
            case PRETTYID:
                optOrder = orderRepository.findOrderByPrettyId(orderId.getPrettyId());
                break;
            case ONEOFORDERIDS_NOT_SET:
                Error.with(EErrorCode.EC_INVALID_ARGUMENT, "One of order Ids must be set").andThrow();
        }
        return optOrder.filter(o -> !o.nullSafeRemoved()).orElseThrow(() ->
                Error.with(EErrorCode.EC_NOT_FOUND, "Order with IDs (%s, %s) not found", orderId.getOrderId(),
                        orderId.getPrettyId()).toEx()
        );
    }

    private Order getOrderByUUIDOrThrow(UUID orderId) {
        return orderRepository.findById(orderId).filter(o -> !o.nullSafeRemoved()).orElseThrow(() ->
                Error.with(EErrorCode.EC_NOT_FOUND, "Order with order id %s not found", orderId).toEx()
        );
    }

    @Override
    public void linkStartrekIssue(TLinkStartrekIssueReq request,
                                  StreamObserver<TLinkStartrekIssueRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            Order order = getOrderOrThrow(req.getOrderId());
            authorizeUserAction(EAdminAction.AA_LINK_ISSUE, order.getDisplayType(), "User is not authorized to pause workflows");
            try (var ignored = NestedMdc.forEntity(order)) {
                logUserAction(EAdminAction.AA_LINK_ISSUE,
                        String.format("Linking issue %s to order", req.getIssueKey()), order.getId());

                TLinkStartrekIssueRsp.Builder responseBuilder = TLinkStartrekIssueRsp.newBuilder()
                        .setTicketId(starTrekService.linkIssueWorkflow(req.getIssueKey(), order.getId()).toString());
                return responseBuilder.build();
            }
        });
    }

    @Override
    public void restoreDolphinOrder(TRestoreDolphinOrderSecureReq request,
                                    StreamObserver<TRestoreDolphinOrderSecureRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            EAdminAction action = EAdminAction.AA_RESTORE_DOLPHIN_ORDER;
            Order order = getOrderOrThrow(req.getOrderId());
            authorizeUserAction(action, order.getDisplayType(), "User is not authorized to restore Dolphin orders");
            try (var ignored = NestedMdc.forEntity(order)) {
                logUserAction(action, "Restoring Dolphin order", order.getId());
                orderAdminOperationsService.restoreDolphinOrder(order.getId(), req.getCancel());
                return TRestoreDolphinOrderSecureRsp.newBuilder().build();
            }
        });
    }

    @Override
    public void regenerateVouchers(TRegenerateVoucherReq request,
                                   StreamObserver<TRegenerateVoucherRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            EAdminAction action = EAdminAction.AA_REGENERATE_VOUCHERS;
            Order order = getOrderOrThrow(req.getOrderId());
            authorizeUserAction(action, order.getDisplayType(), "User is not authorized to regenerate vouchers");
            try (var ignored = NestedMdc.forEntity(order)) {
                logUserAction(action, "Regenerating vouchers", order.getId());
                orderAdminOperationsService.regenerateOrderVouchers(order.getId());
                return TRegenerateVoucherRsp.newBuilder().build();
            }
        });
    }

    // WF maintenance methods

    @Override
    public void changeEventState(TChangeEventStateReq request, StreamObserver<TChangeEventStateRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            var workflowId = ProtoChecks.checkStringIsUuid("workflow id", req.getWorkflowId());
            var workflow =
                    workflowRepository.getWorkflowWithOptimisticLockForced(workflowId).orElseThrow(() -> Error.with(EErrorCode.EC_NOT_FOUND,
                            "Workflow with id = " + workflowId + " not found").toEx());
            Order order = getOrderByUUIDOrThrow(workflow.getEntityId());
            authorizeUserAction(EAdminAction.AA_CHANGE_EVENT_STATE, order.getDisplayType(), "User is not authorized to change state of events");
            try (var ignored = NestedMdc.forEntityId(workflow.getEntityId())) {
                Error.checkState(workflow.getState() == EWorkflowState.WS_PAUSED,
                        String.format("Workflow %s should be PAUSED but is %s", workflowId, workflow.getState()));

                var event =
                        workflowEventRepository.findById(req.getEventId()).orElseThrow(() -> Error.with(EErrorCode.EC_NOT_FOUND,
                                "Event with id = " + req.getEventId() + " not found").toEx());

                logUserAction(EAdminAction.AA_CHANGE_EVENT_STATE,
                        String.format("Change event state from %s to %s for event %s, workflow %s",
                                event.getState(), req.getNewState(), req.getEventId(), workflowId), order.getId());

                event.setState(req.getNewState());

                return TChangeEventStateRsp.getDefaultInstance();
            }
        });
    }

    @Override
    public void sendEventToWorkflow(TSendEventToWorkflowReq request,
                                    StreamObserver<TSendEventToWorkflowRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            var workflowId = ProtoChecks.checkStringIsUuid("workflow id", req.getWorkflowId());
            var workflow =
                    workflowRepository.getWorkflowWithOptimisticLockForced(workflowId).orElseThrow(() -> Error.with(EErrorCode.EC_NOT_FOUND,
                            "Workflow with id = " + workflowId + " not found").toEx());
            Order order = getOrderByUUIDOrThrow(workflow.getEntityId());
            authorizeUserAction(EAdminAction.AA_SEND_EVENT, order.getDisplayType(), "User is not authorized to send events manually");
            try (var ignored = NestedMdc.forEntityId(workflow.getEntityId())) {
                Error.checkState(workflow.getState() == EWorkflowState.WS_PAUSED,
                        String.format("Workflow %s should be PAUSED but is %s", workflowId, workflow.getState()));

                logUserAction(EAdminAction.AA_SEND_EVENT,
                        String.format("Scheduling event %s for workflow %s, data: %s",
                                req.getEventClass(), workflowId, req.getEventData()), order.getId());

                try {
                    Class<?> clazz = Class.forName(req.getEventClass());
                    if (!Message.class.isAssignableFrom(clazz)) {
                        Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Wrong class name given").andThrow();
                    }

                    Message.Builder messageBuilder =
                            (Message.Builder) clazz.getDeclaredMethod("newBuilder").invoke(null);
                    if (req.hasEventData() && !Strings.isNullOrEmpty(req.getEventData().getValue())) {
                        JsonFormat.parser().merge(req.getEventData().getValue(), messageBuilder);
                    }
                    workflowMessageSender.scheduleEvent(workflowId, messageBuilder.build());
                } catch (ClassNotFoundException e) {
                    Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Message class not found")
                            .withCause(e).andThrow();
                } catch (InvalidProtocolBufferException e) {
                    Error.with(EErrorCode.EC_INVALID_ARGUMENT,
                                    "EventData is not correct for message " + req.getEventClass())
                            .withCause(e).andThrow();
                } catch (IllegalAccessException | InvocationTargetException | NoSuchMethodException e) {
                    Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Provided class is not proto message")
                            .withCause(e).andThrow();
                }
                return TSendEventToWorkflowRsp.getDefaultInstance();
            }
        });
    }

    @Override
    public void pauseWorkflow(TPauseWorkflowReq request, StreamObserver<TPauseWorkflowRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            var workflowId = ProtoChecks.checkStringIsUuid("workflow id", req.getWorkflowId());
            Workflow workflow = workflowRepository.getOne(workflowId);
            Order order = getOrderByUUIDOrThrow(workflow.getEntityId());
            authorizeUserAction(EAdminAction.AA_PAUSE_WORKFLOW, order.getDisplayType(), "User is not authorized to pause workflows");
            try (var ignored = NestedMdc.forEntityId(workflow.getEntityId())) {
                logUserAction(EAdminAction.AA_PAUSE_WORKFLOW, "Pausing workflow", order.getId(),
                        String.format("workflowId=%s", workflowId));
                workflowMaintenanceService.pauseRunningWorkflow(workflowId);
                if (req.getChildrenIncluded()) {
                    workflowMaintenanceService.pauseSupervisedRunningWorkflows(workflowId);
                }
                return TPauseWorkflowRsp.newBuilder().build();
            }
        });
    }

    @Override
    public void unpauseWorkflow(TUnpauseWorkflowReq request, StreamObserver<TUnpauseWorkflowRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            var workflowId = ProtoChecks.checkStringIsUuid("workflow id", req.getWorkflowId());
            Workflow workflow = workflowRepository.getOne(workflowId);
            Order order = getOrderByUUIDOrThrow(workflow.getEntityId());
            authorizeUserAction(EAdminAction.AA_UNPAUSE_WORKFLOW, order.getDisplayType(), "User is not authorized to unpause workflows");
            try (var ignored = NestedMdc.forEntityId(workflow.getEntityId())) {
                logUserAction(EAdminAction.AA_UNPAUSE_WORKFLOW, "Unpausing workflow", order.getId(),
                        String.format("workflowId=%s", workflowId));

                workflowMaintenanceService.resumePausedWorkflow(workflowId);
                if (req.getChildrenIncluded()) {
                    workflowMaintenanceService.resumeSupervisedPausedWorkflows(workflowId);
                }
                return TUnpauseWorkflowRsp.newBuilder().build();
            }
        });
    }

    @Override
    public void stopWorkflow(TStopWorkflowReq request, StreamObserver<TStopWorkflowRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            var workflowId = ProtoChecks.checkStringIsUuid("workflow id", req.getWorkflowId());
            Workflow workflow = workflowRepository.getOne(workflowId);
            Order order = getOrderByUUIDOrThrow(workflow.getEntityId());
            authorizeUserAction(EAdminAction.AA_STOP_WORKFLOW, order.getDisplayType(), "User is not authorized to stop workflows");
            try (var ignored = NestedMdc.forEntityId(workflow.getEntityId())) {
                logUserAction(EAdminAction.AA_STOP_WORKFLOW, "Stopping workflow", order.getId(),
                        String.format("workflowId=%s", workflowId));

                workflowMaintenanceService.stopRunningWorkflow(workflowId);
                if (req.getChildrenIncluded()) {
                    workflowMaintenanceService.stopSupervisedRunningWorkflows(workflowId);
                }
                return TStopWorkflowRsp.newBuilder().build();
            }
        });
    }

    @Override
    public void calculateHotelOrderRefund(TCalculateHotelOrderRefundReq request,
                                          StreamObserver<TCalculateHotelOrderRefundRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, r -> {
            Order order = getOrderOrThrow(request.getOrderId());
            authorizeUserAction(EAdminAction.AA_CALCULATE_HOTEL_ORDER_REFUND, order.getDisplayType(), "User is not authorized to calculate " +
                    "hotel order refund");
            return orderAdminOperationsService.calculateHotelOrderRefund(order);
        });
    }

    @Override
    public void refundHotelOrder(TRefundHotelOrderReq request,
                                 StreamObserver<TRefundHotelOrderRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            Order order = getOrderOrThrow(request.getOrderId());
            TCalculateHotelOrderRefundRsp calculatedHotelOrderRefund =
                    orderAdminOperationsService.calculateHotelOrderRefund(order);

            EAdminAction action;

            if (request.getRefundAmount().equals(calculatedHotelOrderRefund.getPaidAmount())) {
                authorizeUserAction(EAdminAction.AA_ONLY_FULL_REFUND_HOTEL_ORDER, order.getDisplayType(), "User is not authorized to full " +
                        "refund order");
                action = EAdminAction.AA_ONLY_FULL_REFUND_HOTEL_ORDER;
            } else {
                authorizeUserAction(EAdminAction.AA_REFUND_HOTEL_ORDER, order.getDisplayType(), "User is not authorized to refund order");
                action = EAdminAction.AA_REFUND_HOTEL_ORDER;
            }

            adminActionTokenService.checkOrderToken(order, request.getAdminActionToken());

            try (var ignored = NestedMdc.forEntity(order)) {
                logUserAction(action, String.format("Scheduling refund user money for order %s, reason: %s",
                                order.getId(), request.getReason()), order.getId());
            }

            Error.checkArgument(!Strings.isNullOrEmpty(request.getReason()), "Empty refund reason");

            EMoneyRefundMode moneyRefundMode = order.calculateDiscountAmount().isZero()
                    ? EMoneyRefundMode.MRM_PROMO_MONEY_FIRST
                    : EMoneyRefundMode.MRM_PROPORTIONAL;

            orderAdminOperationsService.manualRefundHotelOrder(order, request.getRefundAmount(),
                    request.getGenerateFinEvents(), moneyRefundMode, request.getReason());

            return TRefundHotelOrderRsp.newBuilder().build();
        });
    }

    @Override
    public void calculateHotelMoneyOnlyRefund(TCalculateMoneyOnlyRefundReq request,
                                              StreamObserver<TCalculateMoneyOnlyRefundRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, r -> {
            Order order = getOrderOrThrow(request.getOrderId());
            authorizeUserAction(EAdminAction.AA_CALCULATE_HOTEL_MONEY_ONLY_REFUND, order.getDisplayType(),
                    "User is not authorized to refund money");
            return orderAdminOperationsService.calculateMoneyOnlyRefund(order);
        });
    }

    @Override
    public void refundHotelMoneyOnly(TRefundHotelMoneyOnlyReq request,
                                     StreamObserver<TRefundHotelMoneyOnlyRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            Order order = getOrderOrThrow(request.getOrderId());
            authorizeUserAction(EAdminAction.AA_MANUAL_REFUND_MONEY_ONLY, order.getDisplayType(), "User is not authorized to refund money");

            adminActionTokenService.checkOrderToken(order, request.getAdminActionToken());

            try (var ignored = NestedMdc.forEntityId(order.getId())) {
                logUserAction(EAdminAction.AA_MANUAL_REFUND_MONEY_ONLY,
                        String.format("Scheduling refund user money for order %s, reason: %s",
                                order.getId(), request.getReason()), order.getId());
            }

            Error.checkArgument(!Strings.isNullOrEmpty(request.getReason()), "Empty refund reason");

            EMoneyRefundMode moneyRefundMode = order.calculateDiscountAmount().isZero()
                    ? EMoneyRefundMode.MRM_PROMO_MONEY_FIRST
                    : EMoneyRefundMode.MRM_PROPORTIONAL;

            orderAdminOperationsService.manualRefundMoneyOnly(order, request.getRefundUserMoney(),
                    request.getGenerateFinEvents(),
                    request.getNewInvoiceAmount(), request.getNewInvoiceAmountMarkup(), request.getRefundAmount(),
                    moneyRefundMode, request.getReason());

            return TRefundHotelMoneyOnlyRsp.newBuilder().build();
        });
    }

    @Override
    public void modifyHotelOrderDetails(TModifyHotelOrderDetailsReq request,
                                        StreamObserver<TModifyHotelOrderDetailsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, r -> {
            Order order = getOrderOrThrow(request.getParams().getOrderId());
            Error.checkArgument(order instanceof HotelOrder, "Only Hotel Orders are supported");
            Error.checkArgument(!Strings.isNullOrEmpty(request.getParams().getReason()), "Empty refund reason");
            HotelOrder hotelOrder = (HotelOrder) order;

            authorizeUserAction(EAdminAction.AA_MODIFY_HOTEL_ORDER_DETAILS, order.getDisplayType(),
                    "User is not authorized to modify order details");

            adminActionTokenService.checkOrderToken(order, request.getAdminActionToken());

            try (var ignored = NestedMdc.forEntityId(order.getId())) {
                logUserAction(EAdminAction.AA_MODIFY_HOTEL_ORDER_DETAILS,
                        String.format("Modifying details for order %s, reason: %s",
                                order.getId(), request.getParams().getReason()), order.getId());
            }
            orderAdminOperationsService.modifyHotelOrderDetails(hotelOrder, request.getParams());
            return TModifyHotelOrderDetailsRsp.newBuilder().build();
        });
    }

    @Override
    public void getPaymentsState(TGetPaymentsStateReq request, StreamObserver<TGetPaymentsStateRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log,
                req -> adminPartnerPaymentsService.getPaymentsState(req.getBillingClientId(), req.getBillingContractId()));
    }

    @Override
    public void stopPayments(TStopPaymentsReq request, StreamObserver<TStopPaymentsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            EAdminAction action = EAdminAction.AA_PARTNER_PAYMENTS_ENABLING;
            authorizeUserAction(action, EDisplayOrderType.DT_UNKNOWN, "User is not authorized to stop partner payments");
            logUserAction(action, "Stopping partner payments");
            return adminPartnerPaymentsService.stopPayments(req.getBillingClientId(), req.getBillingContractId());
        });
    }

    @Override
    public void resumePayments(TResumePaymentsReq request, StreamObserver<TResumePaymentsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            EAdminAction action = EAdminAction.AA_PARTNER_PAYMENTS_ENABLING;
            authorizeUserAction(action, EDisplayOrderType.DT_UNKNOWN, "User is not authorized to resume partner payments");
            logUserAction(action, "Resuming partner payments");
            return adminPartnerPaymentsService.resumePayments(req.getBillingClientId(), req.getBillingContractId());
        });
    }

    @Override
    public void moveHotelOrderToNewContract(TMoveHotelOrderToNewContractReq request,
                                            StreamObserver<TMoveHotelOrderToNewContractRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, req -> {
            EAdminAction action = EAdminAction.AA_MOVE_ORDER_TO_NEW_CONTRACT;
            HotelOrder order = orderAdminOperationsService.getHotelOrder(request.getOrderId());
            authorizeUserAction(action, order.getDisplayType(), "User is not authorized to move order to new contract");
            logUserAction(action, "Moving order to new contract", order.getId());
            return orderAdminOperationsService.moveHotelOrderToNewContract(req);
        });
    }

    private void authorizeUserAction(EAdminAction action, EDisplayOrderType type, String errorMessage) {
        if (!authAdminService.authorizeUserForAdminAction(UserCredentials.get(), action, type)) {
            Error.with(EErrorCode.EC_PERMISSION_DENIED, errorMessage).andThrow();
        }
    }

    private void logUserAction(EAdminAction action, String desc, UUID orderId, String descPostfix) {
        String userLogin = UserCredentials.get().getLogin();
        Set<EAdminRole> userRoles = authAdminService.getRolesForUser(userLogin);
        StringBuilder descBuilder = new StringBuilder()
                .append(desc)
                .append(String.format(" via admin API; login=%s, role=%s", userLogin, userRoles));

        if (orderId != null) {
            descBuilder.append(String.format(", orderId=%s", orderId));
        }
        if (descPostfix != null) {
            descBuilder.append(String.format(", %s", descPostfix));
        }

        if (action == EAdminAction.AA_GET_PRIVATE_INFO) {
            personalDataRequestedCounter.increment();
        } else if (action == EAdminAction.AA_GET_INFO) {
            personalDataOmitedCounter.increment();
        }
        String actionDesc = descBuilder.toString();
        actionAuditService.auditOrderAction(userLogin, orderId, action, actionDesc);
        actionLogger.info(actionDesc);
    }

    private void logUserAction(EAdminAction action, String desc, UUID orderId) {
        logUserAction(action, desc, orderId, null);
    }

    private void logUserAction(EAdminAction action, String desc) {
        logUserAction(action, desc, null, null);
    }
}
