package ru.yandex.travel.orders.grpc;

import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.commons.proto.EAdminRole;
import ru.yandex.travel.orders.grpc.helpers.TxCallWrapper;
import ru.yandex.travel.orders.idm.proto.OrdersIdmInterfaceV1Grpc;
import ru.yandex.travel.orders.idm.proto.TIdmAddRoleReq;
import ru.yandex.travel.orders.idm.proto.TIdmAddRoleRsp;
import ru.yandex.travel.orders.idm.proto.TIdmGetAllRolesReq;
import ru.yandex.travel.orders.idm.proto.TIdmGetAllRolesRsp;
import ru.yandex.travel.orders.idm.proto.TIdmNameInfo;
import ru.yandex.travel.orders.idm.proto.TIdmRemoveRoleReq;
import ru.yandex.travel.orders.idm.proto.TIdmRemoveRoleRsp;
import ru.yandex.travel.orders.idm.proto.TIdmRole;
import ru.yandex.travel.orders.idm.proto.TIdmRoleGroup;
import ru.yandex.travel.orders.idm.proto.TIdmRolesInfoReq;
import ru.yandex.travel.orders.idm.proto.TIdmRolesInfoRsp;
import ru.yandex.travel.orders.idm.proto.TIdmUser;
import ru.yandex.travel.orders.infrastructure.CallDescriptor;
import ru.yandex.travel.orders.services.AuthorizationAdminService;

import static ru.yandex.travel.orders.infrastructure.CallDescriptor.NO_CALL_ID;

@GrpcService
@RequiredArgsConstructor
@Slf4j
public class OrdersIdmService extends OrdersIdmInterfaceV1Grpc.OrdersIdmInterfaceV1ImplBase {
    public final static String SLUG_NAME = "travel_orders_admin_users";

    private final AuthorizationAdminService adminService;
    private final TxCallWrapper txCallWrapper;

    @Override
    public void rolesInfo(TIdmRolesInfoReq request, StreamObserver<TIdmRolesInfoRsp> responseObserver) {
        ServerUtils.synchronously(log, request, responseObserver, req -> TIdmRolesInfoRsp.newBuilder()
                .addRoleGroups(TIdmRoleGroup.newBuilder()
                        .setSlug(SLUG_NAME)
                        .setName(buildInfo("Travel orders administrators", "Пользователи админки сервиса заказов Я.Путешествий"))
                        .addAllRole(fromAdminRole(Set.of(EAdminRole.values())))
                        .build())
                .build());
    }

    @Override
    public void addRole(TIdmAddRoleReq request, StreamObserver<TIdmAddRoleRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            adminService.addRole(req.getLogin(), req.getRole().getValue());
            return TIdmAddRoleRsp.newBuilder()
                    .build();
        });
    }

    @Override
    public void removeRole(TIdmRemoveRoleReq request, StreamObserver<TIdmRemoveRoleRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            adminService.removeRole(req.getLogin(), req.getRole().getValue());
            return TIdmRemoveRoleRsp.newBuilder()
                    .build();
        });
    }

    @Override
    public void getAllRoles(TIdmGetAllRolesReq request, StreamObserver<TIdmGetAllRolesRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log, req -> {
            List<TIdmUser> collect = adminService.getAllRoles().stream()
                    .map(user -> TIdmUser.newBuilder()
                            .setLogin(user.getLogin())
                            .setRoleGroup(TIdmRoleGroup.newBuilder()
                                    .setSlug(SLUG_NAME)
                                    .addAllRole(fromAdminRole(user.getRoles()))
                                    .build())
                            .build())
                    .collect(Collectors.toList());
            return TIdmGetAllRolesRsp.newBuilder()
                    .addAllUsers(collect)
                    .build();
        });
    }

    private TIdmNameInfo buildInfo(String enName, String ruName) {
        return TIdmNameInfo.newBuilder()
                .setNameEn(enName)
                .setNameRu(ruName)
                .build();
    }

    private Set<TIdmRole> fromAdminRole(Set<EAdminRole> roles) {
        var result = new HashSet<TIdmRole>();
        roles.forEach(role -> {
            TIdmRole.Builder roleBuilder = TIdmRole.newBuilder();
            switch (role) {
                case AR_OPERATOR:
                    roleBuilder.setName(buildInfo("operator", "Оператор"));
                    break;
                case AR_DEVELOPER:
                    roleBuilder.setName(buildInfo("developer", "Разработчик"));
                    break;
                case AR_ADVANCED_OPERATOR:
                    roleBuilder.setName(buildInfo("Advanced Operator", "Продвинутый Оператор"));
                    break;
                case AR_ADV_AVIA_OPERATOR:
                    roleBuilder.setName(buildInfo("Advanced Avia Operator", "Продвинутый Оператор Авиа"));
                    break;
                case AR_ADV_BUS_OPERATOR:
                    roleBuilder.setName(buildInfo("Advanced Bus Operator", "Продвинутый Оператор Автобусов"));
                    break;
                case AR_ADV_HOTEL_OPERATOR:
                    roleBuilder.setName(buildInfo("Advanced Hotels Operator", "Продвинутый Оператор Отелей"));
                    break;
                case AR_ADV_SUBURBAN_OPERATOR:
                    roleBuilder.setName(buildInfo("Advanced Suburban Operator", "Продвинутый Оператор Электричек"));
                    break;
                case AR_ADV_TRAINS_OPERATOR:
                    roleBuilder.setName(buildInfo("Advanced Trains Operator", "Продвинутый Оператор Поездов"));
                    break;
                case AR_BIZDEV:
                    roleBuilder.setName(buildInfo("Bizdev", "Биздев"));
                    break;
                default:
                    return;
            }
            roleBuilder.setValue(role);

            result.add(roleBuilder.build());
        });
        return result;
    }
}
