package ru.yandex.travel.orders.grpc;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;

import com.google.common.base.Strings;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.commons.crypto.HashingUtils;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.admin.proto.TGetOrderReq;
import ru.yandex.travel.orders.admin.proto.TGetOrderRsp;
import ru.yandex.travel.orders.commons.proto.ESnippet;
import ru.yandex.travel.orders.configurations.jdbc.TxScopeType;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrustRefund;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.grpc.helpers.TxCallWrapper;
import ru.yandex.travel.orders.infrastructure.CallDescriptor;
import ru.yandex.travel.orders.proto.OrderNoAuthInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TGetUnauthorizedOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetUnauthorizedOrderInfoRsp;
import ru.yandex.travel.orders.proto.TPaymentStatusChangedReq;
import ru.yandex.travel.orders.proto.TPaymentStatusChangedRsp;
import ru.yandex.travel.orders.proto.TUnauthorizedOrderInfo;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.SimpleTrustRefundRepository;
import ru.yandex.travel.orders.repository.TrustInvoiceRepository;
import ru.yandex.travel.orders.repository.TrustRefundRepository;
import ru.yandex.travel.orders.repository.VoucherRepository;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.services.OrderInfoMapper;
import ru.yandex.travel.orders.services.VoucherInfoMapper;
import ru.yandex.travel.orders.services.admin.AdminActionTokenService;
import ru.yandex.travel.orders.services.admin.AdminUserCapabilitiesManager;
import ru.yandex.travel.orders.services.promo.campaigns.PromoCampaignsService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustInvoiceCallbackReceived;
import ru.yandex.travel.orders.workflow.trust.refund.proto.ETrustRefundState;
import ru.yandex.travel.orders.workflow.trust.refund.proto.TRefreshStatus;
import ru.yandex.travel.workflow.WorkflowMessageSender;

import static java.util.stream.Collectors.toMap;
import static ru.yandex.travel.orders.infrastructure.CallDescriptor.NO_CALL_ID;

@GrpcService(authenticateService = true)
@RequiredArgsConstructor
@Slf4j
public class OrdersNoAuthService extends OrderNoAuthInterfaceV1Grpc.OrderNoAuthInterfaceV1ImplBase {
    private final AuthorizationService authorizationService;
    private final OrderRepository orderRepository;
    private final TrustInvoiceRepository trustInvoiceRepository;
    private final SimpleTrustRefundRepository simpleTrustRefundRepository;
    private final OrderInfoMapper orderInfoMapper;
    private final WorkflowMessageSender messageSender;
    private final TxCallWrapper txCallWrapper;
    private final PromoCampaignsService promoCampaignsService;
    private final VoucherRepository voucherRepository;
    private final VoucherInfoMapper voucherInfoMapper;
    private final TrustRefundRepository trustRefundRepository;
    private final AdminActionTokenService adminActionTokenService;

    @Override
    public void getOrder(TGetOrderReq request, StreamObserver<TGetOrderRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log, req -> {
            Optional<Order> optOrder = Optional.empty();
            switch (req.getOneOfOrderIdsCase()) {
                case ORDERID:
                    UUID orderId = ProtoChecks.checkStringIsUuid("Order id", req.getOrderId());
                    optOrder = orderRepository.findById(orderId);
                    break;
                case PRETTYID:
                    optOrder = orderRepository.findOrderByPrettyId(req.getPrettyId());
                    break;
                case ONEOFORDERIDS_NOT_SET:
                    Error.with(EErrorCode.EC_INVALID_ARGUMENT, "One of order Ids must be set").andThrow();
            }
            optOrder = optOrder.filter(o -> !o.nullSafeRemoved());
            if (optOrder.isPresent()) {
                Order order = optOrder.get();
                try (var ignored = NestedMdc.forEntity(order)) {
                    Map<UUID, List<TrustRefund>> refundMap = order.getInvoices().stream()
                            .map(invoice -> Tuple2.tuple(invoice.getId(),
                                    trustRefundRepository.findAllByInvoiceId(invoice.getId())))
                            .collect(toMap(Tuple2::get1, Tuple2::get2));

                    boolean needReceiptData = req.getSnippetList().contains(ESnippet.S_RECEIPTS_DATA);
                    TGetOrderRsp.Builder responseBuilder = TGetOrderRsp.newBuilder()
                            .setOrderInfo(orderInfoMapper.buildAdminOrderInfoFor(
                                    order,
                                    refundMap,
                                    authorizationService.getOrderOwner(order.getId()),
                                    req.getSnippetList(),
                                    // TODO TRAVELBACK-3201 refactor and remove arguments (no need here)
                                    new AdminUserCapabilitiesManager(Set.of(), false),
                                    adminActionTokenService.buildOrderToken(order),
                                    needReceiptData
                            ))
                            .setPromoCampaigns(orderInfoMapper.buildPromoCampaignsFor(
                                    promoCampaignsService.getOrderPromoCampaignsParticipationInfo(order), order))
                            .addAllVouchers(
                                    voucherInfoMapper.getVouchersInfoFor(
                                            voucherRepository.findAllByOrderId(order.getId()),
                                            req.getSnippetList()
                                    )
                            );
                    return responseBuilder.build();
                }
            } else {
                throw Error.with(EErrorCode.EC_NOT_FOUND,
                        "Order with IDs (" + req.getOrderId() + ", " + req.getPrettyId() + ") not found").toEx();
            }
        }, TxScopeType.READ_ONLY);
    }

    @Override
    public void getOrderInfo(TGetOrderInfoReq request, StreamObserver<TGetOrderInfoRsp> observer) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), observer, log, req -> {
            Order order;
            switch (req.getOneOfOrderIdsCase()) {
                case ORDERID:
                    order = getOrderByIdOrThrow(req.getOrderId());
                    break;
                case PRETTYID:
                    order = getOrderByPrettyIdOrThrow(req.getPrettyId());
                    break;
                case ONEOFORDERIDS_NOT_SET:
                default:
                    throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "One of order ids must be set").toEx();
            }
            try (var ignored = NestedMdc.forEntity(order)) {
                log.debug("Getting order info");
                return TGetOrderInfoRsp.newBuilder().setResult(
                        orderInfoMapper.buildOrderInfoFor(order,
                                authorizationService.getOrderOwner(order.getId()),
                                req.getUpdateOrderOnTheFly(),
                                false,
                                req.getCalculateAggregateStateOnTheFly() ?
                                        OrderInfoMapper.AggregateStateConstructMode.CALCULATE_ON_THE_FLY :
                                        OrderInfoMapper.AggregateStateConstructMode.GET_FROM_DB
                        )
                ).build();
            }
        }, TxScopeType.READ_ONLY);
    }

    // todo(tlg-13) some services like using the api without user credentials in testing environment
    // those services should be fixed and this temporary hack should be removed
    @Override
    public void getUnauthorizedOrderInfo(TGetUnauthorizedOrderInfoReq request,
                                         StreamObserver<TGetUnauthorizedOrderInfoRsp> observer) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), observer, log, req -> {
            Order order;
            switch (req.getOneOfOrderIdsCase()) {
                case ORDERID:
                    order = getOrderByIdOrThrow(req.getOrderId());
                    break;
                case PRETTYID:
                    order = getOrderByPrettyIdOrThrow(req.getPrettyId());
                    break;
                case ONEOFORDERIDS_NOT_SET:
                default:
                    throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "One of order ids must be set").toEx();
            }
            try (var ignored = NestedMdc.forEntity(order)) {
                log.info("Getting unauthorized order info");
                var owner = authorizationService.getOrderOwner(order.getId());
                return TGetUnauthorizedOrderInfoRsp.newBuilder()
                        .setInfo(TUnauthorizedOrderInfo.newBuilder()
                                .setOrderId(order.getId().toString())
                                .setPrettyId(order.getPrettyId())
                                .setOrderType(order.getPublicType())
                                .setOwnerPassportId(Strings.nullToEmpty(owner.getPassportId()))
                                .setEmailHash(HashingUtils.hashEmail(order.getEmail()))
                                .setPhoneHash(HashingUtils.hashPhone(order.getPhone()))
                                .build())
                        .build();
            }
        }, TxScopeType.READ_ONLY);
    }

    @Override
    public void paymentStatusChangedCallback(TPaymentStatusChangedReq request,
                                             StreamObserver<TPaymentStatusChangedRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, req -> {
            //Due to https://st.yandex-team.ru/TRUSTDUTY-1216
            boolean isRefundCallback = req.getAttributesMap().containsKey("trust_refund_id");
            String status = req.getAttributesOrThrow("status");
            if (isRefundCallback) {
                String trustRefundId = req.getAttributesOrThrow("trust_refund_id");
                switch (status) {
                    case "success":
                    case "error":
                    case "failed":
                        simpleTrustRefundRepository.findByTrustRefundIdEquals(trustRefundId).ifPresentOrElse(trustRefund -> {
                            if (trustRefund.getState() == ETrustRefundState.RS_IN_PROCESS && !trustRefund.isSystemRefreshScheduled()) {
                                try (var ignored = NestedMdc.forEntity(trustRefund)) {
                                    trustRefund.setSystemRefreshScheduled(true);
                                    messageSender.scheduleEvent(trustRefund.getWorkflow().getId(),
                                            TRefreshStatus.newBuilder().build());
                                }
                            }
                        }, () -> {
                            log.error("Refund not found for trustRefundId: {}", trustRefundId);
                        });
                        break;
                    default:
                        Error.with(EErrorCode.EC_INVALID_ARGUMENT,
                                "Unhandled basket status in trust refund callback: " + status +
                                        ", trustRefundId: " + trustRefundId).andThrow();
                        break;
                }
            } else {
                String purchaseToken = req.getAttributesOrThrow("purchase_token");
                switch (status) {
                    case "success":
                    case "cancelled":
                        trustInvoiceRepository.findByPurchaseTokenEquals(purchaseToken).ifPresentOrElse(invoice -> {
                            // we're only scheduling event for invoice in case we didn't get it via polling
                            if (invoice.getInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT && invoice.isBackgroundJobActive()) {
                                try (var ignored = NestedMdc.forEntity(invoice)) {
                                    // stop background polling, and get rid of race when we got status change via
                                    // polling
                                    invoice.setBackgroundJobActive(false);
                                    invoice.setNextCheckStatusAt(null);
                                    messageSender.scheduleEvent(invoice.getWorkflow().getId(),
                                            TTrustInvoiceCallbackReceived.newBuilder()
                                                    .setStatus(status)
                                                    .build());
                                }
                            }
                        }, () -> Error.with(EErrorCode.EC_NOT_FOUND, "Invoice not found for purchaseToken "
                                + purchaseToken).andThrow());
                        break;
                    case "refund":
                        break;
                    default:
                        Error.with(EErrorCode.EC_INVALID_ARGUMENT,
                                "Unhandled basket status in trust callback: " + status +
                                        ", purchaseToken: " + purchaseToken).andThrow();
                        break;
                }
            }

            return TPaymentStatusChangedRsp.newBuilder()
                    .build();
        });
    }

    private Order getOrderByIdOrThrow(String protoOrderId) {
        UUID orderId = ProtoChecks.checkStringIsUuid("order id", protoOrderId);
        Optional<Order> order = orderRepository.findById(orderId);

        order = order.filter(o -> !o.nullSafeRemoved());

        if (order.isPresent()) {
            return order.get();
        } else {
            throw Error.with(EErrorCode.EC_NOT_FOUND, "No such order").withAttribute("order_id", orderId).toEx();
        }
    }

    private Order getOrderByPrettyIdOrThrow(String protoPrettyId) {
        Order order = orderRepository.getOrderByPrettyId(protoPrettyId);
        if (order == null) {
            throw Error.with(EErrorCode.EC_NOT_FOUND, "No such order").withAttribute("pretty_id", protoPrettyId).toEx();
        }
        if (order.nullSafeRemoved()) {
            throw Error.with(EErrorCode.EC_NOT_FOUND, "No such order").withAttribute("pretty_id", protoPrettyId).toEx();
        }
        return order;
    }
}
