package ru.yandex.travel.orders.grpc;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import com.google.common.collect.ImmutableSet;
import com.google.protobuf.Message;
import io.grpc.Status;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.Hibernate;
import org.javamoney.moneta.Money;
import org.springframework.data.domain.PageRequest;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.model.PayloadMapper;
import ru.yandex.bolts.collection.Tuple3;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusTicketStatus;
import ru.yandex.travel.bus.model.BusesTicket;
import ru.yandex.travel.commons.crypto.HashingUtils;
import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.BronevikHotelItinerary;
import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.hotels.common.orders.DolphinHotelItinerary;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderState;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.configurations.jdbc.TxScopeType;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderAggregateState;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.Payment;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.grpc.helpers.OrderCreator;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.grpc.helpers.TrainOrderHelper;
import ru.yandex.travel.orders.grpc.helpers.TxCallWrapper;
import ru.yandex.travel.orders.infrastructure.CallDescriptor;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TAddInsuranceReq;
import ru.yandex.travel.orders.proto.TAddInsuranceRsp;
import ru.yandex.travel.orders.proto.TAddServiceReq;
import ru.yandex.travel.orders.proto.TAddServiceRsp;
import ru.yandex.travel.orders.proto.TAeroflotState;
import ru.yandex.travel.orders.proto.TAuthorizeForOrderReq;
import ru.yandex.travel.orders.proto.TAuthorizeForOrderRsp;
import ru.yandex.travel.orders.proto.TCalculateRefundReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReqV2;
import ru.yandex.travel.orders.proto.TCalculateRefundRsp;
import ru.yandex.travel.orders.proto.TChangeTrainRegistrationStatusReq;
import ru.yandex.travel.orders.proto.TChangeTrainRegistrationStatusRsp;
import ru.yandex.travel.orders.proto.TCheckoutReq;
import ru.yandex.travel.orders.proto.TCheckoutRsp;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TEstimateDiscountReq;
import ru.yandex.travel.orders.proto.TEstimateDiscountRsp;
import ru.yandex.travel.orders.proto.TGenerateBusinessTripPdfReq;
import ru.yandex.travel.orders.proto.TGenerateBusinessTripPdfRsp;
import ru.yandex.travel.orders.proto.TGetAeroflotStateReq;
import ru.yandex.travel.orders.proto.TGetAeroflotStateRsp;
import ru.yandex.travel.orders.proto.TGetOrderAggregateStateBatchReq;
import ru.yandex.travel.orders.proto.TGetOrderAggregateStateBatchRsp;
import ru.yandex.travel.orders.proto.TGetOrderAggregateStateReq;
import ru.yandex.travel.orders.proto.TGetOrderAggregateStateRsp;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TGetOrdersInfoReq;
import ru.yandex.travel.orders.proto.TGetOrdersInfoRsp;
import ru.yandex.travel.orders.proto.TGetOrdersInfoWithoutExcludedReq;
import ru.yandex.travel.orders.proto.TGetOrdersInfoWithoutExcludedRsp;
import ru.yandex.travel.orders.proto.TGetUnauthorizedOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetUnauthorizedOrderInfoRsp;
import ru.yandex.travel.orders.proto.TListOrdersReq;
import ru.yandex.travel.orders.proto.TListOrdersRsp;
import ru.yandex.travel.orders.proto.TOrderAggregateState;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TPromoCodeApplicationResult;
import ru.yandex.travel.orders.proto.TRefundCalculation;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TReserveRsp;
import ru.yandex.travel.orders.proto.TStartCancellationReq;
import ru.yandex.travel.orders.proto.TStartCancellationRsp;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartPaymentRsp;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.proto.TStartRefundRsp;
import ru.yandex.travel.orders.proto.TUnauthorizedOrderInfo;
import ru.yandex.travel.orders.repository.OrderAggregateStateRepository;
import ru.yandex.travel.orders.repository.OrderItemRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.services.OrderInfoMapper;
import ru.yandex.travel.orders.services.PromoServiceHelper;
import ru.yandex.travel.orders.services.RefundCalculationService;
import ru.yandex.travel.orders.services.TokenEncrypter;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotOrderStateSync;
import ru.yandex.travel.orders.services.hotels.BusinessTripDocService;
import ru.yandex.travel.orders.services.orders.OrderAggregateStateMapper;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.payments.InvoiceFactory;
import ru.yandex.travel.orders.services.payments.schedule.PaymentBuilder;
import ru.yandex.travel.orders.services.promo.CodeApplicationResult;
import ru.yandex.travel.orders.services.promo.PromoCodeApplicationResult;
import ru.yandex.travel.orders.services.promo.PromoCodeApplicationService;
import ru.yandex.travel.orders.services.promo.PromoCodeCalculationRequest;
import ru.yandex.travel.orders.services.promo.ServiceDescription;
import ru.yandex.travel.orders.services.promo.UserOrderCounterService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.generic.proto.TGenericRefundToken;
import ru.yandex.travel.orders.workflow.order.generic.proto.TServiceRefundInfo;
import ru.yandex.travel.orders.workflow.order.proto.TCheckoutStart;
import ru.yandex.travel.orders.workflow.order.proto.TServiceAdded;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservation;
import ru.yandex.travel.orders.workflow.order.proto.TStartServiceRefund;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TBusRefundToken;
import ru.yandex.travel.orders.workflow.payments.proto.EPaymentState;
import ru.yandex.travel.orders.workflow.payments.proto.TPublish;
import ru.yandex.travel.orders.workflow.payments.proto.TStartPayment;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TAddInsurance;
import ru.yandex.travel.orders.workflow.train.proto.TRegistrationStatusChange;
import ru.yandex.travel.orders.workflow.train.proto.TStartReservationCancellation;
import ru.yandex.travel.orders.workflow.train.proto.TTrainRefundToken;
import ru.yandex.travel.orders.workflows.order.generic.GenericWorkflowService;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.workflow.ClientCallService;
import ru.yandex.travel.workflow.EWorkflowEventState;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static java.util.stream.Collectors.toList;
import static ru.yandex.travel.orders.infrastructure.CallDescriptor.CALL_ID;
import static ru.yandex.travel.orders.infrastructure.CallDescriptor.DEDUPLICATION_KEY;
import static ru.yandex.travel.orders.infrastructure.CallDescriptor.NO_CALL_ID;

@GrpcService(authenticateUser = true, authenticateService = true)
@Slf4j
@RequiredArgsConstructor
public class OrdersService extends OrderInterfaceV1Grpc.OrderInterfaceV1ImplBase {

    private final OrderRepository orderRepository;

    private final OrderItemRepository orderItemRepository;

    private final WorkflowRepository workflowRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final OrderCreator orderCreator;

    private final AuthorizationService authorizationService;

    private final RefundCalculationService refundCalculationService;

    private final TokenEncrypter tokenEncrypter;

    private final OrderInfoMapper orderInfoMapper;

    private final PromoCodeApplicationService promoCodeApplicationService;

    private final OrderAggregateStateRepository orderAggregateStateRepository;

    private final PaymentBuilder paymentBuilder;

    private final InvoiceFactory invoiceFactory;

    private final TxCallWrapper txCallWrapper;

    private final ClientCallService clientCallService;

    private final OrderAggregateStateMapper orderAggregateStateMapper;

    private final PromoServiceHelper promoServiceHelper;

    private final AeroflotOrderStateSync aeroflotOrderStateSync;

    private final UserOrderCounterService userOrderCounterService;

    private final GenericWorkflowService genericWorkflowService;

    private final BusinessTripDocService businessTripDocService;

    @Override
    public void changeTrainRegistrationStatus(TChangeTrainRegistrationStatusReq request,
                                              StreamObserver<TChangeTrainRegistrationStatusRsp> observer
    ) {
        CallDescriptor<TChangeTrainRegistrationStatusReq> callDescriptor = CallDescriptor.readWrite(request, CALL_ID);
        txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), callDescriptor, observer, log, req -> {
            log.info("Starting change registration status");

            Order order = getOrderByIdOrThrow(req.getOrderId());
            // TODO (syukhno) add processing UserActionToken
            if (order.isUserActionScheduled()) {
                Error.with(EErrorCode.EC_FAILED_PRECONDITION,
                        "Unable to change registration status: other action was scheduled").andThrow();
            }
            Error.checkState(OrderCompatibilityUtils.isConfirmed(order), "Invalid order state");
            List<Integer> validatedBlankIds = TrainOrderHelper.validateChangeTrainRegistrationStatusRequest(req, order);
            order.toggleUserActionScheduled(true);

            TRegistrationStatusChange event = TRegistrationStatusChange.newBuilder()
                    .setEnabled(req.getEnabled())
                    .addAllBlankIds(validatedBlankIds).build();
            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(), event);

            return TChangeTrainRegistrationStatusRsp.newBuilder().build();
        });
    }

    @Override
    public void createOrder(TCreateOrderReq request, StreamObserver<TCreateOrderRsp> observer) {
        UUID newOrderId = UUID.randomUUID();
        CallDescriptor<TCreateOrderReq> callDescriptor = CallDescriptor.readWrite(request, DEDUPLICATION_KEY);
        txCallWrapper.synchronouslyWithTxForOrder(newOrderId.toString(), callDescriptor, observer, log,
                req -> orderCreator.createOrderSync(req, newOrderId, getCredentials()));
    }

    private UserCredentials getCredentials() {
        UserCredentials credentials = UserCredentials.get();
        if (credentials == null) {
            throw Error.with(EErrorCode.EC_PERMISSION_DENIED, "credentials are not enabled").toEx();
        }
        return credentials;
    }

    // This method is not currently used and, possibly, will never be
    @Override
    public void addService(TAddServiceReq request, StreamObserver<TAddServiceRsp> observer) {
        CallDescriptor<TAddServiceReq> callDescriptor = CallDescriptor.readWrite(request, DEDUPLICATION_KEY);
        txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), callDescriptor, observer, log, req -> {
            log.info("Adding a new service to the order");
            Order order = getOrderByIdOrThrow(req.getOrderId());

            if (order.getPublicType() != EOrderType.OT_GENERIC) {
                Error.with(EErrorCode.EC_ABORTED, "Unsupported order type: " + order.getPublicType()).andThrow();
            }
            if (order.getEntityState() != EOrderState.OS_RESERVED) {
                Error.with(EErrorCode.EC_ABORTED, "Unsupported order state: " + order.getEntityState()).andThrow();
            }

            // forcing the entity update, it will prevent concurrent creation of multiple services
            order.setUserActionScheduled(true);

            TCreateServiceReq service = req.getService();

            // TODO: check service type here
            Error.checkArgument(false, "Adding service of type %s is not supported", service.getServiceType());
            // TODO handle postpay
            OrderItem item = OrderCreator.addOrderItem(order, service.getServiceType(), service.getSourcePayload(), false);
            item.setItemNumber(order.getOrderItems().stream()
                    .map(OrderItem::getItemNumber).max(Integer::compareTo).orElse(-1) + 1);
            // TODO: fix CPA PartnerOrderId change when order.items.size > 1. CpaOrderSnapshotService.mapBasicOrderInfo
            if (service.hasTrainTestContext()) {
                item.setTestContext(service.getTrainTestContext());
            }

            // storing (generating item id)
            orderItemRepository.save(item);
            Workflow itemWorkflow = Workflow.createWorkflowForEntity((WorkflowEntity<?>) item,
                    order.getWorkflow().getId());
            workflowRepository.save(itemWorkflow);
            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(),
                    TServiceAdded.newBuilder().setServiceId(item.getId().toString()).build());

            return TAddServiceRsp.newBuilder().setServiceId(item.getId().toString()).build();
        });
    }

    @Override
    public void getOrderInfo(TGetOrderInfoReq request, StreamObserver<TGetOrderInfoRsp> observer) {
        // for generic order there's some updated at OrderItemPayloadService#getActualizedPayload
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), observer, req -> {
            Order order;
            switch (req.getOneOfOrderIdsCase()) {
                case ORDERID:
                    order = getOrderByIdOrThrow(req.getOrderId(), true);
                    break;
                case PRETTYID:
                    order = getOrderByPrettyIdOrThrow(req.getPrettyId(), true);
                    break;
                case ONEOFORDERIDS_NOT_SET:
                default:
                    throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "One of order ids must be set").toEx();
            }
            try (var ignored = NestedMdc.forEntity(order)) {
                log.debug("Getting order info");
                TOrderInfo.Builder builder = orderInfoMapper.buildOrderInfoFor(order,
                        authorizationService.getOrderOwner(order.getId()), req.getUpdateOrderOnTheFly(), true,
                        req.getCalculateAggregateStateOnTheFly() ?
                                OrderInfoMapper.AggregateStateConstructMode.CALCULATE_ON_THE_FLY :
                                OrderInfoMapper.AggregateStateConstructMode.GET_FROM_DB
                );
                return TGetOrderInfoRsp.newBuilder().setResult(builder).build();
            }
        });
    }

    @Override
    public void getOrderAeroflotState(TGetAeroflotStateReq request, StreamObserver<TGetAeroflotStateRsp> observer) {
        CallDescriptor<TGetAeroflotStateReq> callDescriptor = CallDescriptor.readOnly(request);
        txCallWrapper.synchronouslyWithTx(callDescriptor, observer, log, req -> {
            var builder = TGetAeroflotStateRsp.newBuilder();
            var orderState = TAeroflotState.newBuilder()
                    .setOrderId(req.getOrderId());

            try {
                List<Order> dbOrders =
                        orderRepository.selectNotRemovedOrders(
                                Set.of((ProtoChecks.checkStringIsUuid("order id",
                                        req.getOrderId()))));

                if (!dbOrders.isEmpty()) {
                    var state = getAeroflotState(dbOrders.get(0));
                    orderState = orderState.setState(state);
                }
            } catch (Exception e) {
                log.error("", e);
            }
            builder.setOrderState(orderState.build());
            return builder.build();
        }, TxScopeType.READ_ONLY);
    }

    /**
     * Для заказа Аэрофлота идем в Аэрофлот и получаем актуальное состояние заказа, по состоянию купонов
     * определяем является ли заказ отмененным и если является то если был оплачен то OS_REFUNDED иначе OS_CANCELLED
     * (в базу не сохраняем т.к. в фоне работает AeroflotOrderRefreshService для этого)
     *
     * @param dbOrder
     */
    private EAeroflotOrderState getAeroflotState(Order dbOrder) {
        var result = aeroflotOrderStateSync.getAeroflotOrderStateUseCache((AeroflotOrder) dbOrder);
        EAeroflotOrderState state = ((AeroflotOrder) dbOrder).getState();
        if (result != null && state == EAeroflotOrderState.OS_CONFIRMED && dbOrder.getPublicType() == EOrderType.OT_AVIA_AEROFLOT) {
            var canCancel = AeroflotOrderStateSync.canCancel(result.getCouponStatusCodes());
            if (canCancel) {
                state = EAeroflotOrderState.OS_EXTERNALLY_CANCELLED;
            }
        }
        return state;
    }

    @Override
    public void getOrderAggregateState(TGetOrderAggregateStateReq request,
                                       StreamObserver<TGetOrderAggregateStateRsp> responseObserver) {
        CallDescriptor<TGetOrderAggregateStateReq> callDescriptor = CallDescriptor.readOnly(request);
        txCallWrapper.synchronouslyWithTx(callDescriptor, responseObserver, log, req -> {
            UUID orderId = ProtoChecks.checkStringIsUuid("order id", request.getOrderId());
            if (req.getCalculateAggregateStateOnTheFly()) {
                Order order = orderRepository.findById(orderId).orElseThrow(() ->
                        Status.Code.NOT_FOUND.toStatus().asRuntimeException()
                );
                TOrderAggregateState aggregateState = orderAggregateStateMapper.mapAggregateStateFromOrder(order);
                var result = aggregateState.toBuilder().setOrderId(order.getId().toString())
                        .setOrderPrettyId(order.getPrettyId());
                return TGetOrderAggregateStateRsp.newBuilder().setOrderAggregateState(result).build();
            } else {
                OrderAggregateState orderAggregateState = orderAggregateStateRepository.getOne(orderId);
                TOrderAggregateState.Builder builder =
                        orderInfoMapper.getOrderAggregateStateBuilder(orderAggregateState);
                return TGetOrderAggregateStateRsp.newBuilder().setOrderAggregateState(builder).build();
            }
        }, TxScopeType.READ_ONLY);
    }

    @Override
    public void getOrderAggregateStateBatch(TGetOrderAggregateStateBatchReq request,
                                            StreamObserver<TGetOrderAggregateStateBatchRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log, req -> {
            List<UUID> orderIds = new ArrayList<>();
            for (String strOrderId : request.getOrderIdList()) {
                orderIds.add(ProtoChecks.checkStringIsUuid("order id", strOrderId));
            }
            List<OrderAggregateState> orderAggregateStates = orderAggregateStateRepository.findAllById(orderIds);

            TGetOrderAggregateStateBatchRsp.Builder result = TGetOrderAggregateStateBatchRsp.newBuilder();
            for (OrderAggregateState st : orderAggregateStates) {
                TOrderAggregateState.Builder builder = orderInfoMapper.getOrderAggregateStateBuilder(st);
                result.addOrderAggregateState(builder);
            }
            return result.build();
        }, TxScopeType.READ_ONLY);
    }

    @Override
    public void reserve(TReserveReq request, StreamObserver<TReserveRsp> observer) {
        CallDescriptor<TReserveReq> callDescriptor = CallDescriptor.readWrite(request, CALL_ID);
        txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), callDescriptor, observer, log, req -> {
            log.info("Starting order reservation");
            Order order = getOrderByIdOrThrow(req.getOrderId());

            // this check with the state checks below make sure the operation isn't called multiple times,
            // the call descriptor will handle cases of repeated same call retries
            Error.checkState(!order.isUserActionScheduled(), "Unable to start reservation: some action in progress");

            EOrderType orderType = order.getPublicType();
            switch (orderType) {
                case OT_HOTEL_EXPEDIA:
                    Error.checkState(((HotelOrder) order).getEntityState() == EHotelOrderState.OS_NEW,
                            "Order can be reserved only in OS_NEW state");
                    break;
                case OT_AVIA_AEROFLOT:
                    Error.checkState(((AeroflotOrder) order).getEntityState() == EAeroflotOrderState.OS_NEW,
                            "Order can be reserved only in OS_NEW state");
                    break;
                case OT_TRAIN:
                    Error.checkState(((TrainOrder) order).getEntityState() == ETrainOrderState.OS_NEW,
                            "Order can be reserved only in OS_NEW state");
                    break;
                case OT_GENERIC:
                    Error.checkState(((GenericOrder) order).getEntityState() == EOrderState.OS_NEW,
                            "Order can be reserved only in OS_NEW state");
                    break;
                default:
                    throw new IllegalArgumentException("unsupported order type: " + orderType);
            }

            // both concurrent and repeated scheduling of multiple TStartReservation events prevention
            order.toggleUserActionScheduled(true);

            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(), TStartReservation.newBuilder().build());
            return TReserveRsp.newBuilder().build();
        });
    }

    @Override
    public void checkout(TCheckoutReq request, StreamObserver<TCheckoutRsp> observer) {
        CallDescriptor<TCheckoutReq> writePartCallDescriptor = CallDescriptor.readWrite(request, CALL_ID);
        // todo(tlg-13,mbobrov): temporary hotfix before TRAVELBACK-1491 is implemented, see the notes below
        if (true) {
            txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), writePartCallDescriptor, observer, log,
                    req -> {
                        log.info("Starting order checkout");
                        Order order = getOrderByIdOrThrow(req.getOrderId());
                        Error.checkArgument(order instanceof GenericOrder,
                                "Only generic orders are expected but got %s", order.getClass());
                        GenericOrder genericOrder = (GenericOrder) order;
                        if (!ensureCheckoutState(order)) {
                            Error.checkArgument(genericOrder.getState() == EOrderState.OS_RESERVED,
                                    "The order has to be in the OS_RESERVED state instead of %s",
                                    genericOrder.getState());
                            // copy-pasted from ReservedStateHandler.handleCheckout,
                            // temporarily used here to avoid busy loops below
                            log.info("The order has been completely formed, starting the payment process");
                            genericOrder.setState(EOrderState.OS_WAITING_PAYMENT);
                            genericOrder.getStateContext().setCheckedOut(true);
                            //genericOrder.setUserActionScheduled(false);
                        }
                        return TCheckoutRsp.newBuilder().build();
                    });
            return;
        } else {
            ProxyErrorObserver noRspObserver = new ProxyErrorObserver(observer);
            AtomicReference<CompletableFuture<EWorkflowEventState>> checkoutFuture = new AtomicReference<>();
            txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), writePartCallDescriptor, noRspObserver, log,
                    req -> {
                        log.info("Starting order checkout");
                        Order order = getOrderByIdOrThrow(req.getOrderId());
                        if (!ensureCheckoutState(order)) {
                            order.setUserActionScheduled(true);
                            checkoutFuture.set(
                                    workflowMessageSender.scheduleEventWithTracking(
                                            order.getWorkflow().getId(),
                                            TCheckoutStart.newBuilder().build()
                                    )
                            );
                        }
                        return null;
                    });
            if (noRspObserver.errorOccurred()) {
                // the first call has failed and the error has been sent to the client, no need to proceed
                return;
            }
            Error.checkState(checkoutFuture.get() != null, "Checkout future must be not null");
            // using two separate call ids to cache only the first part of this method separately
            checkoutFuture.get().whenComplete((r, t) -> {
                        CallDescriptor<TCheckoutReq> readPartCallDescriptor = CallDescriptor.readOnly(request);
                        if (r != null) {
                            synchronouslyWithoutTx(readPartCallDescriptor, observer,
                                    req -> TCheckoutRsp.getDefaultInstance());
                        }
                        if (t != null) {
                            synchronouslyWithoutTx(readPartCallDescriptor, observer, req -> {
                                throw new RuntimeException(t);
                            });
                        }
                    }
            );
        }
    }

    @Override
    public void startPayment(TStartPaymentReq request, StreamObserver<TStartPaymentRsp> observer) {
        CallDescriptor<TStartPaymentReq> callDescriptor = CallDescriptor.readWrite(request, CALL_ID);
        txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), callDescriptor, observer, log, req -> {
            Order order = getOrderByIdOrThrow(req.getOrderId());
            if (Strings.isNullOrEmpty(req.getInvoiceId())) {
                startOrderPayment(order, req);
            } else {
                Payment payment = order.getPayments()
                        .stream()
                        .filter(pi -> pi.getId().toString().equals(req.getInvoiceId()))
                        .findAny()
                        .orElseThrow(() -> Error.with(EErrorCode.EC_NOT_FOUND, "Invoice not found").toEx());
                order.setUpdatedAt(Instant.now());
                startPayment(payment, req);
            }
            return TStartPaymentRsp.newBuilder().build();

        });
    }

    private void startPayment(Payment payment, TStartPaymentReq req) {
        Error.checkState(payment.getState() == EPaymentState.PS_INVOICE_PENDING, "Illegal invoice state");
        log.info("Starting payment of invoice/schedule {}", payment.getId());

        final var startPaymentBuilder = TStartPayment.newBuilder()
                .setPaymentProvider(InvoiceFactory.paymentTypeFromInvoiceType(req.getInvoiceType()))
                .setReturnUrl(ProtoChecks.checkStringIsPresent("return url", req.getReturnUrl()))
                .setConfirmationReturnUrl(req.getConfirmationReturnUrl())
                .setSource(req.getSource())
                .setNewCommonPaymentWebForm(req.getNewCommonPaymentWebForm());
        switch (req.getPaymentTestContextsCase()) {
            case AVIAPAYMENTTESTCONTEXT:
                startPaymentBuilder.setAviaPaymentTestContext(req.getAviaPaymentTestContext());
                break;
            case PAYMENTTESTCONTEXT:
                startPaymentBuilder.setPaymentTestContext(req.getPaymentTestContext());
                break;
        }

        workflowMessageSender.scheduleEvent(payment.getWorkflow().getId(), startPaymentBuilder.build());
    }


    private void startOrderPayment(Order order, TStartPaymentReq req) {
        log.info("Starting order payment. Start payment request: {}", req);
        ensureStartPaymentState(order);

        // forcing the entity update, it will prevent concurrent creation of multiple invoices
        order.setUpdatedAt(Instant.now());

        if (req.getUseNewInvoiceModel()) {
            if (order.getActivePaymentWorkflowId() == null) {
                createPaymentForOrder(order);
                workflowMessageSender.scheduleEvent(order.getActivePaymentWorkflowId(), TPublish.newBuilder().build());
            }
            order.toggleUserActionScheduled(true);
            final TStartPayment.Builder startPaymentBuilder = TStartPayment.newBuilder()
                    .setPaymentProvider(InvoiceFactory.paymentTypeFromInvoiceType(req.getInvoiceType()))
                    .setReturnUrl(ProtoChecks.checkStringIsPresent("return url", req.getReturnUrl()))
                    .setConfirmationReturnUrl(req.getConfirmationReturnUrl())
                    .setSource(req.getSource())
                    .setNewCommonPaymentWebForm(req.getNewCommonPaymentWebForm());
            switch (req.getPaymentTestContextsCase()) {
                case AVIAPAYMENTTESTCONTEXT:
                    startPaymentBuilder.setAviaPaymentTestContext(req.getAviaPaymentTestContext());
                    break;
                case PAYMENTTESTCONTEXT:
                    startPaymentBuilder.setPaymentTestContext(req.getPaymentTestContext());
                    break;
            }
            workflowMessageSender.scheduleEvent(order.getActivePaymentWorkflowId(), startPaymentBuilder.build());
        } else {
            if (orderSupportsMultiplePaymentAttempts(order)) {
                // if we don't support multiple invoices but support multiple payment attempts we may have invoices in
                // payment not authorized state
                if (order.getCurrentInvoice() != null && order.getCurrentInvoice()
                        .getInvoiceState() != ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED) {
                    Error.with(EErrorCode.EC_ABORTED, "Cannot start payment: other invoices exist").andThrow();
                }
            } else {
                // TODO (mbobrov): think of hotel booking flow start payment
                if (!order.getInvoices().isEmpty()) {
                    Error.with(EErrorCode.EC_ABORTED, "Cannot start payment").andThrow();
                }
            }
            if (order.getUseDeferredPayment()) {
                Error.with(EErrorCode.EC_ABORTED,
                        "Orders with deferred payments may not be paid with old invoice model").andThrow();
            }
            Message paymentTestContext = null;
            switch (req.getPaymentTestContextsCase()) {
                case AVIAPAYMENTTESTCONTEXT:
                    paymentTestContext = req.getAviaPaymentTestContext();
                    break;
                case PAYMENTTESTCONTEXT:
                    paymentTestContext = req.getPaymentTestContext();
                    break;
            }
            invoiceFactory.createInvoice(order, req.getInvoiceType(), null,
                    req.getSource(), req.getReturnUrl(), req.getConfirmationReturnUrl(), null,
                    paymentTestContext,
                    req.getNewCommonPaymentWebForm()
            );
        }
    }

    private void createPaymentForOrder(Order order) {
        Payment payment;
        if (order.getUseDeferredPayment()) {
            log.info("Will create a deferred payment schedule");
            // need to unproxy the order for instanceof's to work for order items
            Order unproxiedOrder = (Order) Hibernate.unproxy(order);
            payment = paymentBuilder.buildDeferred(unproxiedOrder);
        } else {
            log.info("Will create an immediate pending invoice");
            payment = paymentBuilder.buildImmediate(order);
        }
        if (payment == null) {
            Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Deferred payment is not possible").andThrow();
        }
        //noinspection ConstantConditions
        order.setActivePaymentWorkflowId(payment.getWorkflow().getId());
    }

    @Override
    public void listOrders(TListOrdersReq request, StreamObserver<TListOrdersRsp> observer) {
        CallDescriptor<TListOrdersReq> callDescriptor = CallDescriptor.readOnly(request);
        txCallWrapper.synchronouslyWithTx(callDescriptor, observer, log, req -> {
            UserCredentials credentials = getCredentials();
            int offset = !req.hasPage() ? 0 : req.getPage().getOffset();
            int limit = !req.hasPage() ? 10 : req.getPage().getLimit();
            Error.checkArgument(req.getTypesCount() > 0, "Type must be provided");
            Error.checkArgument(limit > 0, "Limit must be greater than zero");
            Error.checkArgument(limit <= 50, "Limit must be <= 50 due to security reasons");


            Set<EDisplayOrderType> orderTypes;
            if (req.getTypesCount() == 0) {
                orderTypes = Set.of(EDisplayOrderType.DT_TRAIN, EDisplayOrderType.DT_HOTEL, EDisplayOrderType.DT_AVIA);
            } else {
                orderTypes = ImmutableSet.copyOf(req.getTypesList());
            }

            TListOrdersRsp.Builder builder = TListOrdersRsp.newBuilder();
            Set<EDisplayOrderState> orderStates;
            if (req.getDisplayOrderState() == EDisplayOrderState.OS_UNKNOWN) {
                if (req.getDisplayOrderStatesCount() != 0) {
                    orderStates = new HashSet<>(req.getDisplayOrderStatesList());
                } else {
                    orderStates = new HashSet<>(List.of(EDisplayOrderState.values()));
                    orderStates.remove(EDisplayOrderState.UNRECOGNIZED);
                    orderStates.remove(EDisplayOrderState.OS_UNKNOWN);
                    orderStates.remove(EDisplayOrderState.OS_CANCELLED);
                }
            } else {
                if (req.getDisplayOrderState() == EDisplayOrderState.OS_FULFILLED) {
                    orderStates = Set.of(EDisplayOrderState.OS_FULFILLED, EDisplayOrderState.OS_REFUNDED);
                } else {
                    orderStates = Set.of(req.getDisplayOrderState());
                }
            }

            List<TOrderInfo.Builder> orders;
            List<Order> dbOrders = orderRepository.findOrdersOwnedByUser(
                    credentials.getPassportId(),
                    orderTypes,
                    orderStates,
                    PageRequest.of(offset / limit, limit)); // expecting front to actually paginate

            Map<UUID, OrderAggregateState> aggregateStates = new HashMap<>();

            List<UUID> orderIds = dbOrders.stream().map(Order::getId).collect(Collectors.toUnmodifiableList());
            orderAggregateStateRepository.findAllById(orderIds).forEach(
                    oag -> aggregateStates.put(oag.getId(), oag));

            orders = dbOrders.stream().map(order -> {
                var bOrder = orderInfoMapper.buildOrderInfoFor(order, null, false,
                        false, OrderInfoMapper.AggregateStateConstructMode.BYPASS);
                if (aggregateStates.containsKey(order.getId())) {
                    bOrder.setOrderAggregateState(orderInfoMapper.getOrderAggregateStateBuilder(aggregateStates.get(order.getId())));
                }
                return bOrder;

            }).collect(toList());

            if (orders.size() > limit) {
                orders.remove(orders.size() - 1);
                builder.setHasMoreOrders(true);
            }

            orders.forEach(builder::addOrderInfo);
            builder.setPage(req.getPage());
            return builder.build();
        }, TxScopeType.READ_ONLY);
    }

    @Override
    public void getOrdersInfo(TGetOrdersInfoReq request, StreamObserver<TGetOrdersInfoRsp> observer) {
        CallDescriptor<TGetOrdersInfoReq> callDescriptor = CallDescriptor.readOnly(request);
        txCallWrapper.synchronouslyWithTx(callDescriptor, observer, log, req -> {
            var orderIds = req.getOrderIdsList()
                    .stream()
                    .map(orderId -> ProtoChecks.checkStringIsUuid("order id", orderId))
                    .collect(Collectors.toUnmodifiableSet());
            List<Order> dbOrders = orderRepository.selectNotRemovedOrders(orderIds);
            dbOrders.forEach(this::authorizeOrderForOrThrow);
            List<TOrderInfo.Builder> orders = mapDbOrdersToOrderBuilders(dbOrders);
            var builder = TGetOrdersInfoRsp.newBuilder();
            orders.forEach(builder::addResult);
            return builder.build();
        }, TxScopeType.READ_ONLY);
    }

    @Override
    public void getOrdersInfoWithoutExcluded(TGetOrdersInfoWithoutExcludedReq request,
                                             StreamObserver<TGetOrdersInfoWithoutExcludedRsp> observer) {
        CallDescriptor<TGetOrdersInfoWithoutExcludedReq> callDescriptor = CallDescriptor.readOnly(request);
        txCallWrapper.synchronouslyWithTx(callDescriptor, observer, log, req -> {
            int offset = !req.hasPage() ? 0 : req.getPage().getOffset();
            int limit = !req.hasPage() ? 10 : req.getPage().getLimit();
            Error.checkArgument(limit > 0, "Limit must be greater than zero");

            var excludedOrderIds = req.getExcludedOrderIdsList()
                    .stream()
                    .map(orderId -> ProtoChecks.checkStringIsUuid("order id", orderId))
                    .collect(Collectors.toUnmodifiableSet());
            Set<EDisplayOrderType> orderTypes;
            if (req.getTypesCount() == 0) {
                orderTypes = Set.of(EDisplayOrderType.DT_TRAIN, EDisplayOrderType.DT_HOTEL, EDisplayOrderType.DT_AVIA);
            } else {
                orderTypes = ImmutableSet.copyOf(req.getTypesList());
            }
            Set<EDisplayOrderState> orderStates;
            if (req.getDisplayOrderStatesCount() != 0) {
                orderStates = new HashSet<>(req.getDisplayOrderStatesList());
            } else {
                orderStates = new HashSet<>(List.of(EDisplayOrderState.values()));
                orderStates.remove(EDisplayOrderState.UNRECOGNIZED);
                orderStates.remove(EDisplayOrderState.OS_UNKNOWN);
                orderStates.remove(EDisplayOrderState.OS_CANCELLED);
            }

            List<Order> dbOrders = orderRepository.findOrdersOwnedByUserWithoutExcluded(
                    getCredentials().getPassportId(),
                    excludedOrderIds,
                    orderTypes,
                    orderStates,
                    PageRequest.of(offset / limit, limit));
            dbOrders.forEach(this::authorizeOrderForOrThrow);
            List<TOrderInfo.Builder> orders = mapDbOrdersToOrderBuilders(dbOrders);
            var builder = TGetOrdersInfoWithoutExcludedRsp.newBuilder();
            orders.forEach(builder::addResult);
            if (orders.size() > limit) {
                orders.remove(orders.size() - 1);
                builder.setHasMoreOrders(true);
            }
            return builder.build();
        }, TxScopeType.READ_ONLY);
    }

    private List<TOrderInfo.Builder> mapDbOrdersToOrderBuilders(List<Order> dbOrders) {
        Map<UUID, OrderAggregateState> aggregateStates = new HashMap<>();
        var dbOrderIds = dbOrders.stream().map(Order::getId).collect(Collectors.toUnmodifiableList());
        orderAggregateStateRepository.findAllById(dbOrderIds).forEach(oag -> aggregateStates.put(oag.getId(), oag));

        return dbOrders.stream().map(order -> {
            var bOrder = orderInfoMapper.buildOrderInfoFor(order, null, false,
                    false, OrderInfoMapper.AggregateStateConstructMode.BYPASS);
            if (aggregateStates.containsKey(order.getId())) {
                bOrder.setOrderAggregateState(orderInfoMapper.getOrderAggregateStateBuilder(aggregateStates.get(order.getId())));
            }
            return bOrder;
        }).collect(toList());
    }

    @Override
    public void calculateRefund(TCalculateRefundReq request, StreamObserver<TCalculateRefundRsp> observer) {
        CallDescriptor<TCalculateRefundReq> callDescriptor = CallDescriptor.readOnly(request);
        txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), callDescriptor, observer, log, req -> {
            log.info("Calculating refund");
            Order order = getOrderByIdOrThrow(req.getOrderId());
            TRefundCalculation refundCalculation = refundCalculationService.calculateRefund(order, req);
            String refundToken = tokenEncrypter.toRefundToken(refundCalculation);
            log.info("Finished calculating refund");
            return TCalculateRefundRsp.newBuilder()
                    .setRefundToken(refundToken)
                    .setRefundAmount(refundCalculation.getRefundAmount())
                    .setPenaltyAmount(refundCalculation.getPenaltyAmount())
                    .setExpiresAt(refundCalculation.getExpiresAt())
                    .build();
        });
    }

    @Override
    public void calculateRefundV2(TCalculateRefundReqV2 request, StreamObserver<TCalculateRefundRsp> observer) {
        CallDescriptor<TCalculateRefundReqV2> callDescriptor = CallDescriptor.readOnly(request);
        txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), callDescriptor, observer, log, req -> {
            log.info("Calculating refund v2");
            Order order = getOrderByIdOrThrow(req.getOrderId());
            Error.checkArgument(order instanceof GenericOrder,
                    "Generic order is expected but got %s", order.getClass().getName());
            GenericOrder genericOrder = (GenericOrder) order;
            Error.checkArgument(genericOrder.getState() == EOrderState.OS_CONFIRMED,
                    "Order must be in CONFIRMED state, but %s", genericOrder.getState().toString());
            TRefundCalculation refundCalculation = refundCalculationService.calculateRefundV2(genericOrder, req);
            String refundToken = tokenEncrypter.toRefundToken(refundCalculation);
            log.info("Finished calculating refund v2");
            return TCalculateRefundRsp.newBuilder()
                    .setRefundToken(refundToken)
                    .setRefundAmount(refundCalculation.getRefundAmount())
                    .setPenaltyAmount(refundCalculation.getPenaltyAmount())
                    .setExpiresAt(refundCalculation.getExpiresAt())
                    .build();
        });
    }

    @Override
    @SuppressWarnings("Duplicates")
    public void authorizeForOrder(TAuthorizeForOrderReq request, StreamObserver<TAuthorizeForOrderRsp> observer) {
        CallDescriptor<TAuthorizeForOrderReq> callDescriptor = CallDescriptor.readWrite(request, CALL_ID);
        txCallWrapper.synchronouslyWithTx(callDescriptor, observer, log, req -> {
            UserCredentials credentials = getCredentials();
            Order order;
            switch (req.getOneOfOrderIdsCase()) {
                case ORDERID:
                    order = getOrderByIdOrThrow(req.getOrderId(), false);
                    break;
                case PRETTYID:
                    order = getOrderByPrettyIdOrThrow(req.getPrettyId(), false);
                    break;
                case ONEOFORDERIDS_NOT_SET:
                default:
                    throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "One of order ids must be set").toEx();
            }
            try (var ignored = NestedMdc.forEntity(order)) {
                log.info("Requesting order authorization");
                boolean authorized = authorizationService.checkAndAuthorizeUserForOrder(order, credentials,
                        request.getSecret());
                return TAuthorizeForOrderRsp.newBuilder()
                        .setOrderId(order.getId().toString())
                        .setPrettyId(order.getPrettyId())
                        .setAuthorized(authorized)
                        .build();
            }
        });
    }

    @Override
    public void startRefund(TStartRefundReq request, StreamObserver<TStartRefundRsp> observer) {
        CallDescriptor<TStartRefundReq> callDescriptor = CallDescriptor.readWrite(request, CALL_ID);
        txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), callDescriptor, observer, log, req -> {
            log.info("Starting order refund");
            Order order = getOrderByIdOrThrow(req.getOrderId());
            if (!refundCalculationService.mayInitiateRefundForOrder(order)) {
                Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Unable to start refund").andThrow();
            }
            TRefundCalculation refundCalculation =
                    tokenEncrypter.fromRefundToken(ProtoChecks.checkStringIsPresent("refund token",
                            request.getRefundToken()));
            if (!order.getId().toString().equals(refundCalculation.getOrderId())) {
                throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Corrupted refund token").toEx();
            }
            if (ProtoUtils.toInstant(refundCalculation.getExpiresAt()).isBefore(Instant.now())) {
                Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Refund calculation expired").andThrow();
            }
            if (order.getPublicType() == EOrderType.OT_GENERIC) {
                TGenericRefundToken token = genericWorkflowService.getRefundToken(refundCalculation.getToken());
                for (var serviceRefundToken : token.getServiceList()) {
                    UUID serviceId = UUID.fromString(serviceRefundToken.getServiceId());
                    if (serviceRefundToken.getOneOfServiceRefundInfoCase() == TServiceRefundInfo.OneOfServiceRefundInfoCase.TRAINREFUNDTOKEN) {
                        TTrainRefundToken trainRefundToken = serviceRefundToken.getTrainRefundToken();
                        TrainOrderItem service = order.getOrderItems().stream().filter(x -> x.getId().equals(serviceId))
                                .map(s -> (TrainOrderItem) s).findFirst()
                                .orElseThrow(() -> Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Service not found").toEx());
                        for (var refundPassenger : trainRefundToken.getPassengerList()) {
                            TrainTicket ticket = service.getPayload().getPassengers().stream()
                                    .filter(p -> p.getCustomerId() == refundPassenger.getCustomerId())
                                    .map(TrainPassenger::getTicket).findFirst()
                                    .orElseThrow(() -> Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Passenger not " +
                                            "found").toEx());
                            if (ticket.getRefundStatus() != null) {
                                Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Refund has already started").andThrow();
                            }
                        }
                    } else if (serviceRefundToken.getOneOfServiceRefundInfoCase() == TServiceRefundInfo.OneOfServiceRefundInfoCase.BUSREFUNDTOKEN) {
                        TBusRefundToken busRefundToken = serviceRefundToken.getBusRefundToken();
                        BusOrderItem orderItem = order.getOrderItems().stream().filter(x -> x.getId().equals(serviceId))
                                .map(s -> (BusOrderItem) s).findFirst()
                                .orElseThrow(() -> Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Service not found").toEx());
                        Map<String, BusesTicket> ticketsById = orderItem.getPayload().getOrder().getTickets().stream()
                                .collect(Collectors.toMap(BusesTicket::getId, t -> t));
                        for (var refundTicket : busRefundToken.getTicketsList()) {
                            BusesTicket ticket = ticketsById.get(refundTicket.getTicketId());
                            if (ticket == null) {
                                Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Ticket not found").andThrow();
                            } else if (ticket.getStatus() != BusTicketStatus.SOLD) {
                                Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Ticket can not be refunded").andThrow();
                            }
                        }
                    }
                }
            }
            order.toggleUserActionScheduled(true);
            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(), TStartServiceRefund.newBuilder()
                    .setToken(refundCalculation.getToken())
                    .build());
            return TStartRefundRsp.newBuilder().build();
        });
    }

    @Override
    @SuppressWarnings("Duplicates")
    public void getUnauthorizedOrderInfo(TGetUnauthorizedOrderInfoReq request,
                                         StreamObserver<TGetUnauthorizedOrderInfoRsp> observer) {
        CallDescriptor<TGetUnauthorizedOrderInfoReq> callDescriptor = CallDescriptor.readOnly(request);
        txCallWrapper.synchronouslyWithTx(callDescriptor, observer, log, req -> {
            Order order;
            switch (req.getOneOfOrderIdsCase()) {
                case ORDERID:
                    order = getOrderByIdOrThrow(req.getOrderId(), false);
                    break;
                case PRETTYID:
                    order = getOrderByPrettyIdOrThrow(req.getPrettyId(), false);
                    break;
                case ONEOFORDERIDS_NOT_SET:
                default:
                    throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "One of order ids must be set").toEx();
            }
            try (var ignored = NestedMdc.forEntity(order)) {
                log.info("Getting unauthorized order info");
                var owner = authorizationService.getOrderOwner(order.getId());
                return TGetUnauthorizedOrderInfoRsp.newBuilder()
                        .setInfo(TUnauthorizedOrderInfo.newBuilder()
                                .setOrderId(order.getId().toString())
                                .setPrettyId(order.getPrettyId())
                                .setOrderType(order.getPublicType())
                                .setOwnerPassportId(Strings.nullToEmpty(owner.getPassportId()))
                                .setEmailHash(HashingUtils.hashEmail(order.getEmail()))
                                .setPhoneHash(HashingUtils.hashPhone(order.getPhone()))
                                .build())
                        .build();
            }
        });
    }

    @Override
    public void addInsurance(TAddInsuranceReq request, StreamObserver<TAddInsuranceRsp> observer) {
        CallDescriptor<TAddInsuranceReq> callDescriptor = CallDescriptor.readWrite(request, CALL_ID);
        txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), callDescriptor, observer, log, req -> {
            Order order = getOrderByIdOrThrow(request.getOrderId());
            Error.checkArgument(OrderCompatibilityUtils.isTrainOrder(order), "Invalid order type");
            Error.checkState(!order.isUserActionScheduled(), "Unable to add insurance: other action was scheduled");
            Error.checkState(order.getEntityState() == ETrainOrderState.OS_WAITING_PAYMENT ||
                    order.getEntityState() == EOrderState.OS_RESERVED, "Invalid order state");
            TrainOrderItem item = order.getOrderItems().stream()
                    .filter(x -> x.getPublicType() == EServiceType.PT_TRAIN)
                    .map(x -> (TrainOrderItem) x)
                    .filter(x -> !x.getPayload().isSlaveItem())
                    .findFirst().orElseThrow();
            var insuranceStatus = item.getPayload().getInsuranceStatus();
            Error.checkState(insuranceStatus == InsuranceStatus.PRICED,
                    "Unable to add insurance: insurance was not priced");
            order.toggleUserActionScheduled(true);
            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(), TAddInsurance.newBuilder().build());
            return TAddInsuranceRsp.newBuilder().build();
        });
    }

    @Override
    public void startCancellation(TStartCancellationReq request, StreamObserver<TStartCancellationRsp> observer) {
        CallDescriptor<TStartCancellationReq> callDescriptor = CallDescriptor.readWrite(request, CALL_ID);
        txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), callDescriptor, observer, log, req -> {
            log.info("Starting cancellation");
            Order order = getOrderByIdOrThrow(req.getOrderId());
            if (order.isUserActionScheduled()) {
                Error.with(EErrorCode.EC_FAILED_PRECONDITION,
                        "Unable to start cancellation: other action was scheduled").andThrow();
            }
            switch (order.getPublicType()) {
                case OT_TRAIN:
                    if (order.getEntityState() != ETrainOrderState.OS_WAITING_PAYMENT) {
                        Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Invalid order state").andThrow();
                    }
                    order.setUserActionScheduled(true);
                    workflowMessageSender.scheduleEvent(
                            order.getWorkflow().getId(), TStartReservationCancellation.newBuilder().build());
                    break;
                case OT_HOTEL_EXPEDIA:
                    if (order.getEntityState() != EHotelOrderState.OS_WAITING_PAYMENT) {
                        Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Invalid order state").andThrow();
                    }
                    if (order.getInvoices().stream().anyMatch(i -> i.getInvoiceState() != ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED)) {
                        Error.with(EErrorCode.EC_ABORTED, "Order has active payment(s)").andThrow();
                    }
                    order.setUserActionScheduled(true);
                    workflowMessageSender.scheduleEvent(
                            order.getWorkflow().getId(),
                            ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation.newBuilder()
                                    .setReason(CancellationDetails.Reason.USER_INTENTION.toString()).build()
                    );
                    break;
                case OT_GENERIC:
                    if (order.getEntityState() != EOrderState.OS_RESERVED &&
                            order.getEntityState() != EOrderState.OS_WAITING_PAYMENT) {
                        Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Invalid order state").andThrow();
                    }
                    order.setUserActionScheduled(true);
                    workflowMessageSender.scheduleEvent(
                            order.getWorkflow().getId(),
                            ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation.newBuilder()
                                    .setReason(CancellationDetails.Reason.USER_INTENTION.toString()).build()
                    );
                    break;
                default:
                    Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Invalid order type").andThrow();
                    break;
            }

            return TStartCancellationRsp.newBuilder().build();
        });
    }

    /**
     * Checks all the discounts for the requested service.
     * <p>
     * Calls promo service to check for plus points and other big promo actions. Checks the discount for the submitted
     * promo codes.
     *
     * @implNote promo code is applied for all services at once, while promo service is requested per each service
     * separately and currently works only for hotel orders.
     */
    @Override
    public void estimateDiscount(TEstimateDiscountReq request, StreamObserver<TEstimateDiscountRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTxAndThenAsync(request, responseObserver, log, rq -> {
                    List<ServiceDescription> serviceDescriptions = new ArrayList<>();
                    for (TEstimateDiscountReq.TEstimateDiscountService svc : request.getServiceList()) {
                        ServiceDescription sd = new ServiceDescription();
                        sd.setServiceType(svc.getServiceType());
                        switch (svc.getServiceType()) {
                            case PT_EXPEDIA_HOTEL:
                                ExpediaHotelItinerary expediaHotelItinerary =
                                        ProtoUtils.fromTJson(svc.getSourcePayload(),
                                                ExpediaHotelItinerary.class);
                                sd.setPayload(expediaHotelItinerary);
                                break;
                            case PT_DOLPHIN_HOTEL:
                                DolphinHotelItinerary dolphinItinerary = ProtoUtils.fromTJson(svc.getSourcePayload(),
                                        DolphinHotelItinerary.class);
                                sd.setPayload(dolphinItinerary);
                                break;
                            case PT_TRAVELLINE_HOTEL:
                                TravellineHotelItinerary tlItinerary = ProtoUtils.fromTJson(svc.getSourcePayload(),
                                        TravellineHotelItinerary.class);
                                sd.setPayload(tlItinerary);
                                break;
                            case PT_BNOVO_HOTEL:
                                BNovoHotelItinerary bNovoHotelItinerary = ProtoUtils.fromTJson(svc.getSourcePayload(),
                                        BNovoHotelItinerary.class);
                                sd.setPayload(bNovoHotelItinerary);
                                break;
                            case PT_BRONEVIK_HOTEL:
                                BronevikHotelItinerary bronevikHotelItinerary = ProtoUtils.fromTJson(svc.getSourcePayload(),
                                        BronevikHotelItinerary.class);
                                sd.setPayload(bronevikHotelItinerary);
                                break;
                            case PT_FLIGHT:
                                AeroflotServicePayload parsed =
                                        PayloadMapper.fromJson(svc.getSourcePayload().getValue(),
                                                AeroflotServicePayload.class);
                                sd.setPayload(parsed);
                                sd.setOriginalCost(parsed.getVariant().getOffer().getTotalPrice());
                                break;
                            case PT_TRAIN:
                                TrainReservation reserv = ProtoUtils.fromTJson(svc.getSourcePayload(),
                                        TrainReservation.class);
                                sd.setPayload(reserv);
                                // TODO (mbobrov): GET PRELIMINARY PRICE HERE
                                sd.setOriginalCost(Money.zero(ProtoCurrencyUnit.RUB));
                                break;
                            case PT_BUS:
                                BusReservation busReservation = ProtoUtils.fromTJson(svc.getSourcePayload(),
                                        BusReservation.class);
                                sd.setPayload(busReservation);
                                // TODO (mbobrov): GET PRELIMINARY PRICE HERE
                                sd.setOriginalCost(Money.zero(ProtoCurrencyUnit.RUB));
                                break;
                            default:
                                throw new IllegalArgumentException(
                                        String.format("Can't estimate discount for service type: %s",
                                                svc.getServiceType())
                                );
                        }
                        serviceDescriptions.add(sd);
                    }
                    PromoCodeCalculationRequest applicationRequest = PromoCodeCalculationRequest.builder()
                            .serviceDescriptions(serviceDescriptions)
                            .promoCodes(request.getPromoCodeList())
                            .build();
                    PromoCodeApplicationResult result =
                            promoCodeApplicationService.calculateResult(UserCredentials.get(), applicationRequest);

                    TEstimateDiscountRsp.Builder responseBuilder = TEstimateDiscountRsp.newBuilder();
                    responseBuilder.setTotalCost(ProtoUtils.toTPrice(result.getOriginalAmount()));
                    responseBuilder.setDiscountedCost(ProtoUtils.toTPrice(result.getDiscountedAmount()));
                    responseBuilder.setDiscountAmount(ProtoUtils.toTPrice(result.getDiscountAmount()));

                    for (CodeApplicationResult codeApplicationResult : result.getApplicationResults()) {
                        TPromoCodeApplicationResult.Builder codeResultBuilder =
                                TPromoCodeApplicationResult.newBuilder();
                        codeResultBuilder.setType(codeApplicationResult.getType().getProtoValue())
                                .setCode(codeApplicationResult.getCode());
                        if (codeApplicationResult.getDiscountAmount() != null) {
                            codeResultBuilder.setDiscountAmount(ProtoUtils.toTPrice(codeApplicationResult.getDiscountAmount()));
                        }
                        responseBuilder.addPromoCodeApplicationResults(codeResultBuilder);
                    }

                    return new Tuple3<>(responseBuilder, serviceDescriptions, result);
                },
                tuple3 ->
                        promoServiceHelper.determinePromosForOffers(
                                        tuple3.get2(),
                                        tuple3.get3(),
                                        userOrderCounterService.getUserExistingOrderTypes(UserCredentials.get().getPassportId()),
                                        request.getKVExperimentsList(),
                                        request.getWhiteLabelPartnerId()
                                )
                                .thenApply(
                                        promosForOffer -> tuple3.get1()
                                                .addAllPromosForOffer(promosForOffer)
                                                .build())
        );
    }

    private boolean orderSupportsMultiplePaymentAttempts(Order order) {
        return (order instanceof HotelOrder && ((HotelOrder) order).getPaymentRetryEnabled())
                || OrderCompatibilityUtils.isTrainOrder(order)
                || OrderCompatibilityUtils.isBusOrder(order);
    }

    private <ReqT extends Message, RspT extends Message> void synchronouslyWithoutTx(
            CallDescriptor<ReqT> callDescriptor,
            StreamObserver<RspT> observer,
            Function<ReqT, RspT> handler
    ) {
        ServerUtils.synchronously(log, callDescriptor.getRequest(), observer,
                rq -> {
                    String appCallId = callDescriptor.getCallId();
                    txCallWrapper.ensureOperationPreconditions(callDescriptor);
                    if (!Strings.isNullOrEmpty(appCallId) &&
                            callDescriptor.getCallType() == CallType.READ_WRITE) {
                        return clientCallService.computeIfAbsent(appCallId, () -> handler.apply(rq));
                    } else {
                        return handler.apply(rq);
                    }
                },
                ex -> GrpcExceptionHelper.mapStatusException(log, callDescriptor.getRequest(), ex)
        );
    }

    private Order getOrderByIdOrThrow(String protoOrderId) {
        return getOrderByIdOrThrow(protoOrderId, true);
    }

    private Order getOrderByIdOrThrow(String protoOrderId, boolean authorize) {
        UUID orderId = ProtoChecks.checkStringIsUuid("order id", protoOrderId);
        Optional<Order> order = orderRepository.findById(orderId);

        order = order.filter(o -> !o.nullSafeRemoved());

        if (order.isPresent()) {
            if (authorize) {
                authorizeOrderForOrThrow(order.get());
            }
            return order.get();
        } else {
            throw Error.with(EErrorCode.EC_NOT_FOUND, "No such order").withAttribute("order_id", orderId).toEx();
        }
    }

    private void authorizeOrderForOrThrow(Order order) {
        UserCredentials credentials = getCredentials();
        if (!authorizationService.checkAuthorizationFor(order.getId(), credentials.getSessionKey(),
                credentials.getPassportId())) {
            throw Error.with(EErrorCode.EC_PERMISSION_DENIED, "User is not authorized to access the order")
                    .withAttribute("order_id", order.getId())
                    .withAttribute("session_key", Strings.nullToEmpty(credentials.getSessionKey()))
                    .withAttribute("passport_id", Strings.nullToEmpty(credentials.getPassportId()))
                    .withAttribute("login", Strings.nullToEmpty(credentials.getLogin()))
                    .withAttribute("yandex_uid", credentials.getYandexUid())
                    .toEx();
        }
    }

    private Order getOrderByPrettyIdOrThrow(String protoPrettyId, boolean authorize) {
        Order order = orderRepository.getOrderByPrettyId(protoPrettyId);
        if (order == null) {
            throw Error.with(EErrorCode.EC_NOT_FOUND, "No such order").withAttribute("pretty_id", protoPrettyId).toEx();
        }
        if (order.nullSafeRemoved()) {
            throw Error.with(EErrorCode.EC_NOT_FOUND, "No such order").withAttribute("pretty_id", protoPrettyId).toEx();
        }
        if (authorize) {
            authorizeOrderForOrThrow(order);
        }
        return order;
    }

    private boolean ensureCheckoutState(Order order) {
        boolean isInIllegalState = order.isUserActionScheduled();
        boolean checkedOut = false;
        //noinspection SwitchStatementWithTooFewBranches
        switch (order.getPublicType()) {
            case OT_GENERIC:
                if (order.getEntityState() == EOrderState.OS_WAITING_PAYMENT) {
                    checkedOut = true;
                } else {
                    isInIllegalState |= order.getEntityState() != EOrderState.OS_RESERVED;
                }
                break;
            default:
                throw new IllegalArgumentException("Unsupported order type: " + order.getPublicType());
        }
        if (isInIllegalState) {
            throw Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Unexpected order state on checkout")
                    .withAttribute("order_id", order.getId())
                    .withAttribute("state", order.getEntityState().name())
                    .toEx();
        }
        return checkedOut;
    }

    private void ensureStartPaymentState(Order order) {
        List<Payment> payments = order.getPayments();
        boolean hasInvoicesInProgress =
                payments.stream().anyMatch(pi -> (pi.getState() == EPaymentState.PS_PAYMENT_IN_PROGRESS) ||
                        (pi.getState() == EPaymentState.PS_PARTIALLY_PAID &&
                                pi.getLastAttempt() != null &&
                                TrustInvoice.PENDING_INVOICE_STATES.contains(pi.getLastAttempt().getInvoiceState())
                        ));
        // TODO(ganintsev, tivelkov): кажется эта проверка перекрывается order.userActionScheduled=true
        if (hasInvoicesInProgress) {
            Error.with(EErrorCode.EC_ABORTED,
                    "Unable to start payment when some other invoices are in progress").andThrow();
        }
        EPaymentState paymentScheduleState = EPaymentState.PS_UNKNOWN;
        if (order.getPaymentSchedule() != null) {
            paymentScheduleState = order.getPaymentSchedule().getState();
        }
        Set<Enum<?>> allowedStates = new HashSet<>(2);
        switch (order.getPublicType()) {
            case OT_HOTEL_EXPEDIA:
                allowedStates.add(EHotelOrderState.OS_WAITING_EXTRA_PAYMENT);
                if (paymentScheduleState == EPaymentState.PS_PARTIALLY_PAID) {
                    allowedStates.add(EHotelOrderState.OS_CONFIRMED);
                } else {
                    allowedStates.add(EHotelOrderState.OS_WAITING_PAYMENT);
                }
                break;
            case OT_AVIA_AEROFLOT:
                allowedStates.add(EAeroflotOrderState.OS_WAIT_CARD_TOKENIZED);
                break;
            case OT_GENERIC:
                if (paymentScheduleState == EPaymentState.PS_PARTIALLY_PAID) {
                    allowedStates.add(EOrderState.OS_CONFIRMED);
                } else {
                    allowedStates.add(EOrderState.OS_WAITING_PAYMENT);
                }
                break;
            case OT_TRAIN:
                allowedStates.add(ETrainOrderState.OS_WAITING_PAYMENT);
                break;
            default:
                throw new IllegalArgumentException("Unsupported order type: " + order.getPublicType());
        }
        if (!allowedStates.contains(order.getEntityState())) {
            throw Error.with(EErrorCode.EC_FAILED_PRECONDITION, "Unexpected order state on startPayment")
                    .withAttribute("order_id", order.getId())
                    .withAttribute("state", order.getEntityState().name())
                    .withAttribute("payment_schedule_state", paymentScheduleState.name())
                    .toEx();
        }
    }

    @Override
    public void generateBusinessTripPdf(TGenerateBusinessTripPdfReq request,
                                        StreamObserver<TGenerateBusinessTripPdfRsp> observer) {
        CallDescriptor<TGenerateBusinessTripPdfReq> callDescriptor = CallDescriptor.readWrite(request, CALL_ID);
        txCallWrapper.synchronouslyWithTxForOrder(request.getOrderId(), callDescriptor, observer, log, req -> {
            Order order = getOrderByIdOrThrow(req.getOrderId());
            var canGenerate = businessTripDocService.canGenerateBusinessTripDoc(order);
            if (!canGenerate.isCanGenerate()) {
                throw Status.FAILED_PRECONDITION.withDescription(
                        String.format("Business trip document can not be generated: %s", canGenerate.getMessage())
                ).asRuntimeException();
            }
            businessTripDocService.maybeStartGenerateForOrder(order);

            return TGenerateBusinessTripPdfRsp.newBuilder().build();
        });
    }

    @RequiredArgsConstructor
    private static class ProxyErrorObserver implements StreamObserver<Message> {
        private final StreamObserver<?> target;
        private boolean errorOccurred = false;

        @Override
        public void onNext(Message value) {
            // ignoring the value
        }

        @Override
        public void onError(Throwable t) {
            target.onError(t);
            errorOccurred = true;
        }

        @Override
        public void onCompleted() {
            // ignoring too
        }

        public boolean errorOccurred() {
            return errorOccurred;
        }
    }
}
