package ru.yandex.travel.orders.grpc;

import java.util.UUID;

import com.google.common.base.Strings;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.configurations.jdbc.TxScope;
import ru.yandex.travel.orders.configurations.jdbc.TxScopeType;
import ru.yandex.travel.orders.entities.promo.PromoCode;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivation;
import ru.yandex.travel.orders.proto.PromoCodesUserInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TUserPromoAttributesReq;
import ru.yandex.travel.orders.proto.TUserPromoAttributesRsp;
import ru.yandex.travel.orders.repository.promo.PromoCodeActivationRepository;
import ru.yandex.travel.orders.repository.promo.PromoCodeRepository;
import ru.yandex.travel.orders.services.promo.PromoCodeChecker;
import ru.yandex.travel.orders.services.promo.PromoCodeUnifier;
import ru.yandex.travel.orders.services.promo.UserOrderCounterService;

@Slf4j
@RequiredArgsConstructor
@GrpcService(authenticateUser = true, authenticateService = true)
public class PromoCodeUserService extends PromoCodesUserInterfaceV1Grpc.PromoCodesUserInterfaceV1ImplBase {

    private final TransactionTemplate transactionTemplate;

    private final PromoCodeRepository promoCodeRepository;

    private final PromoCodeActivationRepository promoCodeActivationRepository;

    private final UserOrderCounterService userOrderCounterService;

    private final PromoCodeChecker checker;

    @Override
    public void userPromoAttributes(TUserPromoAttributesReq request,
                                    StreamObserver<TUserPromoAttributesRsp> responseObserver) {
        ServerUtils.synchronously(log, request, responseObserver,
                (req) -> {
                    Long passportId = UserCredentials.get().getParsedPassportIdOrNull();
                    boolean firstOrderPromoEligible = true;
                    if (passportId != null) {
                        firstOrderPromoEligible = TxScope.supplyWithTxScope(
                                TxScopeType.GRPC,
                                () -> transactionTemplate.execute(
                                        ignored -> userOrderCounterService.getNumberOfHotelOrdersConfirmedByUser(passportId) == 0
                                )
                        );
                    }
                    return TUserPromoAttributesRsp.newBuilder()
                            .setFirstOrderPromoEligible(firstOrderPromoEligible)
                            .build();
                },
                ex -> GrpcExceptionHelper.mapStatusException(log, request, ex));
    }

    public UUID activatePromoCode(String codeFromRequest) {

        UserCredentials uc = UserCredentials.get();
        Error.checkArgument(!Strings.isNullOrEmpty(uc.getPassportId()), "Cannot activate promo code without " +
                "passport id");

        String code = PromoCodeUnifier.unifyCode(codeFromRequest);
        try {
            return TxScope.supplyWithTxScope(TxScopeType.GRPC, () -> {
                return transactionTemplate.execute(ignored -> {
                    PromoCode promoCode = promoCodeRepository.findByCodeEquals(code);
                    if (promoCode == null) {
                        throw Error.with(
                                EErrorCode.EC_NOT_FOUND, String.format("Promo code %s not found", code)
                        ).toEx();
                    }
                    PromoCodeActivation activation =
                            promoCodeActivationRepository.lookupActivationForPromoCodeAndPassportId(
                                    uc.getPassportId(), promoCode.getId()
                            );
                    UUID activationId = null;
                    if (activation == null) {
                        checker.ensurePromoCodeCanBeActivated(promoCode);
                        PromoCodeActivation promoCodeActivation = PromoCodeActivation.activate(uc.getPassportId(),
                                promoCode);
                        promoCodeActivationRepository.save(promoCodeActivation);
                        activationId = promoCodeActivation.getId();
                    } else {
                        activationId = activation.getId();
                    }
                    return activationId;
                });
            });
        } catch (DataIntegrityViolationException ex) {
            log.info("Promo code {} already activated. Rolling back tx in order to try and find existing activation",
                    code, ex);
        }

        return TxScope.supplyWithTxScope(TxScopeType.GRPC, () -> {
            return transactionTemplate.execute(ignored -> {
                PromoCode promoCode = promoCodeRepository.findByCodeEquals(code);
                if (promoCode == null) {
                    throw Error.with(EErrorCode.EC_NOT_FOUND, "Promo code %s not found").toEx();
                }
                PromoCodeActivation activation =
                        promoCodeActivationRepository.lookupActivationForPromoCodeAndPassportId(
                                uc.getPassportId(), promoCode.getId()
                        );
                Error.checkState(activation != null, "Activation must be present, but not found");
                return activation.getId();
            });
        });
    }
}
