package ru.yandex.travel.orders.grpc;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.EnumSet;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;

import javax.annotation.PostConstruct;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;

import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.entities.promo.BasePromoCodeGenerationConfig;
import ru.yandex.travel.orders.entities.promo.DiscountApplicationConfig;
import ru.yandex.travel.orders.entities.promo.HotelRestriction;
import ru.yandex.travel.orders.entities.promo.PromoAction;
import ru.yandex.travel.orders.entities.promo.PromoCode;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivation;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivationsStrategy;
import ru.yandex.travel.orders.entities.promo.PromoCodeGenerationType;
import ru.yandex.travel.orders.entities.promo.SimplePromoCodeGenerationConfig;
import ru.yandex.travel.orders.entities.promo.UserTypeRestriction;
import ru.yandex.travel.orders.entities.promo.ValidTillGenerationType;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.grpc.helpers.TxCallWrapper;
import ru.yandex.travel.orders.infrastructure.CallDescriptor;
import ru.yandex.travel.orders.repository.promo.PromoActionRepository;
import ru.yandex.travel.orders.repository.promo.PromoCodeActivationRepository;
import ru.yandex.travel.orders.repository.promo.PromoCodeRepository;
import ru.yandex.travel.orders.services.promo.ApplicationResultType;
import ru.yandex.travel.orders.services.promo.PromoCodeChecker;
import ru.yandex.travel.orders.services.promo.PromoCodeGenerationService;
import ru.yandex.travel.orders.services.promo.PromoCodeUnifier;
import ru.yandex.travel.orders.services.promo.PromoProperties;
import ru.yandex.travel.orders.services.promo.proto.EPromoCodeGenerationType;
import ru.yandex.travel.orders.services.promo.proto.PromoCodesOperatorManagementInterfaceV1Grpc;
import ru.yandex.travel.orders.services.promo.proto.TBlacklistPromoCodeReq;
import ru.yandex.travel.orders.services.promo.proto.TBlacklistPromoCodeRsp;
import ru.yandex.travel.orders.services.promo.proto.TCreatePromoActionReq;
import ru.yandex.travel.orders.services.promo.proto.TCreatePromoActionRsp;
import ru.yandex.travel.orders.services.promo.proto.TCreatePromoCodeReq;
import ru.yandex.travel.orders.services.promo.proto.TCreatePromoCodeRsp;
import ru.yandex.travel.orders.services.promo.proto.TCreatePromoCodesBatchReq;
import ru.yandex.travel.orders.services.promo.proto.TCreatePromoCodesBatchRsp;
import ru.yandex.travel.orders.services.promo.proto.TDiscountApplicationConfig;
import ru.yandex.travel.orders.services.promo.proto.TGenerationConfig;
import ru.yandex.travel.orders.services.promo.proto.TGetPromoActionReq;
import ru.yandex.travel.orders.services.promo.proto.TGetPromoActionResp;
import ru.yandex.travel.orders.services.promo.proto.TGetPromoCodeActivationsCountReq;
import ru.yandex.travel.orders.services.promo.proto.TGetPromoCodeActivationsCountRsp;
import ru.yandex.travel.orders.services.promo.proto.THotelRestriction;
import ru.yandex.travel.orders.services.promo.proto.TPromoCodeActivationAvailableReq;
import ru.yandex.travel.orders.services.promo.proto.TPromoCodeActivationAvailableResp;
import ru.yandex.travel.orders.services.promo.proto.TPromoCodeInfo;
import ru.yandex.travel.orders.services.promo.proto.TResetPromoCodeActivationReq;
import ru.yandex.travel.orders.services.promo.proto.TResetPromoCodeActivationRsp;

import static ru.yandex.travel.orders.infrastructure.CallDescriptor.NO_CALL_ID;

@GrpcService(authenticateService = true)
@EnableConfigurationProperties(PromoProperties.class)
@RequiredArgsConstructor
@Slf4j
public class PromoCodesOperatorManagementService extends PromoCodesOperatorManagementInterfaceV1Grpc.PromoCodesOperatorManagementInterfaceV1ImplBase {

    private final PromoCodeGenerationService promoCodeGenerationService;

    private final PromoActionRepository promoActionRepository;

    private final PromoCodeRepository promoCodeRepository;

    private final PromoCodeChecker promoCodeChecker;

    private final PromoCodeActivationRepository promoCodeActivationRepository;

    private final TxCallWrapper txCallWrapper;

    private final PromoProperties properties;

    private LoadingCache<UUID, TGetPromoActionResp> promoActionDetailsCache;

    @PostConstruct
    public void setUpCache() {
        var loader = new CacheLoader<UUID, TGetPromoActionResp>() {
            @Override
            public TGetPromoActionResp load(UUID key) {
                return getPromoActionDetails(key);
            }
        };
        promoActionDetailsCache = CacheBuilder.from(properties.getActionsCacheConfig()).build(loader);
    }

    @Override
    public void createPromoAction(TCreatePromoActionReq request,
                                  StreamObserver<TCreatePromoActionRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, req -> {
            PromoAction promoAction = new PromoAction();
            promoAction.setId(UUID.randomUUID());
            promoAction.setName(req.getActionName());
            if (req.hasActionValidTill()) {
                promoAction.setValidTill(ProtoUtils.toInstant(req.getActionValidTill()));
            }
            if (req.hasActionValidFrom()) {
                promoAction.setValidFrom(ProtoUtils.toInstant(req.getActionValidFrom()));
            }
            promoAction.setDiscountApplicationConfig(getDiscountApplicationConfig(req));

            if (req.getGenerationType() == EPromoCodeGenerationType.PGT_SIMPLE) {
                promoAction.setPromoCodeGenerationType(PromoCodeGenerationType.SIMPLE_GENERATION);

                var generationConfig = new SimplePromoCodeGenerationConfig();
                generationConfig.setPrefix(req.getPrefix());
                generationConfig.setSuffix(req.getSuffix());

                if (req.getNominal() > 0) {
                    generationConfig.setNominal(req.getNominal());
                }
                generationConfig.setNominalType(req.getNominalType());
                generationConfig.setMaxUsagePerUser(req.getMaxUsageCount());
                generationConfig.setMaxActivations(req.getMaxActivations());

                if (req.getFixedDaysDuration() > 0) {
                    generationConfig.setValidTillGenerationType(ValidTillGenerationType.FIXED_DURATION);
                    generationConfig.setFixedDaysDuration(req.getFixedDaysDuration());
                } else {
                    generationConfig.setValidTillGenerationType(ValidTillGenerationType.FIXED_DATE);
                    generationConfig.setFixedDate(ProtoUtils.toLocalDate(req.getActionValidTill()));
                }

                promoAction.setPromoCodeGenerationConfig(generationConfig);
            }

            if (req.getBudget() > 0) {
                promoAction.setInitialBudget(BigDecimal.valueOf(req.getBudget()));
                promoAction.setRemainingBudget(BigDecimal.valueOf(req.getBudget()));
            }

            promoActionRepository.save(promoAction);
            return TCreatePromoActionRsp.newBuilder()
                    .setId(promoAction.getId().toString())
                    .setName(promoAction.getName())
                    .build();
        });
    }

    private DiscountApplicationConfig getDiscountApplicationConfig(TCreatePromoActionReq req) {
        DiscountApplicationConfig discountApplicationConfig = new DiscountApplicationConfig();
        if (req.hasMinTotalCost()) {
            discountApplicationConfig.setMinTotalCost(ProtoUtils.fromTPrice(req.getMinTotalCost()));
        }
        discountApplicationConfig.setAddsUpWithOtherActions(req.getAddsUpWithOtherActions());
        if (req.hasMaxConfirmedHotelOrders()) {
            if (req.getMaxConfirmedHotelOrders().getValue() > 1) {
                throw new IllegalArgumentException("expected the max confirmed hotel orders to have value 0 or 1");
            }
            discountApplicationConfig.setMaxConfirmedHotelOrders(
                    req.getMaxConfirmedHotelOrders().getValue());
        }
        if (req.getFirstOrderOnlyForCount() > 0) {
            discountApplicationConfig
                    .setFirstOrderOnlyFor(EnumSet.copyOf(req.getFirstOrderOnlyForList()));
        }

        switch (req.getUserTypeRestriction()) {
            case UTR_PLUS_ONLY:
                discountApplicationConfig.setUserTypeRestriction(UserTypeRestriction.PLUS_ONLY);
                break;
            case UTR_STAFF_ONLY:
                discountApplicationConfig.setUserTypeRestriction(UserTypeRestriction.STAFF_ONLY);
                break;
        }
        if (req.getHotelRestrictionsCount() > 0) {
            discountApplicationConfig.setHotelRestrictions(mapProtoHotelRestrictions(req.getHotelRestrictionsList()));
        }
        if (req.hasMaxNominalDiscount()) {
            discountApplicationConfig.setMaxNominalDiscount(ProtoUtils.fromTPrice(req.getMaxNominalDiscount()));
        }
        return discountApplicationConfig;
    }

    private List<HotelRestriction> mapProtoHotelRestrictions(List<THotelRestriction> protoHotelRestrictions) {
        return protoHotelRestrictions.stream().map(rs->{
            HotelRestriction hrs = new HotelRestriction();
            hrs.setPartner(rs.getServiceType());
            hrs.setOriginalId(rs.getOriginalId());
            return hrs;
        }).collect(Collectors.toList());
    }

    @Override
    public void createPromoCode(TCreatePromoCodeReq request, StreamObserver<TCreatePromoCodeRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            PromoCode promoCode = new PromoCode();
            promoCode.setId(UUID.randomUUID());
            promoCode.setPromoAction(
                    promoActionRepository.getOne(
                            ProtoChecks.checkStringIsUuid("promo code id", req.getPromoActionId())
                    )
            );
            promoCode.setCode(PromoCodeUnifier.unifyCode(req.getCode()));
            promoCode.setNominal(BigDecimal.valueOf(req.getNominal()));
            promoCode.setNominalType(req.getNominalType());
            Error.checkState(req.getMaxUsageCount() >= 1,
                    "Usage count with value less than 1 doesn't make sense (you definitely meant to put 1 there)");
            promoCode.setAllowedUsageCount(req.getMaxUsageCount());
            promoCode.setAllowedActivationsCount(0);

            if (req.getMaxActivations() > 0) {
                promoCode.setActivationsStrategy(PromoCodeActivationsStrategy.LIMITED_ACTIVATIONS);
                promoCode.setAllowedActivationsTotal(req.getMaxActivations());
            } else {
                promoCode.setActivationsStrategy(PromoCodeActivationsStrategy.UNLIMITED_ACTIVATIONS);
            }

            if (req.hasValidFrom()) {
                promoCode.setValidFrom(ProtoUtils.toInstant(req.getValidFrom()));
            }
            if (req.hasValidTill()) {
                promoCode.setValidTill(ProtoUtils.toInstant(req.getValidTill()));
            }

            if (req.getPassportIdCount() > 0) {
                Error.checkState(req.getPassportIdCount() <= req.getMaxActivations(),
                        "Too much passport Ids for available activations");
                req.getPassportIdList().forEach(passportId -> {
                    // this code probably should never work as we create a new promo code in the method
                    PromoCodeActivation promoCodeActivation = promoCodeActivationRepository.lookupActivationByCodeAndPassportId(
                            passportId, req.getCode()
                    );
                    Error.checkState(promoCodeActivation == null, "Promo code %s has already been activated for user " +
                            "%s", promoCode.getCode(), passportId);
                    PromoCodeActivation newActivation = PromoCodeActivation.activate(passportId, promoCode);
                    promoCodeActivationRepository.save(newActivation);
                });
            }
            promoCodeRepository.save(promoCode);
            return TCreatePromoCodeRsp.newBuilder().setId(promoCode.getId().toString()).build();
        });
    }

    @Override
    public void resetPromoCodeActivation(TResetPromoCodeActivationReq request,
                                         StreamObserver<TResetPromoCodeActivationRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            Error.checkArgument(req.getPassportId().isEmpty(), "Passport id must be provided");
            Error.checkArgument(req.getCode().isEmpty(), "Promo code must be provided");
            PromoCodeActivation promoCodeActivation = promoCodeActivationRepository.lookupActivationByCodeAndPassportId(
                    req.getPassportId(), req.getCode()
            );
            Error.checkState(promoCodeActivation != null,
                    "Promo code activation for passportId %s and code %s not found",
                    req.getPassportId(), req.getCode());

            promoCodeActivation.setTimesUsed(0);

            return TResetPromoCodeActivationRsp.newBuilder()
                    .setPromoCodeActivationId(promoCodeActivation.getId().toString()).build();
        });
    }

    @Override
    public void createPromoCodesBatch(TCreatePromoCodesBatchReq request,
                                      StreamObserver<TCreatePromoCodesBatchRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            var batchSize = req.getBatchSize() == 0 ? 1 : req.getBatchSize();
            PromoAction promoAction;
            if (!req.getPromoActionId().isEmpty()) {
                promoAction = promoActionRepository.getOne(
                        ProtoChecks.checkStringIsUuid("promo action id", req.getPromoActionId())
                );
            } else if (!req.getPromoActionName().isEmpty()) {
                promoAction =
                        promoActionRepository.findByName(req.getPromoActionName())
                                .orElseThrow(() ->
                                        Error.with(EErrorCode.EC_NOT_FOUND,
                                                "Not found promo action with given name: " + req.getPromoActionName())
                                                .toEx());
            } else {
                throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Promo action ID or name must be provided").toEx();
            }
            LocalDate forDate = LocalDate.now();
            if (req.hasForDate()) {
                forDate = ProtoUtils.toLocalDate(req.getForDate());
            }
            TCreatePromoCodesBatchRsp.Builder rspBuilder = TCreatePromoCodesBatchRsp.newBuilder();
            for (var i = 0; i < batchSize; i++) {
                PromoCode code = promoCodeGenerationService.generatePromoCodeForAction(promoAction, forDate);
                rspBuilder.addPromoCode(TPromoCodeInfo.newBuilder()
                        .setId(code.getId().toString())
                        .setCode(code.getCode()));
            }
            return rspBuilder.build();
        });
    }

    @Override
    public void promoCodeActivationAvailable(TPromoCodeActivationAvailableReq request,
                                             StreamObserver<TPromoCodeActivationAvailableResp> responseObserver) {
        ServerUtils.synchronously(log, request, responseObserver,
                (req) -> {
                    String code = PromoCodeUnifier.unifyCode(request.getCode());
                    ApplicationResultType checkResult =
                            promoCodeChecker.checkPromoCode(promoCodeRepository.findByCodeEquals(code));
                    return TPromoCodeActivationAvailableResp.newBuilder()
                            .setResult(checkResult.getProtoValue())
                            .setErrorMessage(checkResult.getErrorMessage(code))
                            .build();
                },
                ex -> GrpcExceptionHelper.mapStatusException(log, request, ex));
    }

    @Override
    public void getPromoActionDetails(TGetPromoActionReq request,
                                      StreamObserver<TGetPromoActionResp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            UUID uuid = ProtoChecks.checkStringIsUuid("promo code id", req.getPromoActionId());
            try {
                return promoActionDetailsCache.get(uuid);
            } catch (ExecutionException e) {
                log.error("Error getting promo action details {}", uuid, e);
                throw new RuntimeException(e);
            }
        });
    }

    @VisibleForTesting
    TGetPromoActionResp getPromoActionDetails(UUID uuid) {
        PromoAction action = promoActionRepository.getOne(uuid);
        return toTGetPromoActionResp(action);
    }

    @Override
    public void getPromoCodeActivationsCount(TGetPromoCodeActivationsCountReq request,
                                             StreamObserver<TGetPromoCodeActivationsCountRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            PromoCode promoCode = getPromoCodeByCodeOrId(req.getCode(), req.getCodeId());

            Integer totalActivationsCount = promoCodeActivationRepository.countActivationsByPromoCode(promoCode.getId());

            return TGetPromoCodeActivationsCountRsp.newBuilder()
                    .setTotalCount(totalActivationsCount)
                    .build();
        });
    }

    @Override
    public void blacklistPromoCode(TBlacklistPromoCodeReq request,
                                   StreamObserver<TBlacklistPromoCodeRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, req -> {
            PromoCode promoCode = getPromoCodeByCodeOrId(req.getCode(), req.getCodeId());

            Error.checkArgument(!promoCode.isBlacklisted(), "Promo code is already blacklisted");
            promoCode.setBlacklisted(true);
            promoCodeRepository.save(promoCode);

            return TBlacklistPromoCodeRsp.newBuilder().build();
        });
    }

    private TGetPromoActionResp toTGetPromoActionResp(PromoAction action) {
        var response = TGetPromoActionResp.newBuilder()
                .setActionName(action.getName())
                .setPromoCodeGenerationType(PromoCodeGenerationType.mapToProto(action.getPromoCodeGenerationType()));
        if (action.getValidTill() != null) {
            response.setActionValidTill(ProtoUtils.fromInstant(action.getValidTill()));
        }
        if (action.getValidFrom() != null) {
            response.setActionValidFrom(ProtoUtils.fromInstant(action.getValidFrom()));
        }

        setDiscountApplicationConfig(response, action.getDiscountApplicationConfig());

        setGenerationConfig(response, action.getPromoCodeGenerationConfig());

        return response.build();
    }

    private void setDiscountApplicationConfig(TGetPromoActionResp.Builder response,
                                              DiscountApplicationConfig discountEntity) {
        if (discountEntity != null) {
            var discountProto = TDiscountApplicationConfig.newBuilder();
            if (discountEntity.getMinTotalCost() != null) {
                discountProto.setMinTotalCost(ProtoUtils.toTPrice(discountEntity.getMinTotalCost()));
            }
            discountProto.setUserTypeRestriction(UserTypeRestriction.mapToProto(discountEntity.getUserTypeRestriction()))
                    .setAddsUpWithOtherActions(discountEntity.isAddsUpWithOtherActions());
            if (discountEntity.getMaxConfirmedHotelOrders() != null) {
                discountProto.setMaxConfirmedHotelOrders(discountEntity.getMaxConfirmedHotelOrders());
            }
            if (discountEntity.getHotelRestrictions() != null) {
                discountProto
                        .addAllHotelRestrictions(
                                discountEntity.getHotelRestrictions().stream().map(
                                        restriction -> THotelRestriction.newBuilder()
                                                .setOriginalId(restriction.getOriginalId())
                                                .setServiceType(restriction.getPartner())
                                                .build()
                                ).collect(Collectors.toList())
                        );
            }
            discountProto.addAllFirstOrderOnlyFor(discountEntity.getFirstOrderOnlyFor());

            response.setDiscountApplicationConfig(discountProto);
        }
    }

    private void setGenerationConfig(TGetPromoActionResp.Builder response,
                                     BasePromoCodeGenerationConfig generationEntity) {
        if (generationEntity != null) {
            var generationProto = TGenerationConfig.newBuilder()
                    .setNominal(generationEntity.getNominal())
                    .setNominalType(generationEntity.getNominalType())
                    .setMaxUsagePerUser(generationEntity.getMaxUsagePerUser())
                    .setMaxActivations(generationEntity.getMaxActivations())
                    .setValidTillGenerationType(
                            ValidTillGenerationType.mapToProto(generationEntity.getValidTillGenerationType()));
            if (generationEntity.getFixedDate() != null) {
                generationProto.setFixedDate(ProtoUtils.fromLocalDate(generationEntity.getFixedDate()));
            }
            if (generationEntity.getFixedDaysDuration() != null) {
                generationProto.setFixedDaysDuration(generationEntity.getFixedDaysDuration());
            }

            response.setGenerationConfig(generationProto);
        }
    }

    private PromoCode getPromoCodeByCodeOrId(String code, String id) {
        PromoCode promoCode;
        if (!Strings.isNullOrEmpty(code)) {
            promoCode = promoCodeRepository.findByCodeEquals(PromoCodeUnifier.unifyCode(code));
        } else if (!Strings.isNullOrEmpty(id)) {
            promoCode = promoCodeRepository.getOne(ProtoChecks.checkStringIsUuid("promo code id", id));
        } else {
            throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Promo code id or code must be provided").toEx();
        }

        if (promoCode == null) {
            throw Error.with(EErrorCode.EC_NOT_FOUND, "Promo code not found").toEx();
        }
        return promoCode;
    }
}
