package ru.yandex.travel.orders.grpc;

import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.util.UUID;

import com.google.common.base.Strings;
import com.google.protobuf.ByteString;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.grpc.helpers.TxCallWrapper;
import ru.yandex.travel.orders.infrastructure.CallDescriptor;
import ru.yandex.travel.orders.proto.ReportDevInterfaceDoNotUseGrpc;
import ru.yandex.travel.orders.proto.TDevCreatePartnerOrdersReportReq;
import ru.yandex.travel.orders.proto.TDevCreatePartnerOrdersReportRsp;
import ru.yandex.travel.orders.proto.TDevCreatePartnerPaymentOrderReportReq;
import ru.yandex.travel.orders.proto.TDevCreatePartnerPaymentOrderReportRsp;
import ru.yandex.travel.orders.proto.TDevCreatePartnerPayoutReportReq;
import ru.yandex.travel.orders.proto.TDevCreatePartnerPayoutReportRsp;
import ru.yandex.travel.orders.proto.TDevPlanPartnerReportsSendingReq;
import ru.yandex.travel.orders.proto.TDevPlanPartnerReportsSendingRsp;
import ru.yandex.travel.orders.proto.TDevSendPartnerPaymentOrderReportReq;
import ru.yandex.travel.orders.proto.TDevSendPartnerPaymentOrderReportRsp;
import ru.yandex.travel.orders.proto.TDevSendPartnerReportsReq;
import ru.yandex.travel.orders.proto.TDevSendPartnerReportsRsp;
import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.orders.services.report.HotelPartnerPaymentOrderReportSender;
import ru.yandex.travel.orders.services.report.HotelPartnerReportSender;
import ru.yandex.travel.orders.services.report.HotelPartnerReportSenderPlanner;
import ru.yandex.travel.orders.services.report.HotelPartnerReportService;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

import static ru.yandex.travel.orders.infrastructure.CallDescriptor.NO_CALL_ID;

@GrpcService(authenticateService = true)
@RequiredArgsConstructor
@Slf4j
public class ReportDevService extends ReportDevInterfaceDoNotUseGrpc.ReportDevInterfaceDoNotUseImplBase {

    private final HotelPartnerReportService hotelPartnerReportService;

    private final SingleOperationService singleOperationService;

    private final TxCallWrapper txCallWrapper;

    private final Clock clock;

    @Override
    public void createPartnerPayoutReport(TDevCreatePartnerPayoutReportReq request,
                                          StreamObserver<TDevCreatePartnerPayoutReportRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log, rq -> {

            ByteString.Output out = ByteString.newOutput();

            hotelPartnerReportService.writePayoutReportToOutputStream(
                    rq.getBillingPartnerId(),
                    rq.getBillingContractId(),
                    rq.getExternalContractId(),
                    LocalDate.now(),
                    rq.getPartnerLegalName(),
                    ProtoUtils.toLocalDate(rq.getPeriodStart()),
                    ProtoUtils.toLocalDate(rq.getPeriodEnd()),
                    out
            );

            return TDevCreatePartnerPayoutReportRsp.newBuilder()
                    .setBytes(out.toByteString())
                    .build();
        });
    }

    @Override
    public void createPartnerOrdersReport(TDevCreatePartnerOrdersReportReq request,
                                          StreamObserver<TDevCreatePartnerOrdersReportRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log, rq -> {
            ByteString.Output out = ByteString.newOutput();

            hotelPartnerReportService.writeActedOrdersReportToOutputStream(
                    rq.getBillingPartnerId(),
                    rq.getPartnerLegalName(),
                    ProtoUtils.toLocalDate(rq.getPeriodStart()),
                    ProtoUtils.toLocalDate(rq.getPeriodEnd()),
                    out
            );

            return TDevCreatePartnerOrdersReportRsp.newBuilder()
                    .setBytes(out.toByteString())
                    .build();
        });


    }

    @Override
    public void sendPartnerReports(TDevSendPartnerReportsReq request,
                                   StreamObserver<TDevSendPartnerReportsRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, rq -> {
            HotelPartnerReportSender.Params input = new HotelPartnerReportSender.Params();
            input.setEmail(Strings.nullToEmpty(rq.getEmail()));
            input.setBillingClientId(rq.getBillingPartnerId());
            input.setPeriodBegin(ProtoUtils.toLocalDate(rq.getPeriodStart()));
            input.setPeriodEnd(ProtoUtils.toLocalDate(rq.getPeriodEnd()));
            input.setReportAt(ProtoUtils.toInstant(rq.getReportAt()));
            OperationTypes operationType;
            if ("orders".equals(rq.getReportType())) {
                operationType = OperationTypes.HOTELS_SEND_PARTNER_PAYOUT_DETAILS_REPORT;
            } else if ("payout".equals(rq.getReportType())) {
                operationType = OperationTypes.HOTELS_SEND_PARTNER_PAYOUT_ORDER;
            } else {
                throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Unknown report type").toEx();
            }
            UUID operationId = singleOperationService.runOperation(
                    "DevSendPartnerReport " + Instant.now(clock).toEpochMilli(),
                    operationType.getValue(),
                    input
            );
            return TDevSendPartnerReportsRsp.newBuilder().setOperationId(operationId.toString()).build();
        });
    }

    @Override
    public void planPartnerReportsSending(TDevPlanPartnerReportsSendingReq request,
                                          StreamObserver<TDevPlanPartnerReportsSendingRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, rq -> {
            HotelPartnerReportSenderPlanner.Params input = new HotelPartnerReportSenderPlanner.Params();
            input.setPeriodBegin(ProtoUtils.toLocalDate(rq.getPeriodStart()));
            input.setPeriodEnd(ProtoUtils.toLocalDate(rq.getPeriodEnd()));
            if (!Strings.isNullOrEmpty(rq.getReportType())) {
                input.setReportType(rq.getReportType());
            }
            UUID operationId =
                    singleOperationService.runOperation("DevPlanSendPartnerReport " + Instant.now(clock).toEpochMilli(),
                            OperationTypes.HOTELS_PLAN_SENDING_REPORTS_TO_PARTNERS.getValue(), input
                    );
            return TDevPlanPartnerReportsSendingRsp.newBuilder().setOperationId(operationId.toString()).build();
        });
    }

    @Override
    public void createPartnerPaymentOrderReport(TDevCreatePartnerPaymentOrderReportReq request,
                                                StreamObserver<TDevCreatePartnerPaymentOrderReportRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log, rq -> {
            ByteString.Output out = ByteString.newOutput();
            hotelPartnerReportService.writePaymentOrderReportToOutputStream(rq.getPaymentOrderId(),
                    rq.getPaymentBatchId(), out);
            return TDevCreatePartnerPaymentOrderReportRsp.newBuilder()
                    .setBytes(out.toByteString())
                    .build();
        });
    }

    @Override
    public void sendPartnerPaymentOrderReport(TDevSendPartnerPaymentOrderReportReq request,
                                              StreamObserver<TDevSendPartnerPaymentOrderReportRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readWrite(request, NO_CALL_ID), responseObserver, log, rq -> {
            var opName = String.format("DevPartnerPaymentOrderReport_%s_%s",
                    request.getPaymentOrderId(),
                    request.getPaymentBatchId());
            log.info("Will start sending partner payment order reports {} now", opName);
            var input = new HotelPartnerPaymentOrderReportSender.Params();
            input.setBankOrderId(request.getPaymentOrderId());
            input.setPaymentBatchId(request.getPaymentBatchId());
            input.setEmail(request.getEmail());
            var opId = singleOperationService.runOperation(
                    opName,
                    OperationTypes.HOTELS_SEND_PARTNER_PAYMENT_ORDERS_REPORT.getValue(),
                    input);

            return TDevSendPartnerPaymentOrderReportRsp.newBuilder()
                    .setOperationId(opId.toString())
                    .build();
        });
    }
}
