package ru.yandex.travel.orders.grpc;

import java.util.List;
import java.util.function.Function;

import com.google.common.base.Strings;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.entities.TakeoutJob;
import ru.yandex.travel.orders.grpc.helpers.ProtoChecks;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.TakeoutJobRepository;
import ru.yandex.travel.orders.takeout.proto.TCheckOrdersExistenceReq;
import ru.yandex.travel.orders.takeout.proto.TCheckOrdersExistenceRsp;
import ru.yandex.travel.orders.takeout.proto.TFindJobReq;
import ru.yandex.travel.orders.takeout.proto.TFindJobRsp;
import ru.yandex.travel.orders.takeout.proto.TStartJobReq;
import ru.yandex.travel.orders.takeout.proto.TStartJobRsp;
import ru.yandex.travel.orders.takeout.proto.TakeoutOrdersInterfaceV1Grpc;

@GrpcService(authenticateService = true)
@Slf4j
@RequiredArgsConstructor
public class TakeoutOrdersService extends TakeoutOrdersInterfaceV1Grpc.TakeoutOrdersInterfaceV1ImplBase {
    private final TransactionTemplate transactionTemplate;
    private final TakeoutJobRepository takeoutJobRepository;
    private final OrderRepository orderRepository;

    @Override
    public void findJob(TFindJobReq request, StreamObserver<TFindJobRsp> responseObserver) {
        synchronouslyWithTx(request, responseObserver, req -> {
            TakeoutJob job;
            if (Strings.isNullOrEmpty(req.getJobUid())) {
                List<TakeoutJob> lastJob = takeoutJobRepository.getLastJobByPassportId(req.getPassportId(),
                        req.getJobType(), PageRequest.of(0, 1));
                if (lastJob.isEmpty()) {
                    return TFindJobRsp.getDefaultInstance();
                } else {
                    job = lastJob.get(0);
                }
            } else {
                job = takeoutJobRepository.getOne(ProtoChecks.checkStringIsUuid("job id", request.getJobUid()));
            }
            return TFindJobRsp.newBuilder()
                    .setJobUid(ProtoUtils.toStringOrEmpty(job.getId()))
                    .setPayload(ProtoUtils.toTJson(job.getPayload()))
                    .setState(job.getState())
                    .setUpdatedAt(ProtoUtils.fromInstant(job.getUpdatedAt()))
                    .build();
        });
    }

    @Override
    public void startTakeoutJob(TStartJobReq request, StreamObserver<TStartJobRsp> responseObserver) {
        synchronouslyWithTx(request, responseObserver, req -> {
            TakeoutJob job = TakeoutJob.createJob(ProtoChecks.checkStringIsPresent("user id", request.getPassportId()), request.getJobType());
            takeoutJobRepository.save(job);
            return TStartJobRsp.newBuilder().setJobUid(ProtoUtils.toStringOrEmpty(job.getId())).build();
        });
    }

    @Override
    public void checkOrdersExistence(TCheckOrdersExistenceReq request,
                                     StreamObserver<TCheckOrdersExistenceRsp> responseObserver) {
        synchronouslyWithTx(request, responseObserver, req -> TCheckOrdersExistenceRsp.newBuilder()
                .setOrdersExist(
                        orderRepository.findAllOrdersOwnedByUser(request.getPassportId())
                                .stream().anyMatch(o -> !o.nullSafeRemoved())
                ).build());
    }

    private <ReqT, RspT> void synchronouslyWithTx(ReqT request, StreamObserver<RspT> observer,
                                                  Function<ReqT, RspT> handler) {
        ServerUtils.synchronously(log, request, observer,
                rq -> transactionTemplate.execute((ignored) -> handler.apply(rq)),
                ex -> GrpcExceptionHelper.mapStatusException(log, request, ex)
        );
    }
}
