package ru.yandex.travel.orders.grpc;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.BiMap;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ErrorException;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.dataexport.proto.ETellerPaymentState;
import ru.yandex.travel.orders.dataexport.proto.ETellerRefundState;
import ru.yandex.travel.orders.dataexport.proto.ETellerServiceState;
import ru.yandex.travel.orders.dataexport.proto.TTellerPartnerServiceExportReq;
import ru.yandex.travel.orders.dataexport.proto.TTellerPartnerServiceExportRsp;
import ru.yandex.travel.orders.dataexport.proto.TTellerPartnerServiceInfo;
import ru.yandex.travel.orders.dataexport.proto.TTellerPaymentExportReq;
import ru.yandex.travel.orders.dataexport.proto.TTellerPaymentExportRsp;
import ru.yandex.travel.orders.dataexport.proto.TTellerPaymentInfo;
import ru.yandex.travel.orders.dataexport.proto.TTellerRefundExportReq;
import ru.yandex.travel.orders.dataexport.proto.TTellerRefundExportRsp;
import ru.yandex.travel.orders.dataexport.proto.TTellerRefundInfo;
import ru.yandex.travel.orders.dataexport.proto.TellerExportInterfaceV1Grpc;
import ru.yandex.travel.orders.entities.AeroflotInvoice;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.SimpleTrustRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.WellKnownInvoiceDiscriminator;
import ru.yandex.travel.orders.entities.WellKnownOrderItemDiscriminator;
import ru.yandex.travel.orders.grpc.helpers.TxCallWrapper;
import ru.yandex.travel.orders.infrastructure.CallDescriptor;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.repository.InvoiceRepository;
import ru.yandex.travel.orders.repository.OrderItemRepository;
import ru.yandex.travel.orders.repository.SimpleTrustRefundRepository;

import static java.util.stream.Collectors.toList;

@GrpcService(authenticateService = true)
@RequiredArgsConstructor
@Slf4j
public class TellerExportService extends TellerExportInterfaceV1Grpc.TellerExportInterfaceV1ImplBase {
    private final OrderItemRepository orderItemRepository;
    private final InvoiceRepository invoiceRepository;
    private final SimpleTrustRefundRepository refundRepository;
    private final TxCallWrapper txCallWrapper;

    @Override
    public void exportPartnerServices(TTellerPartnerServiceExportReq request,
                                      StreamObserver<TTellerPartnerServiceExportRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log,
                this::exportPartnerServicesSyncImpl);
    }

    private TTellerPartnerServiceExportRsp exportPartnerServicesSyncImpl(TTellerPartnerServiceExportReq rq) {
        Instant from = ProtoUtils.toInstant(rq.getUpdatedAtFrom());
        Instant to = ProtoUtils.toInstant(rq.getUpdatedAtTo());
        List<String> itemTypes = convertServiceTypes(rq.getServiceTypeList());
        List<OrderItem> orderItems = orderItemRepository.findByUpdatedAtAndType(from, to, itemTypes);
        return TTellerPartnerServiceExportRsp.newBuilder()
                .addAllServices(convertServices(orderItems))
                .build();
    }

    @Override
    public void exportPayments(TTellerPaymentExportReq request,
                               StreamObserver<TTellerPaymentExportRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log,
                this::exportPaymentsSyncImpl);
    }

    private TTellerPaymentExportRsp exportPaymentsSyncImpl(TTellerPaymentExportReq rq) {
        Instant from = ProtoUtils.toInstant(rq.getUpdatedAtFrom());
        Instant to = ProtoUtils.toInstant(rq.getUpdatedAtTo());
        List<String> itemTypes = convertPaymentTypes(rq.getInvoiceTypeList());
        List<Invoice> invoices = invoiceRepository.findByUpdatedAtAndType(from, to, itemTypes);
        return TTellerPaymentExportRsp.newBuilder()
                .addAllPayments(convertInvoices(invoices))
                .build();
    }

    @Override
    public void exportRefunds(TTellerRefundExportReq request, StreamObserver<TTellerRefundExportRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log,
                this::exportRefundsSyncImpl);
    }

    private TTellerRefundExportRsp exportRefundsSyncImpl(TTellerRefundExportReq rq) {
        Instant from = ProtoUtils.toInstant(rq.getUpdatedAtFrom());
        Instant to = ProtoUtils.toInstant(rq.getUpdatedAtTo());
        List<SimpleTrustRefund> refunds = refundRepository.findByCreatedAtBetween(from, to);
        return TTellerRefundExportRsp.newBuilder()
                .addAllRefunds(convertRefunds(refunds))
                .build();
    }

    private List<String> convertServiceTypes(List<EServiceType> serviceTypes) {
        BiMap<EServiceType, String> mapping = WellKnownOrderItemDiscriminator
                .ORDER_ITEM_DISCRIMINATOR_TYPE_MAPPING.inverse();
        List<EServiceType> unsupportedTypes = serviceTypes.stream()
                .filter(type -> !mapping.containsKey(type))
                .collect(toList());
        if (!unsupportedTypes.isEmpty()) {
            throw newUnsupportedServiceTypeException(unsupportedTypes);
        }
        return serviceTypes.stream()
                .map(mapping::get)
                .collect(toList());
    }

    private List<String> convertPaymentTypes(List<EInvoiceType> invoiceTypes) {
        BiMap<EInvoiceType, String> mapping = WellKnownInvoiceDiscriminator
                .INVOICE_DISCRIMINATOR_TYPE_MAPPING.inverse();
        List<EInvoiceType> unsupportedTypes = invoiceTypes.stream()
                .filter(type -> !mapping.containsKey(type))
                .collect(toList());
        if (!unsupportedTypes.isEmpty()) {
            throw newUnsupportedInvoiceTypeException(unsupportedTypes);
        }
        return invoiceTypes.stream()
                .map(mapping::get)
                .collect(toList());
    }

    private List<TTellerPartnerServiceInfo> convertServices(List<OrderItem> orderItems) {
        List<TTellerPartnerServiceInfo> services = new ArrayList<>();
        for (OrderItem orderItem : orderItems) {
            services.add(convertServiceInfo(orderItem));
        }
        return services;
    }

    private List<TTellerPaymentInfo> convertInvoices(List<Invoice> invoices) {
        List<TTellerPaymentInfo> payments = new ArrayList<>();
        for (Invoice invoice : invoices) {
            payments.add(convertInvoice(invoice));
        }
        return payments;
    }

    private List<TTellerRefundInfo> convertRefunds(List<SimpleTrustRefund> refunds) {
        List<TTellerRefundInfo> converted = new ArrayList<>();
        for (SimpleTrustRefund refund : refunds) {
            converted.add(convertRefund(refund));
        }
        return converted;
    }

    private TTellerPartnerServiceInfo convertServiceInfo(OrderItem orderItem) {
        Order order = orderItem.getOrder();
        TTellerPartnerServiceInfo.Builder infoBuilder = TTellerPartnerServiceInfo.newBuilder()
                .setOrderId(order.getId().toString())
                .setServiceId(orderItem.getId().toString())
                .setPartnerId(Strings.nullToEmpty(getPartnerId(orderItem)))
                .setCreatedAt(ProtoUtils.fromInstant(order.getCreatedAt()))
                .setUpdatedAt(ProtoUtils.fromInstant(order.getUpdatedAt()))
                .setServiceType(orderItem.getPublicType())
                .setServiceState(getServiceState(orderItem));
        Money price = getPrice(orderItem);
        if (price != null) {
            infoBuilder.setPrice(ProtoUtils.toTPrice(price));
        }
        return infoBuilder.build();
    }

    private TTellerPaymentInfo convertInvoice(Invoice invoice) {
        List<Money> itemsMoney = invoice.getInvoiceItems().stream()
                .map(ii -> Money.of(
                        ii.getOriginalPrice() != null ? ii.getOriginalPrice() : ii.getPrice(),
                        ii.getCurrency()
                ))
                .collect(toList());
        Preconditions.checkArgument(!itemsMoney.isEmpty(), "At least one invoice item is expected");
        Money total = itemsMoney.stream().reduce(Money.of(0, itemsMoney.get(0).getCurrency()), Money::add);
        return TTellerPaymentInfo.newBuilder()
                .setOrderId(invoice.getOrder().getId().toString())
                .setInvoiceId(invoice.getId().toString())
                .setTrustPaymentId(Strings.nullToEmpty(invoice.getTrustPaymentId()))
                .setPurchaseToken(Strings.nullToEmpty(invoice.getPurchaseToken()))
                .setState(getInvoiceState(invoice))
                .setInvoiceType(invoice.getPublicType())
                .setValue(ProtoUtils.toTPrice(total))
                .setCreatedAt(ProtoUtils.fromInstant(invoice.getCreatedAt()))
                .setUpdatedAt(ProtoUtils.fromInstant(invoice.getUpdatedAt()))
                .build();
    }

    private TTellerRefundInfo convertRefund(SimpleTrustRefund refund) {
        Invoice invoice = refund.getInvoice();
        return TTellerRefundInfo.newBuilder()
                .setOrderId(invoice.getOrder().getId().toString())
                .setInvoiceId(invoice.getId().toString())
                .setTrustPaymentId(Strings.nullToEmpty(invoice.getTrustPaymentId()))
                .setTrustRefundId(Strings.nullToEmpty(refund.getTrustRefundId()))
                .setPaymentPurchaseToken(Strings.nullToEmpty(invoice.getPurchaseToken()))
                .setState(getRefundState(refund))
                .setInvoiceType(invoice.getPublicType())
                .setValue(ProtoUtils.toTPrice(refund.getTotalRefundMoney()))
                .setCreatedAt(ProtoUtils.fromInstant(refund.getCreatedAt()))
                .build();
    }

    private String getPartnerId(OrderItem orderItem) {
        switch (orderItem.getPublicType()) {
            case PT_DOLPHIN_HOTEL:
                return ((DolphinOrderItem) orderItem).getDolphinOrderCode();
            default:
                throw newUnsupportedServiceTypeException(orderItem.getPublicType());
        }
    }

    private ETellerServiceState getServiceState(OrderItem orderItem) {
        switch (orderItem.getPublicType()) {
            case PT_DOLPHIN_HOTEL:
                switch (((DolphinOrderItem) orderItem).getState()) {
                    case IS_REFUNDED:
                        return ETellerServiceState.TSS_REFUNDED;
                    case IS_CONFIRMED:
                        return ETellerServiceState.TSS_CONFIRMED;
                    case IS_CANCELLED:
                        return ETellerServiceState.TSS_CANCELLED;
                    default:
                        return ETellerServiceState.TSS_PENDING;
                }
            default:
                throw newUnsupportedServiceTypeException(orderItem.getPublicType());
        }
    }

    private ETellerPaymentState getInvoiceState(Invoice invoice) {
        switch (invoice.getPublicType()) {
            case IT_TRUST:
                switch (((TrustInvoice) invoice).getState()) {
                    case IS_HOLD:
                    case IS_CLEARED:
                        return ETellerPaymentState.TPS_SUCCESSFUL;
                    case IS_REFUNDED:
                        return ETellerPaymentState.TPS_REFUNDED;
                    case IS_PAYMENT_NOT_AUTHORIZED:
                    case IS_CANCELLED:
                        return ETellerPaymentState.TPS_CANCELLED;
                    default:
                        return ETellerPaymentState.TPS_PENDING;
                }
            case IT_AVIA_AEROFLOT:
                switch (((AeroflotInvoice) invoice).getState()) {
                    case IS_CONFIRMED:
                        return ETellerPaymentState.TPS_SUCCESSFUL;
                    case IS_TIMED_OUT:
                    case IS_CANCELLED:
                        return ETellerPaymentState.TPS_CANCELLED;
                    default:
                        return ETellerPaymentState.TPS_PENDING;
                }
            default:
                throw newUnsupportedInvoiceTypeException(List.of(invoice.getPublicType()));
        }
    }

    private ETellerRefundState getRefundState(SimpleTrustRefund refund) {
        switch (refund.getItemState()) {
            case RS_SUCCESS:
                return ETellerRefundState.TRS_SUCCESSFUL;
            case RS_FAILED:
            case RS_ERROR:
                return ETellerRefundState.TRS_CANCELLED;
            default:
                return ETellerRefundState.TRS_PENDING;
        }
    }

    private Money getPrice(OrderItem orderItem) {
        switch (orderItem.getPublicType()) {
            case PT_DOLPHIN_HOTEL:
                return ((DolphinOrderItem) orderItem).getItinerary().getActualPrice();
            default:
                throw newUnsupportedServiceTypeException(orderItem.getPublicType());
        }
    }

    private ErrorException newUnsupportedServiceTypeException(EServiceType serviceType) {
        return newUnsupportedServiceTypeException(List.of(serviceType));
    }

    private ErrorException newUnsupportedServiceTypeException(List<EServiceType> serviceTypes) {
        String typeNames = serviceTypes.size() == 1 ? serviceTypes.get(0).toString() : serviceTypes.toString();
        return new ErrorException(TError.newBuilder()
                .setMessage("Unsupported service type: " + typeNames)
                .setCode(EErrorCode.EC_INVALID_ARGUMENT)
                .build());
    }

    private ErrorException newUnsupportedInvoiceTypeException(List<EInvoiceType> serviceTypes) {
        String typeNames = serviceTypes.size() == 1 ? serviceTypes.get(0).toString() : serviceTypes.toString();
        return new ErrorException(TError.newBuilder()
                .setMessage("Unsupported invoice type: " + typeNames)
                .setCode(EErrorCode.EC_INVALID_ARGUMENT)
                .build());
    }
}
