package ru.yandex.travel.orders.grpc;

import java.time.Clock;
import java.time.Instant;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import io.grpc.stub.StreamObserver;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.orders.grpc.helpers.TxCallWrapper;
import ru.yandex.travel.orders.infrastructure.CallDescriptor;
import ru.yandex.travel.orders.repository.YandexPlusTopupRepository;
import ru.yandex.travel.orders.services.payments.PaymentProfile;
import ru.yandex.travel.orders.services.plus.YandexPlusInitTopupOperation;
import ru.yandex.travel.orders.services.plus.YandexPlusPromoService;
import ru.yandex.travel.orders.yandex_plus.EPaymentProfile;
import ru.yandex.travel.orders.yandex_plus.TGetPlusTopupStatusReq;
import ru.yandex.travel.orders.yandex_plus.TGetPlusTopupStatusRsp;
import ru.yandex.travel.orders.yandex_plus.TSchedulePlusTopupReq;
import ru.yandex.travel.orders.yandex_plus.TSchedulePlusTopupRsp;
import ru.yandex.travel.orders.yandex_plus.YandexPlusServiceGrpc;

@GrpcService(authenticateService = true)
@Slf4j
@RequiredArgsConstructor
public class YandexPlusGrpcService extends YandexPlusServiceGrpc.YandexPlusServiceImplBase {
    private final YandexPlusGrpcServiceProperties yandexPlusGrpcServiceProperties;
    private final YandexPlusPromoService yandexPlusPromoService;
    private final YandexPlusTopupRepository topupRepository;
    private final Clock clock;


    private final TxCallWrapper txCallWrapper;

    @Override
    public void schedulePlusTopup(TSchedulePlusTopupReq request, StreamObserver<TSchedulePlusTopupRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log, req -> {
            Preconditions.checkState(yandexPlusGrpcServiceProperties.isEnabled(), "Yandex plus grpc service is not enabled yet");
            Preconditions.checkArgument(req.getPoints() > 0, "Illegal plus amount");
            Preconditions.checkArgument(!Strings.isNullOrEmpty(req.getPassportId()), "PassportId is required");
            Preconditions.checkArgument(!Strings.isNullOrEmpty(req.getOperationId()), "OperationId is required");
            Preconditions.checkArgument(req.hasTotalAmountForPayload(), "TotalAmountForPayload is required");
            Preconditions.checkArgument(req.hasCommissionAmountForPayload(), "CommissionAmountForPayload is required");
            Preconditions.checkArgument(req.hasVatCommissionAmountForPayload(), "VatCommissionAmountForPayload is required");
            Preconditions.checkArgument(req.getPaymentProfile() == EPaymentProfile.PP_HOTEL, "Only hotel payment profile is supported");
            var paymentProfile = PaymentProfile.HOTEL; // We use same service_id as hotels, so using HOTEL profile here
            Instant topupInstant = Instant.now(clock);
            var uniqueOperationName = getInternalOperationName(req.getOperationId());
            var totalAmountForPayload = ProtoUtils.fromTPrice(req.getTotalAmountForPayload());
            var commissionAmountForPayload = ProtoUtils.fromTPrice(req.getCommissionAmountForPayload());
            var vatCommissionAmountForPayload = ProtoUtils.fromTPrice(req.getVatCommissionAmountForPayload());

            log.info("Scheduling plus topup: {} points will be added to user {} after {}", req.getPoints(), req.getPassportId(), topupInstant);

            var topupData = new YandexPlusInitTopupOperation.TopupData();
            topupData.setPoints(req.getPoints());
            topupData.setCurrency(ProtoCurrencyUnit.fromProtoCurrencyUnit(req.getCurrency()));
            topupData.setPassportId(req.getPassportId());

            var topupDataForExternalOrder = new YandexPlusInitTopupOperation.TopupDataForExternalOrder();
            topupDataForExternalOrder.setExternalOrderId(req.getExternalOrderId());
            topupDataForExternalOrder.setTotalAmountForPayload(totalAmountForPayload.getNumberStripped());
            topupDataForExternalOrder.setCommissionAmountForPayload(commissionAmountForPayload.getNumberStripped());
            topupDataForExternalOrder.setVatCommissionAmountForPayload(vatCommissionAmountForPayload.getNumberStripped());
            topupDataForExternalOrder.setPaymentProfile(paymentProfile);
            topupDataForExternalOrder.setUserIp(req.getUserIp());
            topupDataForExternalOrder.setSkipFinancialEvents(req.getSkipFinancialEvents());
            topupData.setTopupDataForExternalOrder(topupDataForExternalOrder);

            var topupId = yandexPlusPromoService.scheduleTopupOperation(topupData, topupInstant, uniqueOperationName, true);
            if (topupId == null) {
                throw Error.with(EErrorCode.EC_ALREADY_EXISTS, "Topup operation with ID (" + req.getOperationId() + ") already exists").toEx();
            }

            return TSchedulePlusTopupRsp.newBuilder().setScheduled(true).build();
        });
    }

    @Override
    public void getPlusTopupStatus(TGetPlusTopupStatusReq request, StreamObserver<TGetPlusTopupStatusRsp> responseObserver) {
        txCallWrapper.synchronouslyWithTx(CallDescriptor.readOnly(request), responseObserver, log, req -> {
            Preconditions.checkState(yandexPlusGrpcServiceProperties.isEnabled(), "Yandex plus grpc service is not enabled yet");
            var uniqueOperationName = getInternalOperationName(req.getOperationId());
            var scheduledTopupOperationInfo = yandexPlusPromoService.tryGetTopupOperationInfo(uniqueOperationName);
            if (scheduledTopupOperationInfo == null) {
                throw Error.with(EErrorCode.EC_NOT_FOUND, "Topup operation with ID (" + req.getOperationId() + ") not found").toEx();
            }

            var rspBuilder = TGetPlusTopupStatusRsp.newBuilder();
            switch (scheduledTopupOperationInfo.getSingleOperationState()) {
                case ERS_SUCCESS:
                    rspBuilder.setSchedulingStatus(TGetPlusTopupStatusRsp.EScheduledTopupStatus.STC_SUCCEED);
                    break;
                case ERS_FAILURE:
                    rspBuilder.setSchedulingStatus(TGetPlusTopupStatusRsp.EScheduledTopupStatus.STC_FAILED);
                    break;
            }
            rspBuilder.setScheduledAt(ProtoUtils.fromInstant(scheduledTopupOperationInfo.getScheduledAt()));

            if (scheduledTopupOperationInfo.getTopupEntityId() != null) {
                var topupInfo = topupRepository.getOne(scheduledTopupOperationInfo.getTopupEntityId());

                var topupInfoBuilder = TGetPlusTopupStatusRsp.TTopupInfo.newBuilder();

                topupInfoBuilder.setAmount(topupInfo.getAmount());
                topupInfoBuilder.setSucceed(topupInfo.isSuccessfullyCompleted());
                topupInfoBuilder.setTopupEntityId(topupInfo.getId().toString());
                if (topupInfo.getPurchaseToken() != null) {
                    topupInfoBuilder.setPurchaseToken(topupInfo.getPurchaseToken());
                }
                if (topupInfo.getAuthorizedAt() != null) {
                    topupInfoBuilder.setFinishedAt(ProtoUtils.fromInstant(topupInfo.getAuthorizedAt()));
                }

                rspBuilder.setTopupInfo(topupInfoBuilder.build());
            }

            return rspBuilder.build();
        });
    }

    private String getInternalOperationName(String operationId) {
        return "YandexPlusInitTopupExternal:" + operationId;
    }
}
