package ru.yandex.travel.orders.grpc.helpers;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.JsonNode;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.slf4j.MDC;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.model.PayloadMapper;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.commons.experiments.KVExperiment;
import ru.yandex.travel.commons.experiments.OrderExperiments;
import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.BronevikHotelItinerary;
import ru.yandex.travel.hotels.common.orders.DolphinHotelItinerary;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.orders.OrdersApplicationProperties;
import ru.yandex.travel.orders.PrettyIdHelper;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.BNovoOrderItem;
import ru.yandex.travel.orders.entities.BronevikOrderItem;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.FxRate;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.OrderLabelParams;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.promo.PromoCodeActivation;
import ru.yandex.travel.orders.grpc.PromoCodeUserService;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TOrderAggregateState;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.promo.PromoCodeActivationRepository;
import ru.yandex.travel.orders.services.AccountService;
import ru.yandex.travel.orders.services.AuthorizationService;
import ru.yandex.travel.orders.services.OrderInfoMapper;
import ru.yandex.travel.orders.services.OrderOwnerValidator;
import ru.yandex.travel.orders.services.orders.OrderAggregateStateMapper;
import ru.yandex.travel.orders.services.orders.OrderAggregateStateRefresher;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.payments.schedule.PaymentBuilder;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.orders.workflow.hotels.bronevik.proto.EBronevikItemState;
import ru.yandex.travel.orders.workflow.hotels.dolphin.proto.EDolphinItemState;
import ru.yandex.travel.orders.workflow.hotels.expedia.proto.EExpediaItemState;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotItemState;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.EAeroflotOrderState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflows.order.hotel.HotelWorkflowProperties;
import ru.yandex.travel.suburban.model.SuburbanReservation;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static ru.yandex.travel.commons.logging.CommonMdcParams.MDC_ENTITY_ID;

@Slf4j
@RequiredArgsConstructor
@EnableConfigurationProperties({HotelWorkflowProperties.class, OrdersApplicationProperties.class})
@Service
public class OrderCreator {

    private final OrderRepository orderRepository;

    private final WorkflowRepository workflowRepository;

    private final Environment environment;

    private final OrderInfoMapper orderInfoMapper;

    private final AccountService accountService;

    private final AuthorizationService authorizationService;

    private final PromoCodeActivationRepository promoCodeActivationRepository;

    private final OrdersApplicationProperties ordersApplicationProperties;

    private final OrderAggregateStateRefresher orderAggregateStateRefresher;

    private final PaymentBuilder paymentBuilder;

    private final OrderAggregateStateMapper orderAggregateStateMapper;

    private final PromoCodeUserService promoCodeUserService;

    private static EDisplayOrderType determineGenericOrderDisplayType(TCreateOrderReq req) {
        Set<EServiceType> serviceTypes = req.getCreateServicesList().stream()
                .map(TCreateServiceReq::getServiceType)
                .collect(Collectors.toSet());
        if (serviceTypes.equals(Set.of(EServiceType.PT_BNOVO_HOTEL)) ||
                serviceTypes.equals(Set.of(EServiceType.PT_DOLPHIN_HOTEL)) ||
                serviceTypes.equals(Set.of(EServiceType.PT_EXPEDIA_HOTEL)) ||
                serviceTypes.equals(Set.of(EServiceType.PT_BRONEVIK_HOTEL)) ||
                serviceTypes.equals(Set.of(EServiceType.PT_TRAVELLINE_HOTEL))) {
            return EDisplayOrderType.DT_HOTEL;
        } else if (serviceTypes.equals(Set.of(EServiceType.PT_TRAIN))) {
            return EDisplayOrderType.DT_TRAIN;
        } else if (serviceTypes.equals(Set.of(EServiceType.PT_BUS))) {
            return EDisplayOrderType.DT_BUS;
        } else if (serviceTypes.equals(Set.of(EServiceType.PT_SUBURBAN))) {
            return EDisplayOrderType.DT_SUBURBAN;
        } else {
            throw new RuntimeException("Unsupported generic order services set: " + serviceTypes);
        }
    }

    public static OrderItem addOrderItem(Order order, EServiceType type, TJson payload, boolean usePostPay) {
        OrderItem item;
        switch (type) {
            case PT_EXPEDIA_HOTEL:
                item = new ExpediaOrderItem();
                ExpediaHotelItinerary itinerary = ProtoUtils.fromTJson(payload, ExpediaHotelItinerary.class);
                itinerary.setAffiliateId(MessageFormat.format("{0}:{1}", order.getPrettyId(),
                        order.getOrderItems().size()));
                ((ExpediaOrderItem) item).setItinerary(itinerary);
                ((ExpediaOrderItem) item).setState(EExpediaItemState.IS_NEW);
                break;
            case PT_DOLPHIN_HOTEL:
                item = new DolphinOrderItem();
                DolphinHotelItinerary dolphinItinerary = ProtoUtils.fromTJson(payload, DolphinHotelItinerary.class);
                ((DolphinOrderItem) item).setItinerary(dolphinItinerary);
                ((DolphinOrderItem) item).setState(EDolphinItemState.IS_NEW);
                break;
            case PT_TRAVELLINE_HOTEL:
                item = new TravellineOrderItem();
                TravellineHotelItinerary tlItinerary = ProtoUtils.fromTJson(payload, TravellineHotelItinerary.class);
                tlItinerary.getPostPay().setUsed(usePostPay);
                tlItinerary.setYandexNumber(MessageFormat.format("{0}{1}:{2}",
                        tlItinerary.getYandexNumberPrefix(), order.getPrettyId(), order.getOrderItems().size()));
                ((TravellineOrderItem) item).setItinerary(tlItinerary);
                ((TravellineOrderItem) item).setState(ETravellineItemState.IS_NEW);
                break;
            case PT_BNOVO_HOTEL:
                item = new BNovoOrderItem();
                BNovoHotelItinerary bNovoHotelItinerary = ProtoUtils.fromTJson(payload, BNovoHotelItinerary.class);
                bNovoHotelItinerary.getPostPay().setUsed(usePostPay);
                bNovoHotelItinerary.setYandexNumber(MessageFormat.format("{0}{1}:{2}",
                        bNovoHotelItinerary.getYandexNumberPrefix(), order.getPrettyId(),
                        order.getOrderItems().size()));
                ((BNovoOrderItem) item).setItinerary(bNovoHotelItinerary);
                ((BNovoOrderItem) item).setState(EBNovoItemState.IS_NEW);
                break;
            case PT_BRONEVIK_HOTEL:
                item = new BronevikOrderItem();
                BronevikHotelItinerary bronevikHotelItinerary = ProtoUtils.fromTJson(payload, BronevikHotelItinerary.class);
                bronevikHotelItinerary.setYandexNumber(MessageFormat.format("{0}{1}:{2}",
                        bronevikHotelItinerary.getYandexNumberPrefix(), order.getPrettyId(), order.getOrderItems().size()));
                ((BronevikOrderItem) item).setItinerary(bronevikHotelItinerary);
                ((BronevikOrderItem) item).setState(EBronevikItemState.IS_NEW);
                break;
            case PT_FLIGHT:
                item = new AeroflotOrderItem();
                AeroflotServicePayload parsed = PayloadMapper.fromJson(payload.getValue(),
                        AeroflotServicePayload.class);
                ((AeroflotOrderItem) item).setPayload(parsed);
                ((AeroflotOrderItem) item).setState(EAeroflotItemState.IS_NEW);
                break;
            case PT_TRAIN:
                item = new TrainOrderItem();
                TrainReservation reserv = ProtoUtils.fromTJson(payload, TrainReservation.class);
                ((TrainOrderItem) item).setReservation(reserv);
                ((TrainOrderItem) item).setState(EOrderItemState.IS_NEW);
                break;
            case PT_SUBURBAN:
                item = new SuburbanOrderItem();
                SuburbanReservation suburbanReservation = ProtoUtils.fromTJson(payload, SuburbanReservation.class);
                suburbanReservation.setWorkflowData();
                ((SuburbanOrderItem) item).setReservation(suburbanReservation);
                ((SuburbanOrderItem) item).setState(EOrderItemState.IS_NEW);
                break;
            case PT_BUS:
                item = new BusOrderItem();
                BusReservation busReservation = ProtoUtils.fromTJson(payload, BusReservation.class);
                ((BusOrderItem) item).setReservation(busReservation);
                ((BusOrderItem) item).setState(EOrderItemState.IS_NEW);
                break;
            default:
                throw new IllegalArgumentException("Unsupported provider type");
        }
        order.addOrderItem(item);
        return item;
    }

    @TransactionMandatory
    public TCreateOrderRsp createOrderSync(TCreateOrderReq req, UUID newOrderId,
                                           UserCredentials userCredentials) {
        log.info("Creating a new order; type={}, orderId={}, deduplicationKey={}", req.getOrderType(),
                newOrderId, req.getDeduplicationKey());

        UUID deduplicationKey = ProtoChecks.checkStringIsUuid("deduplication key", req.getDeduplicationKey());
        EOrderType orderType = ProtoChecks.checkOrderType("order type", req.getOrderType());

        Error.checkArgument(!req.getUseDeferredPayment() || !req.getUsePostPay(),
                "Cannot use deferred payment and post payment together");

        Order deduplicatedOrder = orderRepository.getOrderByDeduplicationKey(deduplicationKey);
        if (deduplicatedOrder != null) {
            // replacing one MDC id with another
            MDC.put(MDC_ENTITY_ID, deduplicatedOrder.getId().toString());
            log.info("An attempt to create the same order twice");
            Error.with(EErrorCode.EC_ALREADY_EXISTS, "Found duplicate order")
                    .withAttribute("duplicate_order_id", deduplicatedOrder.getId())
                    .withAttribute("deduplication_key", deduplicationKey)
                    .andThrow();
        }

        Order order;
        switch (orderType) {
            case OT_AVIA_AEROFLOT:
                order = new AeroflotOrder();
                ((AeroflotOrder) order).setState(EAeroflotOrderState.OS_NEW);
                order.setDisplayType(EDisplayOrderType.DT_AVIA);
                break;
            case OT_HOTEL_EXPEDIA:
                order = new HotelOrder();
                ((HotelOrder) order).setState(EHotelOrderState.OS_NEW);
                order.setDisplayType(EDisplayOrderType.DT_HOTEL);
                break;
            case OT_TRAIN:
                order = new TrainOrder();
                ((TrainOrder) order).setState(ETrainOrderState.OS_NEW);
                order.setDisplayType(EDisplayOrderType.DT_TRAIN);
                break;
            case OT_GENERIC:
                order = new GenericOrder();
                ((GenericOrder) order).setState(EOrderState.OS_NEW);
                order.setDisplayType(determineGenericOrderDisplayType(req));
                break;
            default:
                throw Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Invalid order type").toEx();
        }

        if (req.hasLabelParams()) {
            OrderLabelParams orderLabelParams = new OrderLabelParams();
            JsonNode payload = ProtoUtils.fromTJson(req.getLabelParams());
            orderLabelParams.setPayload(payload);
            order.setOrderLabelParams(List.of(orderLabelParams));
        }

        Error.checkState(order.getPublicType() == orderType, "Type mismatch");
        order.setId(newOrderId);
        Workflow orderWorkflow = Workflow.createWorkflowForEntity((WorkflowEntity<?>) order);
        orderWorkflow.setSupervisorId(WellKnownWorkflow.ORDER_SUPERVISOR.getUuid());
        workflowRepository.saveAndFlush(orderWorkflow);
        order.setDeduplicationKey(deduplicationKey);
        Long prettyIdSequenceValue;
        do {
            prettyIdSequenceValue = orderRepository.getNextPrettyIdSequenceValue();
        } while (!PrettyIdHelper.checkIdSequenceValue(prettyIdSequenceValue));
        String prettyId = PrettyIdHelper.makePrettyId(prettyIdSequenceValue);
        order.setPrettyId(prettyId);

        var onlyTrains =
                req.getCreateServicesList().stream().allMatch(x -> x.getServiceType() == EServiceType.PT_TRAIN);
        if (orderType != EOrderType.OT_GENERIC || !onlyTrains) {
            Error.checkArgument(req.getCreateServicesCount() == 1, "Expected exactly one service in the order, got %s",
                    req.getCreateServicesCount());
        }
        for (int index = 0; index < req.getCreateServicesCount(); ++index) {
            TCreateServiceReq createServiceReq = req.getCreateServices(index);
            EServiceType providerType = ProtoChecks.checkProviderType(String.format("provider type for service %d",
                            index),
                    createServiceReq.getServiceType());
            Error.checkArgument(createServiceReq.hasSourcePayload(), "Missing source payload for service %s", index);
            OrderItem item = addOrderItem(order, providerType, createServiceReq.getSourcePayload(), req.getUsePostPay());
            item.setItemNumber(index);
            if (createServiceReq.hasHotelTestContext()) {
                item.setTestContext(createServiceReq.getHotelTestContext());
            } else if (createServiceReq.hasTrainTestContext()) {
                item.setTestContext(createServiceReq.getTrainTestContext());
            } else if (createServiceReq.hasAviaTestContext()) {
                item.setTestContext(createServiceReq.getAviaTestContext());
            } else if (createServiceReq.hasBusTestContext()) {
                item.setTestContext(createServiceReq.getBusTestContext());
            } else if (createServiceReq.hasSuburbanTestContext()) {
                item.setTestContext(createServiceReq.getSuburbanTestContext());
            }
        }

        switch (req.getPaymentTestContextsCase()) {
            case AVIAPAYMENTTESTCONTEXT:
                order.setPaymentTestContext(req.getAviaPaymentTestContext());
                break;
            case PAYMENTTESTCONTEXT:
                order.setPaymentTestContext(req.getPaymentTestContext());
                break;
        }

        ECurrency currency = ProtoChecks.checkCurrency("currency", req.getCurrency());
        ProtoCurrencyUnit currencyUnit = ProtoCurrencyUnit.fromProtoCurrencyUnit(currency);

        order.setCurrency(currencyUnit);
        order.setAccount(accountService.createAccount(currencyUnit));

        FxRate fxRate = new FxRate();
        for (int index = 0; index < req.getFxRateCount(); ++index) {
            TCreateOrderReq.TFxRate item = req.getFxRate(index);
            ECurrency itemCurrency = ProtoChecks.checkCurrency(String.format("currency in the FX item %d", index),
                    item.getCurrency());
            BigDecimal itemRate = BigDecimal.valueOf(item.getValue(), item.getPrecision());
            Error.checkArgument(!fxRate.contains(itemCurrency), "FX item %s has duplicate currency %s", index,
                    itemCurrency);
            Error.checkArgument(itemRate.compareTo(BigDecimal.ZERO) > 0, "FX item %s has non-positive rate %s", index
                    , itemRate);
            fxRate.putIfAbsent(itemCurrency, itemRate);
        }

        order.setFxRate(fxRate);
        validateOwner(req, userCredentials);
        order.setEmail(ProtoChecks.checkStringIsPresent("owner email", req.getOwner().getEmail()));
        order.setPhone(ProtoChecks.checkStringIsPresent("owner phone", req.getOwner().getPhone()));
        order.setIp(req.getOwner().getIp());
        order.setLabel(req.getLabel());
        order.setAllowsSubscription(req.getOwner().getAllowsSubscription());
        authorizationService.authorizeForOrder(order.getId(), userCredentials, AuthorizedUser.OrderUserRole.OWNER);
        order.setGeoId(req.getOwner().getGeoId() != 0 ? req.getOwner().getGeoId() : null);

        addPromoCodes(req, order);

        if (order instanceof HotelOrder) {
            // TRAVELBACK-1452 - set payment retry enabled to always be true
            ((HotelOrder) order).setPaymentRetryEnabled(true);
        }
        if (order instanceof GenericOrder && OrderCompatibilityUtils.isHotelOrder(order)) {
            // TRAVELBACK-1452 - set payment retry enabled to always be true
            ((GenericOrder) order).setPaymentRetryEnabled(true);
        }

        tryEnableMockPayment(req, order);

        if (req.hasExperiments()) {
            order.setExperiments(ProtoUtils.fromTJson(req.getExperiments(), OrderExperiments.class));
        }

        List<KVExperiment> kVExperiments = req.getKVExperimentsList().stream()
                .map(item -> new KVExperiment(item.getKey(), item.getValue()))
                .collect(Collectors.toList());
        order.setKVExperiments(kVExperiments);

        order = orderRepository.saveAndFlush(order);

        // Not using ordersApplicationProperties.isRefreshOrderAggregateStateOnWrite()
        // as the initial state object should be created unconditionally not to cause any NotFound errors for pollers.
        orderAggregateStateRefresher.refreshOrderAggregateState(order.getId());

        for (OrderItem item : order.getOrderItems()) {
            Error.checkState(item.getItemState() != null, "Each order item must have the initial workflow state");
            Workflow itemWorkflow = Workflow.createWorkflowForEntity((WorkflowEntity<?>) item, orderWorkflow.getId());
            workflowRepository.saveAndFlush(itemWorkflow);
        }

        order.setEligibleForDeferredPayment(paymentBuilder.canDeferPaymentForOrder(order));
        order.setUseDeferredPayment(req.getUseDeferredPayment());

        orderRepository.flush();


        TOrderInfo.Builder builderForOrder = orderInfoMapper.buildOrderInfoFor(order,
                authorizationService.getOrderOwner(order.getId()), false,
                false, OrderInfoMapper.AggregateStateConstructMode.BYPASS);


        TOrderAggregateState aggregateState = orderAggregateStateMapper.mapAggregateStateFromOrder(order);
        builderForOrder.setOrderAggregateState(aggregateState);

        log.info("Created a new order; orderId={}, deduplicationKey={}",
                newOrderId, req.getDeduplicationKey());
        return TCreateOrderRsp.newBuilder().setNewOrder(builderForOrder).build();
    }

    /**
     * Promocodes are being checked after the order is reserved. This method only creates promo code activations.
     */
    private void addPromoCodes(TCreateOrderReq request, Order order) {
        List<UUID> activationsList = request.getPromoCodeStringsList().stream()
                    .map(promoCodeUserService::activatePromoCode)
                    .collect(Collectors.toList());
        for (UUID activationId : activationsList) {
            Optional<PromoCodeActivation> promoCodeActivation = promoCodeActivationRepository.findById(activationId);
            if (promoCodeActivation.isPresent()) {
                order.addRequestedPromoCodeActivation(promoCodeActivation.get());
            } else {
                throw Error.with(EErrorCode.EC_NOT_FOUND, "No promocode activation found")
                        .withAttribute("activation_id", activationId).toEx();
            }
        }
    }

    private void tryEnableMockPayment(TCreateOrderReq req, Order order) {
        // !!! WARNING !!! check that we're not using prod environment and mock payment is requested
        if (environment.acceptsProfiles("prod")) {
            return;
        }

        boolean mockTrainOrder = ordersApplicationProperties.isTrainOrdersMockPayment()
                && (OrderCompatibilityUtils.isTrainOrder(order)
                || OrderCompatibilityUtils.isSuburbanOrder(order));

        boolean mockPaymentRequested = req.getMockPayment()
                || req.hasAviaPaymentTestContext()
                || req.hasPaymentTestContext()
                || mockTrainOrder;

        order.setMockPayment(mockPaymentRequested);
    }

    private void validateOwner(TCreateOrderReq request, UserCredentials userCredentials) {
        Error.checkArgument(request.hasOwner(), "Missing owner");
        TUserInfo owner = request.getOwner();
        OrderOwnerValidator.validateYandexUid(owner, userCredentials);
        OrderOwnerValidator.validateIfLoggedIn(owner, userCredentials);
    }
}
