package ru.yandex.travel.orders.grpc.helpers;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.proto.TChangeTrainRegistrationStatusReq;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;

import static java.util.stream.Collectors.toList;

public class TrainOrderHelper {

    private TrainOrderHelper() {
    }

    public static List<Integer> validateChangeTrainRegistrationStatusRequest(
            TChangeTrainRegistrationStatusReq request, Order order) {
        Error.checkArgument(OrderCompatibilityUtils.isTrainOrder(order), "Not a train order: %s", order.getId());

        boolean registrationEnabled = request.getEnabled();
        var blankIdsToValidateSet = new HashSet<>(request.getBlankIdsList());
        var validatedBlankIds = new ArrayList<Integer>();

        List<TrainPassenger> allSegmentPassengers = OrderCompatibilityUtils.getTrainOrderItems(order).stream()
                .flatMap(oi -> oi.getReservation().getPassengers().stream())
                .collect(toList());
        for (TrainPassenger passenger : allSegmentPassengers) {
            TrainTicket ticket = passenger.getTicket();
            int blankId = ticket.getBlankId();

            if (blankIdsToValidateSet.contains(blankId)) {
                ImBlankStatus imBlankStatus = ticket.getImBlankStatus();

                if (!isValidImBlankStatus(ticket)) {
                    Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Blank %s in incompatible IM status %s",
                            blankId, imBlankStatus).andThrow();
                }

                if ((imBlankStatus == ImBlankStatus.NO_REMOTE_CHECK_IN && registrationEnabled)
                        || (imBlankStatus == ImBlankStatus.REMOTE_CHECK_IN && !registrationEnabled)) {
                    validatedBlankIds.add(blankId);
                }

                blankIdsToValidateSet.remove(blankId);
            }
        }

        if (validatedBlankIds.isEmpty()) {
            Error.with(EErrorCode.EC_INVALID_ARGUMENT, "None of validated blanks").andThrow();
        }

        if (!blankIdsToValidateSet.isEmpty()) {
            Error.with(EErrorCode.EC_INVALID_ARGUMENT, "Unknown blankIds: {0}", blankIdsToValidateSet.toString())
                    .andThrow();
        }

        return validatedBlankIds;
    }

    private static boolean isValidImBlankStatus(TrainTicket ticket) {
        ImBlankStatus imBlankStatus = ticket.getImBlankStatus();
        return imBlankStatus == ImBlankStatus.NO_REMOTE_CHECK_IN || imBlankStatus == ImBlankStatus.REMOTE_CHECK_IN;
    }
}
