package ru.yandex.travel.orders.infrastructure;

import com.google.common.base.Preconditions;
import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;
import lombok.Getter;

import ru.yandex.travel.orders.grpc.CallType;

public class CallDescriptor<T extends Message> {
    public static final StringValueExtractor CALL_ID = new StringValueExtractor("CallId");
    public static final StringValueExtractor DEDUPLICATION_KEY = new StringValueExtractor("DeduplicationKey");
    public static final EmptyKeyExtractor NO_CALL_ID = new EmptyKeyExtractor();

    @Getter
    private final CallType callType;
    @Getter
    private final String callId;
    @Getter
    private final T request;

    private CallDescriptor(CallType callType, String callId, T request) {
        this.callType = callType;
        this.callId = callId;
        this.request = request;
    }

    public String getName() {
        return request.getClass().getCanonicalName();
    }

    public static <T extends Message> CallDescriptor<T> readWrite(T request, CallIdExtractor callIdExtractor) {
        return new CallDescriptor<>(CallType.READ_WRITE, callIdExtractor.getCallId(request), request);
    }

    public static <T extends Message> CallDescriptor<T> readOnly(T request) {
        return new CallDescriptor<>(CallType.READ_ONLY, "", request);
    }

    public interface CallIdExtractor {
        String getCallId(Message message);
    }

    public static final class StringValueExtractor implements CallIdExtractor {
        private final String fieldName;

        public StringValueExtractor(String fieldName) {
            this.fieldName = fieldName;
        }

        @Override
        public String getCallId(Message message) {
            Descriptors.FieldDescriptor fieldDescriptor = message.getDescriptorForType().findFieldByName(fieldName);
            if (fieldDescriptor == null) {
                throw new IllegalStateException(String.format(
                        "Message %s doesn't have the %s field", message.getClass().getName(), fieldName));
            }
            Preconditions.checkState(fieldDescriptor.getJavaType() == Descriptors.FieldDescriptor.JavaType.STRING,
                    String.format("Tried to get string field %s from message %s. But it was of type %s",
                            fieldName, message, fieldDescriptor.getJavaType())
            );
            return (String) message.getField(fieldDescriptor);
        }
    }

    public static final class EmptyKeyExtractor implements CallIdExtractor {
        @Override
        public String getCallId(Message message) {
            return "";
        }
    }
}
